﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dev.h>
#include <nw/demo.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>


namespace nw      {
namespace eftdemo {


//---------------------------------------------------------------------------
//! @brief        デモカメラ
//---------------------------------------------------------------------------
class Camera
{
    //---------------------------------------------------------
public:

    //---------------------------------------------------------------------------
    //! @brief        移動用のイベントです。
    //---------------------------------------------------------------------------
    class Event
    {
    public:
        typedef enum
        {
            ROTATE_BASED_ON_TARGET   = 0x00000001, //!< ターゲットを基準に回転をおこなう
            ROTATE_VERTICAL_360      = 0x00000002, //!< 縦にも360度回転する
            EXPAND_BASED_ON_TARGET   = 0x00000004, //!< ターゲットを基準に伸縮をおこなう
            POSITION_RESET           = 0x00000008, //!< 位置のリセット
            MOVE_ALONG_AXIS          = 0x00000010, //!< 座標軸に沿って並行移動する             (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
            MOVE_ALONG_VIEW          = 0x00000020, //!< XYZ共に視線ベクトルに沿って並行移動する(デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
            REVERSE_ON_VIEW_CAMERA   = 0x00000040, //!< ビューカメラの操作の場合に反転する
        } FlagType;


        //---------------------------------------------------------------------------
        //! @brief        コンストラクタです。
        //---------------------------------------------------------------------------
        /* ctor */ Event()
            : m_Flags( 0 )
            , m_Move( 0.0f, 0.0f, 0.0f )
            , m_RotateH( 0.0f )
            , m_RotateV( 0.0f )
            , m_Zoom( 0.0f )
            , m_Twist( 0.0f )
        {
        }

        //---------------------------------------------------------------------------
        //! @brief        設定をクリアします。
        //---------------------------------------------------------------------------
        void  Clear()
        {
            *this = Event();
        }

        //---------------------------------------------------------------------------
        //! @brief        ズーム値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  Zoom()
        {
            return m_Zoom;
        }
        f32   Zoom() const
        {
            return m_Zoom;
        }

        //---------------------------------------------------------------------------
        //! @brief        水平方向の回転値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  RotateH()
        {
            return m_RotateH;
        }
        f32   RotateH() const
        {
            return m_RotateH;
        }

        //---------------------------------------------------------------------------
        //! @brief        垂直方向の回転値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  RotateV()
        {
            return m_RotateV;
        }
        f32   RotateV() const
        {
            return m_RotateV;
        }

        //---------------------------------------------------------------------------
        //! @brief        ビュー方向を軸にした回転値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  Twist()
        {
            return m_Twist;
        }
        f32   Twist() const
        {
            return m_Twist;
        }

        //---------------------------------------------------------------------------
        //! @brief        平行移動値を設定/取得します。
        //---------------------------------------------------------------------------
        const nw::math::Vector3& Move() const
        {
            return m_Move;
        }
        nw::math::Vector3& Move()
        {
            return m_Move;
        }

        //---------------------------------------------------------------------------
        //! @brief        移動イベントのオプションフラグを設定/取得します。
        //---------------------------------------------------------------------------
        u32&   Flags()
        {
            return m_Flags;
        }
        u32    Flags() const
        {
            return m_Flags;
        }

        //---------------------------------------------------------------------------
        //! @brief        前方への移動値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  MoveForward()
        {
            return m_Move.z;
        }
        f32   MoveForward() const
        {
            return m_Move.z;
        }

        //---------------------------------------------------------------------------
        //! @brief        アップベクトル方向への移動値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  MoveUp()
        {
            return m_Move.y;
        }
        f32   MoveUp() const
        {
            return m_Move.y;
        }

        //---------------------------------------------------------------------------
        //! @brief        ライトベクトル方向への移動値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  MoveRight()
        {
            return m_Move.x;
        }
        f32   MoveRight() const
        {
            return m_Move.x;
        }


    private:
        u32  m_Flags;

        //-----------------
        // 回転オプション
        //   ROTATE_BASED_ON_TARGET     ：ターゲットを基準に回転 (デフォルトは首振り回転)
        //   ROTATE_VERTICAL_360        ：縦方向にも360度回転    (デフォルトは真上/真下まで)
        //   REVERSE_ON_VIEW_CAMERA     ：ビューカメラ操作の場合に反転
        f32  m_RotateH;    // 垂直回転係数 標準値[-1.0f, 1.0f]
        f32  m_RotateV;    // 水平回転係数 標準値[-1.0f, 1.0f]

        //-----------------
        // 移動オプション
        //   MOVE_ALONG_AXIS            ：座標軸に沿って並行移動する              (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
        //   MOVE_ALONG_VIEW            ：XYZ共に視線ベクトルに沿って並行移動する (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
        //   REVERSE_ON_VIEW_CAMERA     ：ビューカメラ操作の場合に反転
        //     (※MOVE_ALONG_AXISとMOVE_ALONG_VIEWフラグは共存不可)

        nw::math::Vector3 m_Move;

        //-----------------
        // ズームオプション
        //   EXPAND_BASED_ON_TARGET     ：ターゲットを基準に伸縮 (デフォルトはターゲットが動く)
        f32  m_Zoom;  // 伸縮係数 標準値[-1.0f, 1.0f]

        //-----------------
        // ツイストオプション
        f32  m_Twist;           // ツイスト係数  標準値[-1.0f, 1.0f]
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */          Camera();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual /* dtor */  ~Camera() {}

    //---------------------------------------------------------------------------
    //! @brief        カメラ更新。
    //---------------------------------------------------------------------------
    void               UpdateCamera( nw::dev::Pad* pad, nw::dev::Mouse* mouse, bool isMouseAvailable, bool isAltModified );

    //---------------------------------------------------------------------------
    //! @brief       入力に従ってイベントの更新を行います。。
    //---------------------------------------------------------------------------
    void MakeCameraEventClassic( Event* events, nw::dev::Pad* pad, nw::dev::Mouse* mouse, bool isMouseAvailable, bool isAltModified, f32 frameSpeed = 1.0f );


    //---------------------------------------------------------------------------
    //! @brief       他のLookAtObjectの設定をコピー
    //!
    //! @param[out]  dirObj コピー元の LookAtObjec です。
    //---------------------------------------------------------------------------
    void                SetDirection( Camera* dirObj );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトの座標を設定します。
    //---------------------------------------------------------------------------
    void                SetPos( f32 x, f32 y, f32 z );
    void                SetPos( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトの座標を取得します。
    //---------------------------------------------------------------------------
    const nw::math::VEC3*   GetPos()     const;
    void                GetPos( nw::math::VEC3* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       ターゲットの座標を設定します。
    //!
    //! @param[in]    x       設定する注視点のX座標
    //! @param[in]    y       設定する注視点のY座標
    //! @param[in]    z       設定する注視点のZ座標
    //---------------------------------------------------------------------------
    void                SetLookAtPos( f32 x, f32 y, f32 z );
    //---------------------------------------------------------------------------
    //! @param[in]    vec     設定する注視点の座標
    //---------------------------------------------------------------------------
    void                SetLookAtPos( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットの座標を取得します。
    //!
    //! @return       注視点座標の const ポインタ
    //---------------------------------------------------------------------------
    const nw::math::VEC3*   GetLookAtPos()     const;

    //---------------------------------------------------------------------------
    //! @param[in]    pos     注視点の座標を取得するためのポインタ
    //---------------------------------------------------------------------------
    void                GetLookAtPos( nw::math::VEC3* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       Up ベクトルを設定します。
    //!
    //! @param[in]    x       設定するUp ベクトルのX座標
    //! @param[in]    y       設定するUp ベクトルのY座標
    //! @param[in]    z       設定するUp ベクトルのZ座標
    //---------------------------------------------------------------------------
    void                SetUpVec( f32 x, f32 y, f32 z );
    //---------------------------------------------------------------------------
    //! @param[in]    vec     設定するUp ベクトル
    //---------------------------------------------------------------------------
    void                SetUpVec( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief       Up ベクトルの取得
    //!
    //! @return       Up ベクトルを示す const ポインタ
    //---------------------------------------------------------------------------
    const nw::math::VEC3*   GetUpVec() const;

    //---------------------------------------------------------------------------
    //! @param[in]    pos     Up ベクトルを取得するためのポインタ
    //---------------------------------------------------------------------------
    void                GetUpVec( nw::math::VEC3* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトとターゲットの距離を取得します。
    //!
    //! @return       オブジェクト位置から注視点の距離を返します。
    //---------------------------------------------------------------------------
    f32                 GetDistance () const;

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトとターゲットの距離を設定します。
    //!
    //! @details     注視点を固定し、オブジェクトの位置を移動します。
    //!
    //! @param[in]    dist    オブジェクト位置から注視点の距離を設定します。
    //---------------------------------------------------------------------------
    void                SetDistance ( f32 dist );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトをターゲットへ近づけます。
    //---------------------------------------------------------------------------
    void                AddDistanceByMovingObj( f32 length );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットをオブジェクトへ近づけます。
    //---------------------------------------------------------------------------
    void                AddDistanceByMovingTarget( f32 length );

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲を指定します。
    //---------------------------------------------------------------------------
    void                SetDistanceRange( f32 min, f32 max )
    {
        m_DistanceMin = min;
        m_DistanceMax = max;
    }

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲の最大値を取得します。
    //---------------------------------------------------------------------------
    f32                 GetDistanceMax() const
    {
        return m_DistanceMax;
    }

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲の最小値を取得します。
    //---------------------------------------------------------------------------
    f32                 GetDistanceMin() const
    {
        return m_DistanceMin;
    }

    //---------------------------------------------------------------------------
    //! @brief       垂直方向の角度を取得します。
    //---------------------------------------------------------------------------
    u32                 GetAngleV() const
    {
        RecalcIf_( RECALC_ANGLE );
        return m_AngleV;
    }

    //---------------------------------------------------------------------------
    //! @brief       水平方向の角度を取得します。
    //---------------------------------------------------------------------------
    u32                 GetAngleH() const
    {
        RecalcIf_( RECALC_ANGLE );
        return m_AngleH;
    }

    //---------------------------------------------------------------------------
    //! @brief       軸の回転角度を取得します。
    //---------------------------------------------------------------------------
    f32                 GetTwist() const
    {
        return m_Twist;
    }

    //---------------------------------------------------------------------------
    //! @brief       軸の回転角度を設定します。
    //---------------------------------------------------------------------------
    void                SetTwist( f32 twist )
    {
        m_Twist = twist;
    }

    //---------------------------------------------------------------------------
    //! @brief       LookAt オブジェクトを平行移動します。
    //!
    //! @param[in]    x       X 方向の移動距離です。
    //! @param[in]    y       Y 方向の移動距離です。
    //! @param[in]    z       Z 方向の移動距離です。
    //---------------------------------------------------------------------------
    void                Move( f32 x, f32 y, f32 z );

    //---------------------------------------------------------------------------
    //! @brief       ビュー方向に移動します。
    //!
    //! @param[in]   right   ビューのライトベクトル方向への移動距離です。
    //! @param[in]   forward ビューの正面方向への移動距離です。
    //! @param[in]   up      ビューのアップベクトル方向への移動距離です。
    //! @param[in]   view    ビューカメラ位置を指定します。
    //---------------------------------------------------------------------------
    void                MoveView( f32 right, f32 forward, f32 up, const Camera& view );

    //---------------------------------------------------------------------------
    //! @brief       ビューベクトルを XZ 平面に投影したベクトルを基準に平行移動します。
    //!
    //! @param[in]   right   ビューを XZ 平面に投影したライトベクトル方向への移動距離です。
    //! @param[in]   forward ビューを XZ 平面に投影した正面方向への移動距離です。
    //! @param[in]   view    ビューカメラ位置を指定します。
    //---------------------------------------------------------------------------
    void                MoveViewHorizontal( f32 right, f32 forward, const Camera& view );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトを中心に回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Rotate( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットを中心に回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Revolve( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトを中心に360度回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Rotate360( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットを中心に360度回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Revolve360( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       位置を最後にプリセットした位置へ戻します。
    //---------------------------------------------------------------------------
    void                Reset();

    //---------------------------------------------------------------------------
    //! @brief       Reset 時に状態を戻せるように現在の設定を記憶します。
    //---------------------------------------------------------------------------
    void                Preset();

    //---------------------------------------------------------------------------
    //! @brief       Twist の状態のみを Preset します。
    //---------------------------------------------------------------------------
    void                PresetTwist();

    //---------------------------------------------------------------------------
    //! @brief       Bind されているオブジェクトに追従します。
    //---------------------------------------------------------------------------
    void                Follow( const Camera* boundObj );

    //---------------------------------------------------------------------------
    //! @brief       Bindされているオブジェクトとの相対位置を更新します。
    //---------------------------------------------------------------------------
    void                UpdateFollowingBase( const Camera* boundObj );

    //---------------------------------------------------------------------------
    //! @brief       現在の座標・アングルをプリント出力します。
    //---------------------------------------------------------------------------
    void                Dump() const;

    //---------------------------------------------------------------------------
    //! @brief       設定された目的位置まで、TIME_TO_GOALのフレーム数だけ掛けてシームレスに移動させます。
    //---------------------------------------------------------------------------
    void                MoveToDestination();

    //---------------------------------------------------------------------------
    //! @brief       イベントクラスの状態を元に移動します。
    //---------------------------------------------------------------------------
    void                MoveByEvent( const Event& refEvent, const Camera* view = NULL );

    //---------------------------------------------------------------------------
    //! @brief       姿勢行列を取得します。
    //---------------------------------------------------------------------------
    const nw::math::Matrix34&  GetMatrix() const
    {
        this->GetTargetMtx_( &m_Matrix );
        return m_Matrix;
    }


protected:
    mutable nw::math::VEC3  m_Pos;               //!< @details :private カメラ位置
    mutable nw::math::VEC3  m_LookAtPos;         //!< @details :private ターゲット位置

    mutable f32             m_Distance;          //!< @details :private 注視点からの距離
    mutable u32             m_AngleV;            //!< @details :private カメラの水平回転角度
    mutable u32             m_AngleH;            //!< @details :private カメラの垂直回転角度
    f32                     m_Twist;             //!< @details :private カメラのツイスト角度

    mutable nw::math::VEC3  m_LookVec;           //!< @details :private 視線方向ベクトル
    mutable nw::math::VEC3  m_UpVec;             //!< @details :private 視線上方向ベクトル

    f32                     m_DistanceMax;       //!< @details :private 距離最大値
    f32                     m_DistanceMin;       //!< @details :private 距離最小値

    nw::math::VEC3          m_PosReset;          //!< @details :private リセット時の位置
    nw::math::VEC3          m_LookAtPosReset;    //!< @details :private リセット時のターゲット位置
    f32                     m_TwistReset;        //!< @details :private リセット時のツイスト角度

    mutable nw::math::Matrix34  m_Matrix;        //!< @details :private

    enum
    {
        RECALC_NONE = 0,    //!< @details :private 再計算の必要なし
        RECALC_OBJECT,      //!< @details :private カメラ位置を再計算
        RECALC_TARGET,      //!< @details :private ターゲット位置を再計算
        RECALC_ANGLE        //!< @details :private 回転角度を再計算
    };

    mutable u32         m_RecalcFlg;         //!< @details :private 再計算フラグ

    void                RecalcIf_( u32 recalcFlg ) const; //!< @details :private 特定の条件の場合に再計算します。
    void                Recalc_()          const;         //!< @details :private 再計算します。
    void                RecalcAngle_()     const;         //!< @details :private オブジェクトと注視点の座標から角度と距離を計算します。
    void                RecalcPos_()       const;         //!< @details :private 注視点座標と角度からカメラの座標を計算します。
    void                RecalcLookAtPos_() const;         //!< @details :private オブジェクト座標と角度から注視点座標を計算します。
    void                ClampDistance_();                 //!< @details :private 位置と注視点の距離が設定されたMin/Maxの範囲から外れていないかをチェックし、範囲内にクランプします。

    //! @details :private
    void                GetTargetMtx_( nw::math::MTX34* mtx ) const;   // 注視点を中心とした座標系への変換行列を取得

    //! 垂直角度を設定
    //! @details :private
    void         SetAngleV_( u32 angleV ) const
    {
        m_AngleV = angleV;
    }

    //! 水平角度を設定
    //! @details :private
    void         SetAngleH_( u32 angleH )
    {
        m_AngleH = angleH;
    }

    bool           m_KeyP; // キーボード入力キー
    bool           m_KeyA; // キーボード入力キー
};

} // namespace eftdemo
} // namespace nw
