﻿using System;
using System.IO;
using System.Threading;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    using FailureCommand = Tuple<string, string>;

    [TestClass]
    public class TestBase
    {
        public TestContext TestContext { get; set; }
        public bool IsLongTest = false;
        public const string LongTestParameterName = "IsLongTest";

        // AssemblyInitialize は TestApplicationManagerInterface.cs で定義している。

        [TestInitialize]
        public void Initialize()
        {
            m_Maker = new MakeTestApplication(this.TestContext);
            IsLongTest = this.TestContext.Properties.Contains(LongTestParameterName);
            DoInitialize();
        }

        // 共通ユーティリティ
        [TestCleanup]
        public void Cleanup()
        {
            DoCleanup();
            if (m_Maker != null)
            {
                m_Maker.Cleanup();
                m_Maker = null;
            }
        }

        public virtual void DoInitialize()
        {
        }

        public virtual void DoCleanup()
        {
        }

        public static void CleanupImpl(DevMenuCommandSystem command)
        {
            command.Execute(new string[]
            {
                "gamecard erase",
                "sdcard format",
                "factoryreset do",
            });
            command.ResetAndExecute(new string[]
            {
                // Reset するために実行
                "application uninstall --all"
            });
        }

        private static void WaitForLaunchedApplication(DevMenuCommandSystem command)
        {
            Thread.Sleep(1000); // 1 sec
            // 起動中のアプリケーションを終了させる
            // これがないと、 ExitCode が 0 にならない
            command.Execute(new string[] { });
        }

        public static void SuccessLaunchApplication(DevMenuCommandSystem command, ulong id)
        {
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application launch 0x{0:x16}", id),
            }));
            WaitForLaunchedApplication(command);
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application list-record-detail 0x{0:x16}", id)
            },
                "\\\"terminateResult\\\": \\\"0x00000000\\\""
            ));
        }

        public static void FailureLaunchApplication(DevMenuCommandSystem command, ulong id, bool isRunning = false, UInt32 result = 0)
        {
            string[] errorMsg = result == 0 ? null : new string[]
            {
                "Unexpected error.",
                string.Format("result = 0x{0:x8}", result)
            };

            bool isSuccess = command.FailureExecute(string.Format("application launch 0x{0:x16}", id), errorMsg);

            Assert.IsTrue(isSuccess == !isRunning);
            if (isRunning)
            {
                WaitForLaunchedApplication(command);

                if (result == 0)
                {
                    Assert.IsFalse(command.Execute(new string[] {
                        string.Format("application list-record-detail 0x{0:x16}", id)
                    },
                        "\\\"terminateResult\\\": \\\"0x00000000\\\""
                    ));
                }
                else
                {
                    Assert.IsTrue(command.FailureExecute(string.Format("application list-record-detail 0x{0:x16}", id), string.Format("\\\"terminateResult\\\": \\\"0x{0:x8}\\\"", result)));
                }
            }
        }

        public static void FailureExecuteCommands(DevMenuCommandSystem command, IEnumerable<FailureCommand> commandList)
        {
            foreach (var arg in commandList)
            {
                Assert.IsTrue(command.FailureExecute(arg.Item1, arg.Item2));
            }
        }

        public static void ResetAndFailureExecuteCommands(DevMenuCommandSystem command, IEnumerable<FailureCommand> commandList)
        {
            bool first = true;
            foreach (var arg in commandList)
            {
                if (first)
                {
                    Assert.IsTrue(command.ResetAndFailureExecute(arg.Item1, arg.Item2));
                    first = false;
                }
                else
                {
                    Assert.IsTrue(command.FailureExecute(arg.Item1, arg.Item2));
                }
            }
        }

        public static void IndividualExecute(DevMenuCommandSystem command, IEnumerable<string> commandList)
        {
            foreach (var arg in commandList)
            {
                command.Execute(new string[] { arg });
            }
        }

        public static void WriteToGameCard(DevMenuCommandSystem command, NspInfo app, NspInfo patch)
        {
            string patchOption = "";
            if (patch != null)
            {
                patchOption = " --on-card-patch " + patch.Path;
            }
            if (app != null)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "gamecard write " + app.Path + patchOption
                }));
            }
        }

        public static void InstallIfNecessary(DevMenuCommandSystem command, NspInfo app, NspInfo patch, NspInfo dummy, string storage)
        {
            List<string> commandList = new List<string>();

            if (storage == GameCardName)
            {
                var appInfo = app == null ? dummy : app;
                WriteToGameCard(command, appInfo, patch);
            }
            else
            {
                if (app != null)
                {
                    commandList.Add(string.Format("application install {0} -s {1}", app.Path, storage));
                }
                if (patch != null)
                {
                    commandList.Add(string.Format("patch install {0} -s {1}", patch.Path, storage));
                }
                if (commandList.Count > 0)
                {
                    Assert.IsTrue(command.Execute(commandList.ToArray()));
                }
            }
        }

        public static void Install(DevMenuCommandSystem command, NspInfo app, NspInfo patch, NspInfo dummy, string appStorage, string patchStorage)
        {
            if (appStorage == patchStorage)
            {
                InstallIfNecessary(command, app, patch, dummy, appStorage);
            }
            else
            {
                InstallIfNecessary(command, app, null, dummy, appStorage);
                InstallIfNecessary(command, null, patch, dummy, patchStorage);
            }
        }

        public static void Install(DevMenuCommandSystem command, NspInfo app, string appStorage)
        {
            Install(command, app, null, null, appStorage, null);
        }

        public static void Uninstall(DevMenuCommandSystem command, ulong id, string appStorage, string patchStorage)
        {
            var commandList = new List<string>();
            if (appStorage == GameCardName || patchStorage == GameCardName)
            {
                commandList.Add("gamecard erase");
            }
            commandList.Add(string.Format("application uninstall 0x{0:x16}", id));

            Assert.IsTrue(command.Execute(commandList.ToArray()));
        }

        public static void RemovePatch(DevMenuCommandSystem command, ulong id, string patchStorage)
        {
            if (patchStorage == GameCardName)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    "gamecard erase"
                }));
            }
            else
            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("patch uninstall 0x{0:x16} -s {1}", id, patchStorage),
                }));
            }
        }

        public static void ForceReset(DevMenuCommandSystem command)
        {
            // コマンドなしでリセットすることで強制電源断の代わりとする
            command.ResetAndExecute(new string[] { });
        }

        public static void Reboot(DevMenuCommandSystem command)
        {
            // DevMenuCommand の結果が出力する前に接続が切れる可能性があるので、 NoWait にする
            var options = new DevMenuCommandBase.RunOnTargetOptions();
            options.NoWait = true;
            command.Execute(new string[] {
                "power reboot"
            }, options: options);

            // Reboot される前にコマンドが実行されないように待つ
            Thread.Sleep(10 * 1000);
        }

        public static void MakeTestApplicationMetaFile(string filename, ulong id, IEnumerable<string> propertyList)
        {
            // NXFP2-a64 であることを仮定する
            using (StreamWriter sw = new StreamWriter(filename, false, System.Text.Encoding.UTF8))
            {
                var header = new string[]
                {
                    "<NintendoSdkMeta>",
                    "   <Core>",
                    "       <Name>Application</Name>",
                    string.Format("       <ProgramId>0x{0:X16}</ProgramId>", id),
                    "       <Is64BitInstruction>True</Is64BitInstruction>",
                    "       <ProcessAddressSpace>AddressSpace64Bit</ProcessAddressSpace>",
                    "   </Core>",
                    "   <Application>",
                };
                var footer = new string[]
                {
                    "   </Application>",
                    "</NintendoSdkMeta>",
                };

                foreach (var line in header)
                {
                    sw.WriteLine(line);
                }
                foreach (var property in propertyList)
                {
                    sw.WriteLine("      " + property);
                }
                foreach (var line in footer)
                {
                    sw.WriteLine(line);
                }
            }
        }

        protected delegate void CheckFunc(ulong id);

        public const string GameCardName = "gamecard";
        public readonly string[] Storages = new string[] { "builtin", "sdcard" };
        public readonly string[] StoragesWithGc = new string[] { "builtin", "sdcard", GameCardName };

        protected MakeTestApplication m_Maker;

    }
}
