﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestApplicationManagerInterface : TestBase
    {
        [AssemblyInitialize]
        public static void AssemblyInitialize(TestContext context)
        {
            var command = new DevMenuCommandSystem(context);
            CleanupImpl(command);
        }

        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "gamecard erase",
                "application uninstall --all",
            });
        }

        private class OccupiedSize
        {
            public static string BuiltInName  = "Built-in";
            public static string SdCardName = "SD Card";

            public string StorageName { get; private set; }
            public ulong ApplicationSize { get; private set; }
            public ulong PatchSize { get; private set; }
            public ulong AocSize { get; private set; }

            public string Parse(string outputLog)
            {
                using (var sr = new StringReader(outputLog))
                {
                    string line;
                    Regex storagePattern = new Regex(@"\[([a-zA-Z\-\s]+)\]");
                    Regex contentPattern = new Regex(@"([a-zA-Z\-\s]+)\s+([0-9\.]+) B");

                    while ((line = sr.ReadLine()) != null)
                    {
                        if (line == "----------------------------------------------------------")
                        {
                            // byte 表記のみ対応
                            var storageMatch = storagePattern.Match(sr.ReadLine());
                            Assert.IsTrue(storageMatch.Success);
                            StorageName = storageMatch.Groups[1].Value;

                            for (int i = 0; i < 3; i++)
                            {
                                var contentLine = sr.ReadLine();
                                var contentMatch = contentPattern.Match(contentLine);
                                Assert.IsTrue(contentMatch.Success);
                                var contentType = contentMatch.Groups[1].Value.TrimEnd();
                                var contentSize = ulong.Parse(contentMatch.Groups[2].Value.Replace(".0", "")); // Byte の場合は、 0.0 の時以外小数点がない
                                switch (contentType)
                                {
                                    case "Application":
                                        ApplicationSize = contentSize;
                                        break;
                                    case "Patch":
                                        PatchSize = contentSize;
                                        break;
                                    case "Add-On Content":
                                        AocSize = contentSize;
                                        break;
                                    default:
                                        Assert.IsTrue(false, "unknown content type: " + contentType);
                                        break;
                                }

                            }
                            return sr.ReadToEnd();
                        }
                    }
                }

                return outputLog;
            }
        }

        private class OccupiedSizeCommand
        {
            public List<OccupiedSize> OccupiedSizeList;

            public OccupiedSizeCommand(DevMenuCommandSystem command, ulong id)
            {
                OccupiedSizeList = new List<OccupiedSize>();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application occupied-size 0x{0:x16}", id)
                }));

                var currentLog = command.LastOutput;
                while(true)
                {
                    var occupiedSize = new OccupiedSize();
                    var remindLog = occupiedSize.Parse(currentLog);
                    if (remindLog != currentLog)
                    {
                        currentLog = remindLog;
                        OccupiedSizeList.Add(occupiedSize);
                    }
                    else
                    {
                        break;
                    }
                }
            }

        }

        private string GetContentHashDirectoryName(DevMenuCommandSystem command, string contentId, bool isSdCard)
        {
            string storageName = isSdCard ? "sdcard" : "bisuser";

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("filesystem list-all {0}", storageName),
            }));

            using (var sr = new StringReader(command.LastOutput))
            {
                string line;
                var findString = string.Format("{0}.nca", contentId);
                string dirName = null;
                var dirRegex = new Regex(@"\s+([0-9A-Z]+)/");
                while ((line = sr.ReadLine()) != null)
                {
                    if (line.IndexOf(findString) >= 0)
                    {
                        return dirName;
                    }

                    var match = dirRegex.Match(line);
                    if (match.Success)
                    {
                        dirName = match.Groups[1].Value;
                    }
                }
            }

            Assert.IsTrue(false, string.Format("contentId is not found in {0}. id={1}", storageName, contentId));
            return null;
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void ApplicationOccupiedSize()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);
            var aocv0 = m_Maker.MakeAddOnContent(id);

            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "application install " + appv0.Path,
                "patch install -s sdcard " + patchv1.Path,
                "addoncontent install " + aocv0.Path
            });

            // OccupiedSize の正常系
            var beforeOccupiedSize = new OccupiedSizeCommand(command, id);
            Assert.IsTrue(beforeOccupiedSize.OccupiedSizeList.Count == 2);

            var beforeBuiltIn = beforeOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.BuiltInName);
            Assert.IsTrue(beforeBuiltIn.ApplicationSize > 0);
            Assert.IsTrue(beforeBuiltIn.PatchSize == 0);
            Assert.IsTrue(beforeBuiltIn.AocSize > 0);

            var beforeSdCard = beforeOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.SdCardName);
            Assert.IsTrue(beforeSdCard.ApplicationSize == 0);
            Assert.IsTrue(beforeSdCard.PatchSize > 0);
            Assert.IsTrue(beforeSdCard.AocSize == 0);

            // インストールされているコンテンツの一部が無くなっている場合に、消費サイズも減っているかを確認する
            var crypto256 = new System.Security.Cryptography.SHA256CryptoServiceProvider();

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-content-meta-database -s builtin",
            }));
            var builtInCnmt = new JavaScriptSerializer().Deserialize<List<ContentMetaDatabaseForJson>>(command.LastOutput);
            Assert.IsTrue(builtInCnmt.Count == 2);
            var builtInApplicationCnmt = builtInCnmt.Single(x => x.type == "Application");
            var builtInProgramCnmt = builtInApplicationCnmt.contentInfoList.Single(x => x.type == "Program");
            var builtInHashDirName = GetContentHashDirectoryName(command, builtInProgramCnmt.id, false);
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("filesystem delete-file -s builtin /Contents/registered/{0}/{1}.nca", builtInHashDirName, builtInProgramCnmt.id)
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-content-meta-database -s sdcard",
            }));
            var sdcardCnmt = new JavaScriptSerializer().Deserialize<List<ContentMetaDatabaseForJson>>(command.LastOutput);
            Assert.IsTrue(sdcardCnmt.Count == 1);
            var sdcardPatchCnmt = sdcardCnmt.Single(x => x.type == "Patch");
            var sdcardProgramCnmt = sdcardPatchCnmt.contentInfoList.Single(x => x.type == "Program");
            var sdcardHashDirName = GetContentHashDirectoryName(command, sdcardProgramCnmt.id, true);
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("filesystem delete-file -s sdcard /Nintendo/Contents/registered/{0}/{1}.nca", sdcardHashDirName, sdcardProgramCnmt.id)
            }));

            var afterOccupiedSize = new OccupiedSizeCommand(command, id);
            Assert.IsTrue(afterOccupiedSize.OccupiedSizeList.Count == 2);

            var afterBuiltIn = afterOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.BuiltInName);
            Assert.IsTrue(afterBuiltIn.ApplicationSize < beforeBuiltIn.ApplicationSize);
            Assert.IsTrue(afterBuiltIn.PatchSize == beforeBuiltIn.PatchSize);
            Assert.IsTrue(afterBuiltIn.AocSize == beforeBuiltIn.AocSize);

            var afterSdCard = afterOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.SdCardName);
            Assert.IsTrue(afterSdCard.ApplicationSize == beforeSdCard.ApplicationSize);
            Assert.IsTrue(afterSdCard.PatchSize < beforeSdCard.PatchSize);
            Assert.IsTrue(afterSdCard.AocSize == beforeSdCard.AocSize);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void OldContentOccupiedSizeTest()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);
            var patchv2 = m_Maker.MakePatch(id, 2, appv0.Path);
            var aocv0 = m_Maker.MakeAddOnContent(id);
            var aocv1 = m_Maker.MakeAddOnContent(id, version: 1);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
                "patch install -s sdcard " + patchv1.Path,
                "addoncontent install " + aocv0.Path
            }));

            var beforeOccupiedSize = new OccupiedSizeCommand(command, id);
            Assert.IsTrue(beforeOccupiedSize.OccupiedSizeList.Count == 2);

            var beforeBuiltIn = beforeOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.BuiltInName);
            Assert.IsTrue(beforeBuiltIn.ApplicationSize > 0);
            Assert.IsTrue(beforeBuiltIn.PatchSize == 0);
            Assert.IsTrue(beforeBuiltIn.AocSize > 0);

            var beforeSdCard = beforeOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.SdCardName);
            Assert.IsTrue(beforeSdCard.ApplicationSize == 0);
            Assert.IsTrue(beforeSdCard.PatchSize > 0);
            Assert.IsTrue(beforeSdCard.AocSize == 0);

            // OnCardPatch と OnCardAoc に新しいものを入れる

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("gamecard write {0} --on-card-patch {1} --on-card-aoc {2}", appv0.Path, patchv2.Path, aocv1.Path),
            }));

            var afterOccupiedSize = new OccupiedSizeCommand(command, id);
            Assert.IsTrue(afterOccupiedSize.OccupiedSizeList.Count == 2);

            var afterBuiltIn = afterOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.BuiltInName);
            Assert.AreEqual(afterBuiltIn.ApplicationSize, beforeBuiltIn.ApplicationSize);
            Assert.AreEqual(afterBuiltIn.PatchSize, beforeBuiltIn.PatchSize);
            Assert.AreEqual(afterBuiltIn.AocSize, beforeBuiltIn.AocSize);

            var afterSdCard = afterOccupiedSize.OccupiedSizeList.Single(x => x.StorageName == OccupiedSize.SdCardName);
            Assert.AreEqual(afterSdCard.ApplicationSize, beforeSdCard.ApplicationSize);
            Assert.AreEqual(afterSdCard.PatchSize, beforeSdCard.PatchSize);
            Assert.AreEqual(afterSdCard.AocSize, beforeSdCard.AocSize);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void CopyIdentifierTest()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            {
                ulong id = 0x0100394000059000;
                var app = m_Maker.MakeApplication(id, 0, 4, true);

                Assert.IsTrue(command.Execute(new string[] {
                    "application install " + app.Path + " -s sdcard",
                    string.Format("application copy-identifier 0x{0:x16} --storage sdcard", id),
                    string.Format("application uninstall 0x{0:x16}", id),
                    "application install " + app.Path + " -s builtin",
                    string.Format("application copy-identifier 0x{0:x16} --storage builtin", id),
                    string.Format("application uninstall 0x{0:x16}", id),
                }));
            }

            {
                ulong id = 0x0100394000059001;
                var app = m_Maker.MakeApplication(id);
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("gamecard write {0}", app.Path)
                }));
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application copy-identifier 0x{0:x16} --storage card", id),
                }));
            }
        }
    }
}
