﻿using System;
using System.Collections.Generic;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestAddOnContents : TestBase
    {
        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[]
            {
                "debug enable-mount-sdcard",
                "sdcard format",
            });

            command.ResetAndExecute(new string[] {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
            });
        }

        void TestHasAocRight(DevMenuCommandBase command, ulong id)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-view"
            }));
            var view = new JavaScriptSerializer().Deserialize<List<ApplicationViewForJson>>(command.LastOutput);
            Assert.AreEqual(view[0].id, string.Format("0x{0:X16}", id));
            Assert.IsTrue(view[0].hasAddOnContentRecord);
            Assert.IsTrue(view[0].hasAllAddOnContentEntity);
            Assert.IsTrue(view[0].hasAnyAddOnContentEntity);
            Assert.IsFalse(view[0].isCleanupAddOnContentWithNoRightsRecommended);
        }

        void TestLostAocRights (DevMenuCommandBase command, ulong id)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-view"
            }));
            var view = new JavaScriptSerializer().Deserialize<List<ApplicationViewForJson>>(command.LastOutput);
            Assert.AreEqual(view[0].id, string.Format("0x{0:X16}", id));
            Assert.IsTrue(view[0].hasAddOnContentRecord);
            Assert.IsTrue(view[0].hasAllAddOnContentEntity);
            Assert.IsTrue(view[0].hasAnyAddOnContentEntity);
            Assert.IsTrue(view[0].isCleanupAddOnContentWithNoRightsRecommended);
        }

        delegate void TestFunc();

        [TestMethod]
        [Timeout(300 * 1000)]
        public void NoRightsAddOnContents()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, version: 0);
            // index は 1 始まり
            var aoc1 = m_Maker.MakeAddOnContent(id, index: 1, version: 0, ticket: false);
            var aoc2 = m_Maker.MakeAddOnContent(id, index: 2, version: 0, ticket: true);
            var aoc3 = m_Maker.MakeAddOnContent(id, index: 3, version: 0, ticket: true);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appv0.Path,
                "addoncontent install " + aoc1.Path,
                "addoncontent install " + aoc2.Path,
                "addoncontent install " + aoc3.Path,
            }));

            TestFunc ListAllAoc = () =>
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "addoncontent list"
                }, new string[]
                {
                    string.Format("0x{0:X16}\\s+1", id),
                    string.Format("0x{0:X16}\\s+2", id),
                    string.Format("0x{0:X16}\\s+3", id),
                }
                ));
            };

            ListAllAoc();

            SuccessLaunchApplication(command, id);

            TestHasAocRight(command, id);
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("addoncontent verify 0x{0:X16}", id)
            }));

            // チケットを削除する
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("ticket delete 0x{0:X16}", id + 0x1002)
            }));

            // 起動前は View が変わらない
            TestHasAocRight(command, id);

            SuccessLaunchApplication(command, id);

            // 不要な AOC があることを検知
            TestLostAocRights(command, id);

            // 実体はまだ存在する
            ListAllAoc();

            // 権利がない AOC が存在することを確認
            // ResultApplicationTicketNotFound
            Assert.IsTrue(command.FailureExecute(string.Format("addoncontent verify 0x{0:X16}", id), "There are addoncontents with no rights. module = 16, description = 700"));

            // 不要な AOC を削除
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("addoncontent cleanup 0x{0:X16}", id),
                string.Format("addoncontent withdraw-cleanup-recommendation 0x{0:X16}", id)
            }));

            // index=2 のコンテンツがなくなっている
            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent list"
            }, new string[]
            {
                string.Format("0x{0:X16}\\s+1", id),
                string.Format("0x{0:X16}\\s+3", id),
            }));
            Assert.IsFalse(command.Execute(new string[]
            {
                "addoncontent list"
            }, new string[]
            {
                string.Format("0x{0:X16}\\s+2", id),
            }));

            SuccessLaunchApplication(command, id);
            TestHasAocRight(command, id);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestHasAnAddOnContentsEntityFlag()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, version: 0);
            // index は 1 始まり
            var aoc1 = m_Maker.MakeAddOnContent(id, index: 1, version: 0, ticket: false);
            var aoc2 = m_Maker.MakeAddOnContent(id, index: 2, version: 0, ticket: true);
            var aoc3 = m_Maker.MakeAddOnContent(id, index: 3, version: 0, ticket: true);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appv0.Path,
                "addoncontent install " + aoc1.Path,
                "addoncontent install -s sdcard " + aoc2.Path,
                "addoncontent install " + aoc3.Path,
            }));

            TestFunc ListAllAoc = () =>
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "addoncontent list"
                }, new string[]
                {
                    string.Format("0x{0:X16}\\s+1", id),
                    string.Format("0x{0:X16}\\s+3", id),
                }
                ));
                Assert.IsTrue(command.Execute(new string[]
                {
                    "addoncontent list -s sdcard"
                }, new string[]
                {
                    string.Format("0x{0:X16}\\s+2", id),
                }));
            };

            ListAllAoc();

            SuccessLaunchApplication(command, id);

            TestHasAocRight(command, id);

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("addoncontent verify 0x{0:X16}", id)
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "sdcard remove",
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-view"
            }));
            var view = new JavaScriptSerializer().Deserialize<List<ApplicationViewForJson>>(command.LastOutput);
            Assert.AreEqual(view[0].id, string.Format("0x{0:X16}", id));
            Assert.IsTrue(view[0].hasAddOnContentRecord);
            Assert.IsFalse(view[0].hasAllAddOnContentEntity);
            Assert.IsTrue(view[0].hasAnyAddOnContentEntity);
            Assert.IsFalse(view[0].isCleanupAddOnContentWithNoRightsRecommended);

            // 実体がない Aoc があっても起動できる
            SuccessLaunchApplication(command, id);

            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                "application list-view"
            }));
            view = new JavaScriptSerializer().Deserialize<List<ApplicationViewForJson>>(command.LastOutput);
            Assert.AreEqual(view[0].id, string.Format("0x{0:X16}", id));
            Assert.IsTrue(view[0].hasAddOnContentRecord);
            Assert.IsTrue(view[0].hasAllAddOnContentEntity);
            Assert.IsTrue(view[0].hasAnyAddOnContentEntity);
            Assert.IsFalse(view[0].isCleanupAddOnContentWithNoRightsRecommended);

            SuccessLaunchApplication(command, id);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestNoEntityAddonContent()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, version: 0);
            // index は 1 始まり
            var aoc1 = m_Maker.MakeAddOnContent(id, index: 1, version: 0, ticket: false);
            var aoc2 = m_Maker.MakeAddOnContent(id, index: 2, version: 0, ticket: true);
            var aoc3 = m_Maker.MakeAddOnContent(id, index: 3, version: 0, ticket: true);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appv0.Path,
                "addoncontent install " + aoc1.Path,
                "addoncontent install -s sdcard " + aoc2.Path,
                "addoncontent install " + aoc3.Path,
            }));

            TestFunc ListAllAoc = () =>
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "addoncontent list"
                }, new string[]
                {
                    string.Format("0x{0:X16}\\s+1", id),
                    string.Format("0x{0:X16}\\s+3", id),
                }
                ));
                Assert.IsTrue(command.Execute(new string[]
                {
                    "addoncontent list -s sdcard"
                }, new string[]
                {
                    string.Format("0x{0:X16}\\s+2", id),
                }));
            };

            ListAllAoc();

            SuccessLaunchApplication(command, id);

            TestHasAocRight(command, id);
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("addoncontent verify 0x{0:X16}", id)
            }));

            // チケットを削除する
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("ticket delete 0x{0:X16}", id + 0x1002)
            }));

            // 起動前は View が変わらない
            TestHasAocRight(command, id);

            SuccessLaunchApplication(command, id);

            // 不要な AOC があることを検知
            TestLostAocRights(command, id);

            // 実体はまだ存在する
            ListAllAoc();

            // 権利がない AOC が存在することを確認
            // ResultApplicationTicketNotFound
            Assert.IsTrue(command.FailureExecute(string.Format("addoncontent verify 0x{0:X16}", id), "There are addoncontents with no rights. module = 16, description = 700"));

            // チケットのないコンテンツが入っている SD カードを抜く
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("debug disable-mount-sdcard"),
            }));

            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                string.Format("addoncontent verify 0x{0:X16}", id)
            }));

            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = new JavaScriptSerializer().Deserialize<List<ApplicationViewForJson>>(command.LastOutput);
                Assert.AreEqual(view[0].id, string.Format("0x{0:X16}", id));
                Assert.IsTrue(view[0].hasAddOnContentRecord);
                Assert.IsFalse(view[0].hasAllAddOnContentEntity);
                Assert.IsTrue(view[0].hasAnyAddOnContentEntity);
                Assert.IsTrue(view[0].isCleanupAddOnContentWithNoRightsRecommended);
            }

            // フラグを折る
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("addoncontent withdraw-cleanup-recommendation 0x{0:X16}", id)
            }));

            SuccessLaunchApplication(command, id);

            // フラグが立たない
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = new JavaScriptSerializer().Deserialize<List<ApplicationViewForJson>>(command.LastOutput);
                Assert.AreEqual(view[0].id, string.Format("0x{0:X16}", id));
                Assert.IsTrue(view[0].hasAddOnContentRecord);
                Assert.IsFalse(view[0].hasAllAddOnContentEntity);
                Assert.IsTrue(view[0].hasAnyAddOnContentEntity);
                Assert.IsFalse(view[0].isCleanupAddOnContentWithNoRightsRecommended);
            }

            // SD カードを再挿入
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("debug enable-mount-sdcard"),
            }));

            Reboot(command);

            // 起動前は View が変わらない
            TestHasAocRight(command, id);

            SuccessLaunchApplication(command, id);

            // 不要な AOC があることを検知
            TestLostAocRights(command, id);

            // 実体はまだ存在する
            ListAllAoc();

            // 権利がない AOC が存在することを確認
            // ResultApplicationTicketNotFound
            Assert.IsTrue(command.FailureExecute(string.Format("addoncontent verify 0x{0:X16}", id), "There are addoncontents with no rights. module = 16, description = 700"));

            // 不要な AOC を削除
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("addoncontent cleanup 0x{0:X16}", id),
                string.Format("addoncontent withdraw-cleanup-recommendation 0x{0:X16}", id)
            }));

            // index=2 のコンテンツがなくなっている
            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent list"
            }, new string[]
            {
                string.Format("0x{0:X16}\\s+1", id),
                string.Format("0x{0:X16}\\s+3", id),
            }));
            Assert.IsFalse(command.Execute(new string[]
            {
                "addoncontent list"
            }, new string[]
            {
                string.Format("0x{0:X16}\\s+2", id),
            }));

            SuccessLaunchApplication(command, id);
            TestHasAocRight(command, id);
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestCorruptAoc()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, version: 0);
            const int NumAoc = 17; // 16 + 1 個
            var aocList = new List<NspInfo>();
            for (int i = 0; i < NumAoc; i++)
            {
                // index は 1 始まり
                aocList.Add(m_Maker.MakeAddOnContent(id, index: i + 1, version: 0));
            }

            var command = new DevMenuCommandSystem(this.TestContext);

            Action NoCorruptAocTest = () =>
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application verify 0x{0:x16}", id),
                },
                    "No corrupted contents."
                ));
            };

            Action HasCorruptAocTest = () =>
            {
                Assert.IsTrue(command.FailureExecute(
                    string.Format("application verify 0x{0:x16}", id),
                    "The application is corrupted."
                ));
            };

            Action<string> CorruptTest = (string commandOption) =>
            {
                NoCorruptAocTest();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("addoncontent corrupt 0x{0:x16} {1}", id, commandOption)
                }));

                HasCorruptAocTest();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("addoncontent uninstall 0x{0:x16} {1}", id, commandOption)
                }));

                NoCorruptAocTest();
            };

            Func<int, string> getStorageString = (int index) =>
            {
                return ((index % 2) == 0) ? "builtin" : "sdcard";
            };

            var commandList = new List<string>();
            commandList.Add("application install " + appv0.Path);
            for (var i = 0; i < aocList.Count; i++)
            {
                var aoc = aocList[i];
                var storage = getStorageString(i);
                commandList.Add(string.Format("addoncontent install {0} -s {1}", aoc.Path, storage));
                if (commandList.Count >= 5)
                {
                    Assert.IsTrue(command.Execute(commandList));
                    commandList.Clear();
                }
            }
            if (commandList.Count > 0)
            {
                Assert.IsTrue(command.Execute(commandList));
                commandList.Clear();
            }

            for (var i = 0; i < aocList.Count; i++)
            {
                var aoc = aocList[i];
                var storage = getStorageString(i);
                CorruptTest(string.Format("-i {0} -s {1}", i + 1, storage));
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("addoncontent install {0} -s {1}", aoc.Path, storage),
                }));
            }
        }
    }
}
