﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    [TestClass]
    public class TestBase
    {
        public TestContext TestContext { get; set; }

        protected const ulong ExFatBootImageId = 0x010000000000081b;
        protected const string ExFatBootImageIdString = "0x010000000000081b";

        protected const ulong TestAppForRecoverExFatId          = 0x0100aaaaaaaa0000;
        protected const ulong TestSystemUpdateForRecoverExFatId = 0x0100aaaabbbb0000;

        protected static string HostTargetName;
        protected static string ClientTargetName;

        protected static MakeTestApplication AppMaker;
        protected static MakeTestSystemContent SysMaker;

        protected static bool HasHostGameCard;
        protected static bool HasClientGameCard;
        protected static bool HasHostSdCard;
        protected static bool HasClientSdCard;

        protected static DevMenuCommandSystem HostCommand;
        protected static DevMenuCommandSystem ClientCommand;

        protected static ControlTarget ControlTarget;

        private static string[] CleanupCommandList0;
        private static string[] CleanupCommandList1;

        public static string IdString(ulong id)
        {
            return string.Format("0x{0:x16}", id);
        }

        protected static void SetupTestEnvironment(TestContext context)
        {
            if (ControlTarget == null)
            {
                ControlTarget = new ControlTarget(context);
                var targetList = ControlTarget.ListTarget();
                Assert.IsTrue(targetList != null && targetList.Length >= 2);
                HostTargetName = targetList[0];
                ClientTargetName = targetList[1];

                HostCommand = new DevMenuCommandSystem(context, HostTargetName);
                ClientCommand = new DevMenuCommandSystem(context, ClientTargetName);

                AppMaker = new MakeTestApplication(context);
                SysMaker = new MakeTestSystemContent(context);

                Assert.IsTrue(HostCommand.Execute(new string[]
                {
                    "gamecard status"
                }));
                HasHostGameCard = !HostCommand.LastOutput.Contains("Not Inserted");

                Assert.IsTrue(ClientCommand.Execute(new string[]
                {
                    "gamecard status"
                }));
                HasClientGameCard = !ClientCommand.LastOutput.Contains("Not Inserted");

                Assert.IsTrue(HostCommand.Execute(new string[]
                {
                    "sdcard status"
                }));
                HasHostSdCard = !HostCommand.LastOutput.Contains("Sdcard is not inserted.");

                Assert.IsTrue(ClientCommand.Execute(new string[]
                {
                    "sdcard status"
                }));
                HasClientSdCard = !ClientCommand.LastOutput.Contains("Sdcard is not inserted.");
            }


            CleanupCommandList0 = new string[]
            {
                "localcontentshare set-system-update-id \"0x0\"",
                "localcontentshare set-system-delivery-protocol-version 1",
                "localcontentshare set-application-delivery-protocol-version 1",
                "localcontentshare set-acceptable-application-delivery-protocol-version 0",
                "localcontentshare set-required-system-version 0",
                "localcontentshare set-async-task-result 0x0",
            };
            CleanupCommandList1 = new string[]
            {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
            };

            CleanupBase(context);
        }

        protected static void CleanupBase(TestContext context)
        {
            CleanupExternalStorage();

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(CleanupCommandList0, ignoreError: true);
                command.Execute(CleanupCommandList1, ignoreError: true);
                using (var exFat = new ExFatRecovery(command, context)) { }
                // exFat の dispose でリセットされるので、特に明示的にリセットしない
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(CleanupCommandList0, ignoreError: true);
                command.Execute(CleanupCommandList1, ignoreError: true);
                using (var exFat = new ExFatRecovery(command, context)) { }
                // exFat の dispose でリセットされるので、特に明示的にリセットしない
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        protected static void CleanupExternalStorage(bool doReset = false)
        {
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(new string[]
                {
                    HasHostGameCard ? "gamecard erase" : "gamecard status",
                    HasHostSdCard ? "sdcard format" : "sdcard status",
                }, ignoreError: true);
                if (doReset)
                {
                    ForceReset(command);
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(new string[]
                {
                    HasClientGameCard ? "gamecard erase" : "gamecard status",
                    HasClientSdCard ? "sdcard format" : "sdcard status",
                }, ignoreError: true);
                if (doReset)
                {
                    ForceReset(command);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        protected static void ForceReset(DevMenuCommandSystem command)
        {
            ControlTarget.Execute("reset -t " + command.TargetName);
            Thread.Sleep(10 * 1000);
        }

        protected static void Reboot(DevMenuCommandSystem command)
        {
            var options = new DevMenuCommandBase.RunOnTargetOptions();
            options.NoWait = true;
            options.SuppressAutoKill = false;
            command.Execute(new string[]
            {
                "power reboot"
            }, options: options);

            Thread.Sleep(10 * 1000);
        }

        protected void UninstallExFat(DevMenuCommandSystem command, bool doReset = true)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("systemprogram uninstall {0}", ExFatBootImageIdString),
                "systemupdate update-bootimages",
                "systemupdate clear-exfat-status",
            }));
            if (doReset)
            {
                ForceReset(command);
            }
        }

        protected void InstallTestApplicationImpl(DevMenuCommandSystem command, int version, List<NspInfo> appList, string appStorage = null, string patchStorage = null)
        {
            List<string> commandList = new List<string>();

            string appStorageArgs = string.IsNullOrEmpty(appStorage) ? "" : "-s " + appStorage + " ";
            string patchStorageArgs = string.IsNullOrEmpty(patchStorage) ? "" : "-s " + patchStorage + " ";

            commandList.Add(string.Format("application install {0} {1} --force", appList[0].Path,  appStorageArgs));

            if (version > 0)
            {
                commandList.Add(string.Format("patch install {0} {1} --force", appList[version].Path,  patchStorageArgs));
            }
            Assert.IsTrue(command.Execute(commandList));
        }

        protected static List<SystemProgramList> GetSystemProgramList(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "systemprogram list"
            }));
            return SystemProgramList.Deserialize(command.LastOutput);
        }

        protected static bool VerifyNetworkEnvironment(DevMenuCommandSystem command)
        {
            bool isEnabled = command.Execute(new string[]
            {
                "network get-global-ip-addr"
            });
            if (isEnabled)
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "servicediscovery info"
                }));
                if (!command.LastOutput.Contains("[td1]"))
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "servicediscovery import-all td1env"
                    }));
                    ForceReset(command);
                }
            }
            return isEnabled;
        }

        protected static List<ApplicationViewForJson> GetApplicationViewList(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-view"
            }));
            return ApplicationViewForJson.Deserialize(command.LastOutput);
        }

        protected static ApplicationViewForJson GetApplicationViewImpl(DevMenuCommandSystem command, string id)
        {
            var viewList = GetApplicationViewList(command);
            Assert.IsTrue(viewList.Any(x => x.id == id));
            return viewList.Single(x => x.id == id);
        }
    }
}
