﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    /*
     * アプリケーションのダウンロードタスク＆適用タスクに関するテスト
     */
    [TestClass]
    public class ApplicationDownloadTaskTest : TestBase
    {
        // 内部鍵を利用したコンテンツを利用
        private const string TestAppIdString = "0x0100eee00197e000";
        private const string TestPatchIdString = "0x0100eee00197e800";
        private const string TestAocIdString = "0x0100eee00197f001";
        private const string TestDeltaAppIdString = "0x0100f6900049a000";
        private const string TestDeltaPatchIdString = "0x0100f6900049a800";
        private static bool IsHostNetworkEnabled;
        private static bool IsClientNetworkEnabled;
        private static bool IsNetworkEnabled;

        public enum TestTaskType
        {
            AddOnContent,
            Application,
            NewPatch,
            SamePatch,
        };

        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            SetupTestEnvironment(context);

            var hostTask = Task.Run(() =>
            {
                IsHostNetworkEnabled = VerifyNetworkEnvironment(HostCommand);
            });

            var clientTask = Task.Run(() =>
            {
                IsClientNetworkEnabled = VerifyNetworkEnvironment(ClientCommand);
            });

            hostTask.Wait();
            clientTask.Wait();

            IsNetworkEnabled = IsHostNetworkEnabled && IsClientNetworkEnabled;

            CleanupBase(context);
        }

        [TestCleanup]
        public void TestCleanup()
        {
            var cleanupCommand = new string[]
            {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        private ApplicationViewForJson GetApplicationView(DevMenuCommandSystem command, string id = TestAppIdString)
        {
            return GetApplicationViewImpl(command, id);
        }

        private void SetupSenderEnvironment(DevMenuCommandSystem command, UInt32 version)
        {
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application download {0}", TestAppIdString),
                string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestAppIdString, TestPatchIdString, version),
                string.Format("application wait-download {0}", TestAppIdString)
            }));
        }

        private void CleanupPatch(DevMenuCommandSystem command, string id = TestAppIdString)
        {
            command.Execute(new string[]
            {
                "patch uninstall " + id,
                "patch uninstall " + id+ " -s sdcard"
            }, ignoreError: true);
        }

        private void CreateAocDownloadTask(DevMenuCommandSystem command, bool skipRunningCheck = false)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application create-download-task {0} --type AddOnContent --id {1}", TestAppIdString, TestAocIdString),
            }));

            while (!skipRunningCheck)
            {
                var view = GetApplicationView(command);
                Assert.IsTrue(view.isDownloading);
                Assert.IsFalse(view.hasAddOnContentRecord);
                Assert.IsTrue(view.progress != null);
                if (view.progress.isRunning && view.progress.downloaded > 0)
                {
                    break;
                }
            }

        }

        private void CreateApplicationDownloadTask(DevMenuCommandSystem command, bool skipRunningCheck = false)
        {
            // カードの抜き差しの代わりに既存のアプリの実体を消す
            command.Execute(new string[]
            {
                string.Format("application delete-entity {0} --app", TestAppIdString),
            });

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application create-download-task {0}", TestAppIdString),
            }));

            while (!skipRunningCheck)
            {
                var view = GetApplicationView(command);
                Assert.IsTrue(view.isDownloading);
                Assert.IsFalse(view.hasMainEntity);
                Assert.IsTrue(view.progress != null);
                if (view.progress.isRunning && view.progress.downloaded > 0)
                {
                    break;
                }
            }
        }

        private void CreatePatchDownloadTask(DevMenuCommandSystem command, UInt32 version, UInt32 baseVersion = 0, bool skipRunningCheck = false)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestAppIdString, TestPatchIdString, version),
            }));

            while (!skipRunningCheck)
            {
                var view = GetApplicationView(command);
                Assert.IsTrue(view.isDownloading);
                Assert.AreEqual(baseVersion, view.version);
                Assert.IsTrue(view.progress != null);
                if (view.progress.isRunning && view.progress.downloaded > 0)
                {
                    break;
                }
            }
        }

        private void TestDownloadTaskBase(DevMenuCommandSystem command, Action beforeCreateTaskAction, Action testAction,  Action<TestTaskType> verifyAction, UInt32 newPatchVersion = 2 << 16, UInt32 sendPatchVersion = 1 << 16, UInt32 baseApplicationVersion = 0, bool skipRunningCheck = false)
        {
            // Aoc のタスク
            beforeCreateTaskAction();

            CreateAocDownloadTask(command, skipRunningCheck);

            testAction();

            verifyAction(TestTaskType.AddOnContent);

            // Application のタスク
            beforeCreateTaskAction();

            CreateApplicationDownloadTask(command, skipRunningCheck);

            testAction();

            verifyAction(TestTaskType.Application);

            // Patch のタスク (タスクのバージョンが新しい)
            beforeCreateTaskAction();

            CreatePatchDownloadTask(command, newPatchVersion, baseApplicationVersion, skipRunningCheck);

            testAction();

            verifyAction(TestTaskType.NewPatch);

            // Patch のタスク (タスクのバージョンが同じ)
            beforeCreateTaskAction();

            CreatePatchDownloadTask(command, sendPatchVersion, baseApplicationVersion, skipRunningCheck);

            testAction();

            verifyAction(TestTaskType.SamePatch);
        }

        private void OnlySendPatchTest(DevMenuCommandSystem command, Barrier barrier, Action testAction)
        {
            const UInt32 PatchVersion = 1 << 16;
            SetupSenderEnvironment(command, PatchVersion);

            // Aoc のタスク
            barrier.SignalAndWait();
            testAction();

            // Application のタスク
            barrier.SignalAndWait();
            testAction();

            // Patch のタスク (タスクのバージョンが新しい)
            barrier.SignalAndWait();
            testAction();

            // Patch のタスク (タスクのバージョンが同じ)
            barrier.SignalAndWait();
            testAction();
        }

        private Action SendDefaultPatchAction(DevMenuCommandSystem command)
        {
            return () =>
            {
                command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 240", TestAppIdString)
                });
            };
        }

        private void SendDefaultPatchTest(DevMenuCommandSystem command, Barrier barrier)
        {
            var testAction = SendDefaultPatchAction(command);

            const UInt32 PatchVersion = 1 << 16;
            SetupSenderEnvironment(command, PatchVersion);

            // Aoc のタスク
            barrier.SignalAndWait();
            testAction();

            // Application のタスク
            barrier.SignalAndWait();
            testAction();

            // Patch のタスク (タスクのバージョンが新しい)
            barrier.SignalAndWait();
            testAction();

            // Patch のタスク (タスクのバージョンが同じ)
            barrier.SignalAndWait();
            testAction();
        }

        private void PrepareDefaultReceiverEnvironmentBeforeCreateTask(DevMenuCommandSystem command, Barrier barrier)
        {
            // 一度既存のタスクを削除する
            command.Execute(new string[]
            {
                string.Format("application cancel-download {0}", TestAppIdString),
            });

            // 受信済みのパッチを消しておく
            CleanupPatch(command);

            var viewList = GetApplicationViewList(command);
            if (!viewList.Any(x => x.id == TestAppIdString))
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application download {0}", TestAppIdString),
                    string.Format("application disable-auto-update {0}", TestAppIdString),
                }));
            }

            // 送信側のタスクを待っている間に受信が終わるのを防ぐため、タスクを作る前にシグナルする
            barrier.SignalAndWait();
        }

        private Action MakeActionToPrepareDefaultReceiverEnvironmentBeforeCreateTask(DevMenuCommandSystem command, Barrier barrier)
        {
            return () =>
            {
                PrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier);
            };
        }

        private Action<TestTaskType> MakeVerifyTaskIsRestoreAction(DevMenuCommandSystem command, bool isTaskSuccess, string id = TestAppIdString)
        {
            return (TestTaskType type) =>
            {
                var view = GetApplicationView(command, id);
                if (isTaskSuccess && type == TestTaskType.SamePatch)
                {
                    Assert.IsFalse(view.isDownloading);
                }
                else
                {
                    Assert.IsTrue(view.isDownloading);
                    Assert.IsTrue(view.progress != null);
                }
            };
        }

        private void ReceiveDefaultPatch(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
            }));
            var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
            Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
        }

        private void MakeApplyDeltaTask(DevMenuCommandSystem command)
        {
            {
                var viewList = GetApplicationViewList(command);
                if (!viewList.Any(x => x.id == TestDeltaAppIdString))
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application download {0}", TestDeltaAppIdString),
                    }));
                }

                var view = GetApplicationView(command, TestDeltaAppIdString);
                {
                    if (!view.hasPatchEntity)
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestDeltaAppIdString, TestDeltaPatchIdString, 2 << 16),
                            string.Format("application wait-download {0}", TestDeltaAppIdString),
                        }));
                    }
                }
            }

            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestDeltaAppIdString, TestDeltaPatchIdString, 10 << 16),
            }));

            // ダウンロードに時間が掛かるので、ある程度待ってからポーリングする
            Thread.Sleep(30 * 1000);
            while (true)
            {
                var view = GetApplicationView(command, TestDeltaAppIdString);
                if (view.isApplyingDelta)
                {
                    break;
                }
            }
        }

        private void WaitApply(DevMenuCommandSystem command)
        {
            while (true)
            {
                var view = GetApplicationView(command, TestDeltaAppIdString);
                if (!view.isApplyingDelta)
                {
                    break;
                }
                Thread.Sleep(1 * 1000);
            }
        }

        private void CleanupDeltaFlag(DevMenuCommandSystem command)
        {
            command.Execute(new string[]
            {
                "patch set-prefer-delta false"
            });
            ForceReset(command);
        }

        private void IsWaitingTaskTestOnReceivingSystemUpdateImpl(bool isApplyTask)
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                using (var _ = new ScopeExit(() =>
                    {
                        command.Execute(new string[] { "localcontentshare set-required-system-version 0" });
                        ForceReset(command);
                    }))
                {
                    // 再起動させるため
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "localcontentshare set-required-system-version 2"
                    }));

                    // アプリ本体が無くても送れるので、アプリのダウンロードは省略
                    Assert.IsTrue(command.ResetAndExecute(new string[]
                    {
                        string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestAppIdString, TestPatchIdString, 2 << 16),
                        string.Format("application wait-download {0}", TestAppIdString),
                    }));

                    barrier.SignalAndWait();

                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    });
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                var systemProgramList = GetSystemProgramList(command);
                // 一個しかないはず
                var systemUpdate = systemProgramList.Single(x => x.type == "SystemUpdate");
                var systemUpdateV1 = SysMaker.MakeTestSystemUpdateMeta(Convert.ToUInt64(systemUpdate.id, 16), 1, new MakeTestSystemContent.ContentMetaInfo[] { });
                var currentSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(Convert.ToUInt64(systemUpdate.id, 16), systemUpdate.version, new MakeTestSystemContent.ContentMetaInfo[] { });


                using (var _ = new ScopeExit(() =>
                    {
                        command.Execute(new string[]
                        {
                            string.Format("systemprogram install {0} --force", currentSystemUpdate.Path),
                        });
                        // CleanupDeltaFlag で再起動してくれる
                        // そのため、 !isApplyTask の時も呼ぶ
                        CleanupDeltaFlag(command);
                    }))
                {
                    // システムの設定
                    {
                        var commandList = new List<string>();
                        commandList.Add(string.Format("systemprogram install {0} --force", systemUpdateV1.Path));
                        if (isApplyTask)
                        {

                            commandList.Add("patch set-prefer-delta true");
                        }
                        Assert.IsTrue(command.Execute(commandList));
                        ForceReset(command);
                    }


                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application download {0}", TestAppIdString),
                    }));
                    if (isApplyTask)
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestAppIdString, TestPatchIdString, 1 << 16),
                            string.Format("application wait-download {0}", TestAppIdString),
                        }));
                    }

                    // IsWaiting にさせるため、アプリケーションを起動状態にしておく
                    var options = new DevMenuCommandBase.RunOnTargetOptions();
                    options.NoWait = true;
                    options.SuppressAutoKill = false;

                    command.Execute(new string[]
                    {
                        string.Format("application launch {0}", TestAppIdString),
                    }, options: options);

                    options.NoWait = false;
                    options.SuppressAutoKill = true;

                    // タスクを作成する
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestAppIdString, TestPatchIdString, 2 << 16),
                    }, options: options));

                    // ダウンロードが終わるのを待つ
                    Thread.Sleep(10 * 1000);
                    while (true)
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "application list-view"
                        }, options: options));
                        var viewList = ApplicationViewForJson.Deserialize(command.LastOutput);
                        var view = viewList.Single(x => x.id == TestAppIdString);
                        if (view.isWaitingPatchInstall)
                        {
                            break;
                        }
                        Thread.Sleep(1 * 1000);
                    }

                    var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                    if (File.Exists(resumeFile))
                    {
                        File.Delete(resumeFile);
                    }

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
                    }, options: options));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                        Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                        Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                        Assert.IsTrue(File.Exists(resumeFile));
                    }

                    Reboot(command);

                    // 再起動の前後でコミット待ちのタスクが ns::GetApplicationView() の呼び出しによって状態が変わることで失敗することを確認する
                    // 状態が変わったまま通信を続けると、lcs 内の状態が怪しくなるので、ホストに再接続する前にエラーになる
                    Assert.IsTrue(command.FailureExecute(
                        string.Format("localcontentshare receive-application {0} --resume \"{1}\" --timeout 300", TestAppIdString, resumeFile),
                        // !isApplyingDelta の時も lcs::ResultApplyDeltaTaskExists が返ってくる
                        "Unexpected error. result = 0x000342d3." // lcs::ResultApplyDeltaTaskExists
                    ));
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }


        [TestMethod]
        [Timeout(600 * 1000)]
        public void RestoreApplicationDownloadTask()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action testAction = () =>
                {
                    ReceiveDefaultPatch(command);
                };

                TestDownloadTaskBase(command, MakeActionToPrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier), testAction, MakeVerifyTaskIsRestoreAction(command, true));
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void RestoreApplicationDownloadTaskByCancel()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action testAction = () =>
                {
                    Assert.IsTrue(command.FailureExecute(
                        string.Format("localcontentshare receive-application {0} --timeout 5", TestAppIdString),
                        "Timeout"
                    ));
                };

                // Same Version のパッチのタスクも復帰するので、全て同じチェックにしておく
                Action<TestTaskType> verifyAction = (TestTaskType type) =>
                {
                    var view = GetApplicationView(command);
                    Assert.IsTrue(view.isDownloading);
                };

                TestDownloadTaskBase(command, MakeActionToPrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier), testAction, verifyAction);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void RestoreApplicationDownloadTaskByDisabledWifi()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                using (var _ = new ScopeExit(() => { command.Execute(new string[] { "debug disable-nifm-testing-mode" }); ForceReset(command); }))
                {
                    Assert.IsTrue(ClientCommand.Execute(new string[]
                    {
                        "debug enable-nifm-testing-mode"
                    }));
                    ForceReset(ClientCommand);

                    Action beforeCreateTaskAction = () =>
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "network enable-wireless-communication-for-test",
                        }));
                        PrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier);
                    };

                    Action testAction = () =>
                    {
                        var options = new DevMenuCommandBase.RunOnTargetOptions();
                        options.NoWait = true;
                        options.SuppressAutoKill = false;
                        command.Execute(new string[]
                        {
                            string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString),
                        }, options: options);

                        options.NoWait = false;
                        options.SuppressAutoKill = true;
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "network disable-wireless-communication-for-test"
                        }, options: options));

                        // 念のため少し待つ
                        Thread.Sleep(1 * 1000);
                    };

                    TestDownloadTaskBase(command, beforeCreateTaskAction, testAction, MakeVerifyTaskIsRestoreAction(command, false));
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void NotRestoreApplicationDownloadTaskByForceReset()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action testAction = () =>
                {
                    var options = new DevMenuCommandBase.RunOnTargetOptions();

                    options.NoWait = true;
                    options.SuppressAutoKill = false;
                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString),
                    }, options: options);

                    options.NoWait = false;
                    options.SuppressAutoKill = true;
                    while (true)
                    {
                        var storageString = HasClientSdCard ? "-s sdcard" : "";
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "application list-place-holder " + storageString
                        }, options: options));
                        var placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                        if (placeHolderList.Count > 0)
                        {
                            break;
                        }
                    }

                    ForceReset(command);
                };

                Action<TestTaskType> verifyAction = (TestTaskType _) =>
                {
                    var view = GetApplicationView(command);
                    Assert.IsFalse(view.isDownloading);
                };
                TestDownloadTaskBase(command, MakeActionToPrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier), testAction, verifyAction);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void RestoreNotEnoughSpaceTask()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action beforeCreateTask = () =>
                {
                    PrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier);
                    if (HasClientSdCard)
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "debug fill-sdcard-free-space"
                        }));
                    }
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "debug fill-nand-free-space"
                    }));
                };

                Action cleanupStorage = () =>
                {
                    if (HasClientSdCard)
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "debug empty-sdcard-free-space"
                        }));
                    }
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "debug empty-nand-free-space"
                    }));
                };

                Action testAction = () =>
                {
                    // Runnable から NotEnoughSpace になるには、若干のタイムラグがある
                    int i = 0;
                    for (;i < 10; i++)
                    {
                        var view = GetApplicationView(command);
                        Assert.IsTrue(view.progress != null);
                        if (view.progress.state == "NotEnoughSpace")
                        {
                            break;
                        }
                        Thread.Sleep(1 * 1000);
                    }
                    Assert.IsTrue(i < 10);
                    cleanupStorage();

                    ReceiveDefaultPatch(command);
                };

                using (var _ = new ScopeExit(cleanupStorage))
                {
                    TestDownloadTaskBase(command, beforeCreateTask, testAction, MakeVerifyTaskIsRestoreAction(command, true), skipRunningCheck: true);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void RestoreFatalTask()
        {
            if (!IsNetworkEnabled || !HasClientSdCard)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action beforeCreateTask = () =>
                {
                    PrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier);

                    // SD カードを再挿入するため、再起動する
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "debug enable-mount-sdcard",
                        "sdcard format"
                    }));
                    ForceReset(command);
                };

                Action testAction = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "debug disable-mount-sdcard",
                    }));

                    Reboot(command);

                    // Fatal になるまでに若干のタイムラグがある
                    int i = 0;
                    for (; i < 10; i++)
                    {
                        var view = GetApplicationView(command);
                        Assert.IsTrue(view.progress != null);
                        if (view.progress.state == "Fatal")
                        {
                            break;
                        }
                    }
                    Assert.IsTrue(i < 10);

                    ReceiveDefaultPatch(command);
                };

                using (var _ = new ScopeExit(() => { command.Execute(new string[] { "debug enable-mount-sdcard" }); ForceReset(command); }))
                {
                    TestDownloadTaskBase(command, beforeCreateTask, testAction, MakeVerifyTaskIsRestoreAction(command, true));
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void RestoreSuspendTask()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendDefaultPatchTest(HostCommand, barrier);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action testAction = () =>
                {
                    // TODO: nim のエラーエミュレーションが入ったら、そちらに切り替える
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application set-task-state {0} --state Suspended", TestAppIdString)
                    }));

                    var view = GetApplicationView(command);
                    Assert.IsTrue(view.progress != null);
                    Assert.IsTrue(view.progress.state == "Suspended");

                    ReceiveDefaultPatch(command);
                };

                TestDownloadTaskBase(command, MakeActionToPrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier), testAction, MakeVerifyTaskIsRestoreAction(command, true));
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void SenderTaskTest()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                const UInt32 PatchVersion = 1 << 16;
                SetupSenderEnvironment(command, PatchVersion);

                Action testAction = () =>
                {
                    var options = new DevMenuCommandBase.RunOnTargetOptions();
                    options.NoWait = true;
                    options.SuppressAutoKill = false;
                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                    }, options: options);

                    Thread.Sleep(2 * 1000);

                    {
                        var view = GetApplicationView(command);
                        Assert.IsTrue(view.progress != null);
                        Assert.IsTrue(view.progress.downloaded > 0);
                    }

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application cancel-download {0}", TestAppIdString)
                    }));
                };

                // Aoc のタスク
                barrier.SignalAndWait();
                CreateAocDownloadTask(command);
                testAction();

                // Application のタスク
                barrier.SignalAndWait();
                CreateApplicationDownloadTask(command);
                testAction();

                // Patch のタスク
                barrier.SignalAndWait();
                CreatePatchDownloadTask(command, 2 << 16, baseVersion: PatchVersion);
                testAction();
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application download {0}", TestAppIdString)
                }));

                Action receiveAction = () =>
                {
                    // Result を見ない
                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                    });
                };

                // Aoc のタスク & Application のタスク & Patch のタスク
                for (int i = 0; i < 3; i++)
                {
                    barrier.SignalAndWait();
                    receiveAction();
                    CleanupPatch(command);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void RecoverTaskForDeliverSystemAndPatch()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            const ulong SystemDataId = 0x0100111111110000;
            const ulong SystemUpdateId = 0x0100111122220000;

            var systemDataList = new List<NspInfo>();
            systemDataList.Add(AppMaker.MakeSystemData(SystemDataId));
            systemDataList.Add(AppMaker.MakeSystemData(SystemDataId, size: 0x5000000, version: 1));

            var systemUpdateList = new List<NspInfo>();
            systemUpdateList.Add(SysMaker.MakeTestSystemUpdateMeta(SystemUpdateId, 0, new MakeTestSystemContent.ContentMetaInfo[]
            {
                new MakeTestSystemContent.ContentMetaInfo(systemDataList[0], 0),
            }));
            systemUpdateList.Add(SysMaker.MakeTestSystemUpdateMeta(SystemUpdateId, 1, new MakeTestSystemContent.ContentMetaInfo[]
            {
                new MakeTestSystemContent.ContentMetaInfo(systemDataList[1], 1 << 16),
            }));

            Action<DevMenuCommandSystem, int> SwitchSystem = (DevMenuCommandSystem command, int version) =>
            {
                command.Execute(new string[]
                {
                    "localcontentshare set-system-update-id 0x0",
                    string.Format("systemprogram uninstall 0x{0:x16}", SystemUpdateId),
                    string.Format("systemprogram uninstall 0x{0:x16}", SystemDataId),
                }, ignoreError: true);
                Assert.IsTrue(command.Execute(new string[]
                {
                    "localcontentshare set-required-system-version 1",
                    string.Format("localcontentshare set-system-update-id 0x{0:X16}", SystemUpdateId),
                    "systemprogram install " + systemDataList[version].Path,
                    "systemprogram install " + systemUpdateList[version].Path,
                }));
                ForceReset(command);
            };

            Action<DevMenuCommandSystem> CleanupCommand = (DevMenuCommandSystem command) =>
            {
                command.Execute(new string[]
                {
                    "localcontentshare set-required-system-version 0",
                    "localcontentshare set-system-update-id 0x0",
                    string.Format("systemprogram uninstall 0x{0:x16}", SystemUpdateId),
                    string.Format("systemprogram uninstall 0x{0:x16}", SystemDataId),
                }, ignoreError: true);
                ForceReset(command);
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                using (var _ = new ScopeExit(() => { CleanupCommand(command); }))
                {
                    SwitchSystem(command, 1);
                    SendDefaultPatchTest(command, barrier);
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                using (var _ = new ScopeExit(() => { CleanupCommand(command); }))
                {
                    string resumeFile = Path.Combine(AppMaker.GetOutputDirectory().Replace("\"", ""), "resumeClient.txt");
                    if (File.Exists(resumeFile))
                    {
                        File.Delete(resumeFile);
                    }

                    Action beforeCreateTaskAction = () =>
                    {
                        SwitchSystem(command, 0);
                        PrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier);
                    };

                    Action testAction = () =>
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("localcontentshare receive-application {0} --timeout 180 --output-resume-file \"{1}\"", TestAppIdString, resumeFile),
                        }));
                        var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Suspened, outputs.state);
                        Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, outputs.reason);

                        Reboot(command);

                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("localcontentshare receive-application {0} --timeout 180 --resume \"{1}\"", TestAppIdString, resumeFile)
                        }));
                        outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                    };

                    TestDownloadTaskBase(command, beforeCreateTaskAction, testAction, MakeVerifyTaskIsRestoreAction(command, true));
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void NotCancelAnotherApplicationDownloadTask()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            ulong appId = 0x0100000011110000;
            string appIdString = "0x0100000011110000";
            var app = AppMaker.MakeApplication(appId);
            var patch = AppMaker.MakePatch(appId, 1, app.Path);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                Assert.IsTrue(command.Execute(new string[]
                {
                    "application install " + app.Path,
                    "application install " + patch.Path
                }));
                OnlySendPatchTest(HostCommand, barrier, () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 180", appIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                });
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application install " + app.Path,
                }));

                Action beforeCreateTaskAction = () =>
                {
                    CleanupPatch(command, appIdString);
                    PrepareDefaultReceiverEnvironmentBeforeCreateTask(command, barrier);
                };

                Action testAction = () =>
                {
                    var options = new DevMenuCommandBase.RunOnTargetOptions();
                    options.NoWait = true;
                    options.SuppressAutoKill = false;
                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", appIdString)
                    }, options: options);
                };

                Action<TestTaskType> verifyAction = (TestTaskType type) =>
                {
                    Thread.Sleep(2 * 1000);
                    var view = GetApplicationView(command);
                    Assert.IsTrue(view.isDownloading);
                    Assert.IsTrue(view.progress != null);
                    Assert.IsTrue(view.progress.downloaded > 0);
                };

                TestDownloadTaskBase(command, beforeCreateTaskAction, testAction, verifyAction);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void ApplyTaskTestForSender()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            ulong appId = 0x0100000011110000;
            string appIdString = "0x0100000011110000";
            var app = AppMaker.MakeApplication(appId);
            var patch = AppMaker.MakePatch(appId, 1, app.Path);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                using (var _ = new ScopeExit(() => { CleanupDeltaFlag(command); }))
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "patch set-prefer-delta true"
                    }));
                    ForceReset(command);
                    // 送信時に該当アプリが適用中だったら失敗する
                    {
                        MakeApplyDeltaTask(command);
                        Assert.IsTrue(command.FailureExecute(
                            string.Format("localcontentshare send-application {0} --timeout 180", TestDeltaAppIdString),
                            "Unexpected error. result = 0x000342d3." // lcs::ResultApplyDeltaTaskExists
                        ));
                        WaitApply(command);
                        CleanupPatch(command, TestDeltaAppIdString);
                    }

                    // 適用中とは関係ないアプリだったら送信できる
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "application install " + app.Path,
                            "application install " + patch.Path
                        }));

                        MakeApplyDeltaTask(command);
                        barrier.SignalAndWait();
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("localcontentshare send-application {0} --timeout 180", appIdString)
                        }));
                        var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                        WaitApply(command);
                    }

                    // 適用したパッチが送れるかを念のため確認
                    {
                        barrier.SignalAndWait();

                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("localcontentshare send-application {0} --timeout 180", TestDeltaAppIdString)
                        }));
                        var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                    }
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application install " + app.Path,
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", appIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                    CleanupPatch(command, appIdString);
                }

                // 適用後のパッチを受け取れることを念のため確認する
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application download {0}", TestDeltaAppIdString)
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", TestDeltaAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void ApplyTaskTestForReceiver()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var barrier = new Barrier(2);

            ulong appId = 0x0100000011110000;
            string appIdString = "0x0100000011110000";
            var app = AppMaker.MakeApplication(appId);
            var patch = AppMaker.MakePatch(appId, 1, app.Path);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                // 受信側のテスト
                {
                    // アプリ本体が無くても送れるので、アプリのダウンロードは省略
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestDeltaAppIdString, TestDeltaPatchIdString, 10 << 16),
                        string.Format("application wait-download {0}", TestDeltaAppIdString),
                    }));

                    barrier.SignalAndWait();

                    var options = new DevMenuCommandBase.RunOnTargetOptions();
                    options.NoWait = true;
                    options.SuppressAutoKill = false;

                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestDeltaAppIdString)
                    }, options: options);

                    // 受信側の処理待ち
                    barrier.SignalAndWait();

                    // 前のジョブを強制終了
                    command.Execute(new string[] { });

                    barrier.SignalAndWait();

                    // クライアント側は MakeApplyDeltaTask を実行しているので、この間にアプリをインストールしておく
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application install " + app.Path,
                        "application install " + patch.Path
                    }));

                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", appIdString)
                    }, options: options);

                    barrier.SignalAndWait();
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                using (var _ = new ScopeExit(() => { CleanupDeltaFlag(command); }))
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "patch set-prefer-delta true"
                    }));
                    ForceReset(command);


                    // 受信時に該当アプリが適用中だったら失敗する
                    {
                        barrier.SignalAndWait();

                        MakeApplyDeltaTask(command);
                        Assert.IsTrue(command.FailureExecute(
                            string.Format("localcontentshare receive-application {0} --timeout 180", TestDeltaAppIdString),
                            "Unexpected error. result = 0x000342d3." // lcs::ResultApplyDeltaTaskExists
                        ));

                        // 送信側の待ち状態を解除
                        barrier.SignalAndWait();

                        WaitApply(command);
                        CleanupPatch(command, TestDeltaAppIdString);
                    }

                    // 適用中とは関係ないアプリだったら受信できる
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("application install {0}", app.Path)
                        }));

                        barrier.SignalAndWait();

                        MakeApplyDeltaTask(command);
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            string.Format("localcontentshare receive-application {0} --timeout 180", appIdString)
                        }));
                        var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                        barrier.SignalAndWait();

                        WaitApply(command);

                        var view = GetApplicationView(command, TestDeltaAppIdString);
                        Assert.AreEqual(10u << 16, view.version);
                    }
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        //
        // 本当はシステム更新が挟まらない IsWaitingTask のケースも作りたいけど、途中でアプリケーションを止める術がないので、あきらめる
        //

        [TestMethod]
        [Timeout(600 * 1000)]
        public void IsWaitingDownloadTaskWithReceivingSystemUpdate()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            IsWaitingTaskTestOnReceivingSystemUpdateImpl(false);
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void IsWaitingApplyTaskOnReceivingSystemUpdate()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            IsWaitingTaskTestOnReceivingSystemUpdateImpl(true);
        }
    }
}
