# -*- coding: utf-8 -*-
from __future__ import division, print_function, absolute_import, unicode_literals

"""
Date   : 20151007

Purpose:
 python標準のloggingパッケージを簡単に利用するためのサポートモジュールです。
 Allow you to use logging package of standard python library easier.


How to use:
 from kiki.log import console_logger as logger

 このようにインポートすればこのモジュールが提供するデフォルトの
 loggerインスタンスを利用可能です。
 もし独自のhandlerを利用したい場合はpython標準のloggingパッケージを利用してください。

 In order to use a default instance created by this module,
 all you have to do is import "console_logger" instance like above.
 If you want to use a custom logging handler, I think it is better to try to use
 standard logging package directly.


Tips:
 Pycharmを使って開発しているのであればGrep Consoleというプラグインの利用を推奨します。
 正規表現によってフォーマットされたログ出力に色を付けたりフィルタリングすることが容易になります。

 For Pycharm IDE users, I recommend you to use Grep Console plugin.
 It can colorize and filter console logs with Regular Expressions.
"""

import os
import sys
import logging

from kiki.util import fmttime

logger_name_hash = {}
DEFAULT_FORMAT = u'<%(asctime)s> [%(levelname)s]\t[%(name)s]\t[%(module)s]\t| %(message)s'

FILE_NAME = os.path.splitext(os.path.basename(sys.argv[0]))[0]
LOG_DIR = os.getcwd() + "/log/" + fmttime.now_s() + "_" + FILE_NAME + "/"


def create_console_logger(name="mainc", fmt=DEFAULT_FORMAT, level=logging.DEBUG):
    """ This method help you to create general logger for console.

    :param name:    namespace of logger
    :param fmt:     output format
    :param level:   filtering level
    :return:        logger instance
    """
    if name in logger_name_hash:
        return logger_name_hash.get(name)
    else:
        logger = logging.getLogger(name)
        logger.setLevel(level)

        # handler for console
        csh = logging.StreamHandler()
        csh.setLevel(level)
        csh.setFormatter(logging.Formatter(fmt))
        logger.addHandler(csh)

        logger_name_hash[name] = logger
        return logger


def create_file_logger(name="main", path=None, fmt=DEFAULT_FORMAT, level=logging.DEBUG):
    """ This method help you to create general logger for console and file stream.

    :param name:        namespace of logger
    :param path:    output log file path
    :param fmt:         output format
    :param level:       filtering level
    :return:            logger instance
    """
    if name in logger_name_hash:
        return logger_name_hash.get(name)
    else:
        if path is None:
            if not os.path.isdir(LOG_DIR):
                os.makedirs(LOG_DIR)
            path = LOG_DIR + fmttime.now_f() + "_" + FILE_NAME + ".txt"

        logger = logging.getLogger(name)
        logger.setLevel(level)

        # handler for console
        csh = logging.StreamHandler()
        csh.setLevel(level)
        csh.setFormatter(logging.Formatter(fmt))
        logger.addHandler(csh)

        # handler for file stream
        fh = logging.FileHandler(path)
        fh.setLevel(level)
        fh.setFormatter(logging.Formatter(fmt))
        logger.addHandler(fh)

        logger_name_hash[name] = logger
        return logger


#############################################################################################
# create default instance
#############################################################################################
# for kikimora debug
kiki_logger = create_console_logger(name="kiki")

# for general use (without file stream)
console_logger = create_console_logger()

# for general use (with file stream)
file_logger = create_file_logger()

#############################################################################################
# examples
#############################################################################################
if __name__ == "__main__":
    # 一般的な使い方 : general use
    kiki_logger.debug('log for debug')
    kiki_logger.info('general information')
    kiki_logger.warning('warning')
    kiki_logger.error('error!')
    kiki_logger.critical('A critical error has occurred! You must check it immediately!')

    # ログの継承 : inherit logger

    # loggerは.で区切られた名前空間によってhandlerを継承することができます
    # ここではすでにkikiという名前のloggerが存在しているので、
    # それに継承させる形でlogger2を作成してみます
    #
    # Logger is separated by namespaces that is split by .(dot)
    # Logger can inherit handlers from ancestor logger.
    # Let's create extended logger instance.
    logger2 = logging.getLogger("kiki.logger2")

    # 親レベルのloggerのlevelを変えるとその配下にも影響が及びます。便利ですね。
    #
    # As you can see, if you change a filter level at super level, it would effect those children.
    kiki_logger.setLevel(logging.INFO)

    kiki_logger.debug('log for debug')
    kiki_logger.info('general information')
    kiki_logger.warning('warning')

    logger2.debug('log for debug')
    logger2.info('general information')
    logger2.warning('warning')

    # ファイル出力の方法 : use file_logger (completely same with a general one)

    # デフォルトでは、実行されたトップレベルのモジュールと同じディレクトリにlogという名前のディレクトリが作成されます
    #
    # Log file will be output into the same path with running top level module.
    file_logger.info("[Japanese] マルチバイト文字をloggerで使う方法")
    file_logger.info("")
    file_logger.info("こんな感じでマルチバイト文字も使うことができます")
    file_logger.info("もし簡単に使いたいのであれば、次のパッケージをファイルの先頭で読み込むといいです")
    file_logger.info("from __future__ import unicode_literals")
    file_logger.info("")

    file_logger.info("[English] How to use Multi-byte characters in logger.")
    file_logger.info("")
    file_logger.info("You can also use multi-byte characters! like this, やっほー！")
    file_logger.info("If you want to use multi-byte characters easier,"
                     " it is better to put a following package at the beginning of the file.")
    file_logger.info("from __future__ import unicode_literals")
    file_logger.info("")
