﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

#include "CaptureProcessorWrapper.h"

using namespace captureProcessor;

TEST(CaptureProcessorSuite, LabelingTest)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    ImageIdType imageId;
    int labelingAreaCount = 0;
    LabelingResultHandleType resultHandle;

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_ExecuteLabelingFunction(&labelingAreaCount, &resultHandle, imageId));

    int resultCount;
    LabelingResult labelingResult;
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetLabelingResultsFunction(&resultCount, &labelingResult, resultHandle, 1));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_ReleaseLabelingResultFunction(resultHandle));

    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    SerialNumberString serialNumber;
    ASSERT_GE(argc, 2);
    strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

    ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
        serialNumber, false), CaptureResult::CaptureResult_Success);

    // ラベリング実行前に結果取得
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidLabelingResultHandle,
        captureProcessorWrapper.m_GetLabelingResultsFunction(&resultCount, &labelingResult, resultHandle, 1));

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_ExecuteLabelingFunction(&labelingAreaCount, &resultHandle, imageId));

    LabelingResult *resultList = NULL;
    if (labelingAreaCount > 0)
    {
        resultList = new LabelingResult[labelingAreaCount];

        EXPECT_EQ(CaptureResult::CaptureResult_Success,
            captureProcessorWrapper.m_GetLabelingResultsFunction(&resultCount, resultList, resultHandle, labelingAreaCount));

        // 取得結果数チェック
        EXPECT_EQ(labelingAreaCount, resultCount);

        // 数を減らして実行
        int harfResultCount = labelingAreaCount / 2;
        EXPECT_EQ(CaptureResult::CaptureResult_Success,
            captureProcessorWrapper.m_GetLabelingResultsFunction(&resultCount, resultList, resultHandle, harfResultCount));
        EXPECT_EQ(harfResultCount, resultCount);

        if (resultList != NULL)
        {
            delete[] resultList;
        }
    }

    // 不正な画像 ID に対して実行
    ImageIdType invalidImageId = { 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidImageId,
        captureProcessorWrapper.m_ExecuteLabelingFunction(&labelingAreaCount, &resultHandle, invalidImageId));

    // 不正なラベリング結果ハンドルに対して実行
    LabelingResultHandleType invalidResultHandle = { 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidLabelingResultHandle,
        captureProcessorWrapper.m_GetLabelingResultsFunction(&resultCount, &labelingResult, invalidResultHandle, 1));

    EXPECT_EQ(CaptureResult::CaptureResult_InvalidLabelingResultHandle,
        captureProcessorWrapper.m_ReleaseLabelingResultFunction(invalidResultHandle));

    // ラベリング処理結果解放
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_ExecuteLabelingFunction(&labelingAreaCount, &resultHandle, imageId));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_ReleaseLabelingResultFunction(resultHandle));

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}
