﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "CaptureProcessorWrapper.h"

using namespace captureProcessor;

TEST(CaptureProcessorrSuite, GetDotColorTest)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;
    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int dotColor[ColorChannelCount] = { 0 };
    int lowerColor[ColorChannelCount] = { 0 };
    int upperColor[ColorChannelCount] = { 0 };
    ImageIdType imageId = { 0 };

    int xPosition = 0;
    int yPosition = 0;

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, xPosition, yPosition));
    EXPECT_EQ(MatchingResult::MatchingResult_NotInitialized,
        captureProcessorWrapper.m_DetectDotColorFunction(xPosition, yPosition, lowerColor, upperColor));
    EXPECT_EQ(MatchingResult::MatchingResult_NotInitialized,
        captureProcessorWrapper.m_WaitDetectDotColorFunction(xPosition, yPosition, lowerColor, upperColor, 1000));


    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(0, false, CaptureResolution::CaptureResolution_720p);
    // 不正な画像 ID に対しての実行
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidImageId,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, xPosition, yPosition));

    // 正常な画像 ID に対しての実行
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, xPosition, yPosition));

    // 範囲外の座標に対して実行
    int width = 0;
    int height = 0;
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageSizeFunction(&width, &height, imageId));

    EXPECT_EQ(CaptureResult::CaptureResult_Unexpected,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, -1, 0));
    EXPECT_EQ(CaptureResult::CaptureResult_Unexpected,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, 0, -1));
    EXPECT_EQ(CaptureResult::CaptureResult_Unexpected,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, width, 0));
    EXPECT_EQ(CaptureResult::CaptureResult_Unexpected,
        captureProcessorWrapper.m_GetDotColorFunction(dotColor, imageId, 0, height));


    // 左上の画素値をチェック
    // TIPS: アニメーションが無い前提で、取得した静止画のが措置を元に行う
    unsigned char *imageBuffer = new unsigned char[width * height * ColorChannelCount];
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageDataFunction(imageBuffer, width * height * ColorChannelCount, imageId));
    int expectLowerColor[3] = { 0 };
    int expectUpperColor[3] = { 0 };
    int invalidLowerColor[3] = { 0 };
    int invalidUpperColor[3] = { 0 };
    for (int i = 0; i < ColorChannelCount; i++)
    {
        expectLowerColor[i] = static_cast<int>(imageBuffer[i] - 10);
        if (expectLowerColor[i] < 0)
        {
            expectLowerColor[i] = 0;
        }
        expectUpperColor[i] = static_cast<int>(imageBuffer[i] + 10);
        if (expectUpperColor[i] > 255)
        {
            expectUpperColor[i] = 255;
        }

        invalidLowerColor[i] = (static_cast<int>(imageBuffer[i]) + 128) % 256;
        if ((invalidLowerColor[i] + 20) > 255)
        {
            invalidLowerColor[i] = 235;
        }

        invalidUpperColor[i] = invalidLowerColor[i] + 20;
    }
    delete[] imageBuffer;

    // 一致を期待した DetectDotColor 実行
    EXPECT_EQ(MatchingResult::MatchingResult_Success,
        captureProcessorWrapper.m_DetectDotColorFunction(0, 0, expectLowerColor, expectUpperColor));
    EXPECT_EQ(MatchingResult::MatchingResult_Success,
        captureProcessorWrapper.m_WaitDetectDotColorFunction(0, 0, expectLowerColor, expectUpperColor, 1000));
    // 不一致を期待した DetectDotColor の実行
    EXPECT_EQ(MatchingResult::MatchingResult_Failed,
        captureProcessorWrapper.m_DetectDotColorFunction(0, 0, invalidLowerColor, invalidUpperColor));
    EXPECT_EQ(MatchingResult::MatchingResult_Failed,
        captureProcessorWrapper.m_WaitDetectDotColorFunction(0, 0, invalidLowerColor, invalidUpperColor, 1000));


    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}
