﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "CaptureProcessorWrapper.h"
#include "testCaptureProcessor_TestDefinition.h"
#include "../Common/testAutoTestAssistTool_Util.h"

using namespace captureProcessor;


TEST(CaptureProcessorSuite, DetectObjectTestInUninitialized)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int positionX;
    int positionY;
    ImageIdType imageId = { 0 };
    MatchingResult matchingResult;

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    matchingResult = captureProcessorWrapper.m_DetectObjectFunction(&positionX, &positionY, imageId, 0.8);
    EXPECT_EQ(matchingResult, MatchingResult::MatchingResult_NotInitialized);

    matchingResult = captureProcessorWrapper.m_WaitDetectObjectFunction(&positionX, &positionY, imageId, 0.8, 1000);
    EXPECT_EQ(matchingResult, MatchingResult::MatchingResult_NotInitialized);

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

TEST_P(CaptureModeTest, DetectObjectTest)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int positionX;
    int positionY;
    ImageIdType correctImageId = { 0 };
    ImageIdType wrongImageId = { 0 };


    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    auto mode = GetParam();
    std::string correctImagePath;
    std::string wrongImagePath;
    if (mode == CaptureMode::CaptureMode_720p)
    {
        correctImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        wrongImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_720p);
    }
    else if (mode == CaptureMode::CaptureMode_1080p)
    {
        correctImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        wrongImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_1080p);
    }
    else if (mode == CaptureMode::CaptureMode_DevKit)
    {
        int     argc = nnt::GetHostArgc();
        char**  argv = nnt::GetHostArgv();
        SerialNumberString serialNumber;
        ASSERT_GE(argc, 2);
        strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

        correctImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        wrongImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
            serialNumber, false), CaptureResult::CaptureResult_Success);
    }
    else
    {
        ASSERT_TRUE(false);
    }

    // 画像読み込み
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &correctImageId, correctImagePath.c_str()), CaptureResult::CaptureResult_Success);
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &wrongImageId, wrongImagePath.c_str()), CaptureResult::CaptureResult_Success);

    // テンプレートマッチング(成功)
    EXPECT_EQ(captureProcessorWrapper.m_DetectObjectFunction(
        &positionX, &positionY, correctImageId, 0.8), MatchingResult::MatchingResult_Success);

    EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectFunction(
        &positionX, &positionY, correctImageId, 0.8, 1000), MatchingResult::MatchingResult_Success);


    // テンプレートマッチング(失敗)
    EXPECT_EQ(captureProcessorWrapper.m_DetectObjectFunction(
        &positionX, &positionY, wrongImageId, 0.8), MatchingResult::MatchingResult_Failed);

    EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectFunction(
        &positionX, &positionY, wrongImageId, 0.8, 1000), MatchingResult::MatchingResult_Failed);


    //-------------------------------------
    // 不正なパラメーターでの実行
    //-------------------------------------
    // 不正な画像 ID に対して実行
    ImageIdType invalidImageId = { 0 };
    EXPECT_EQ(captureProcessorWrapper.m_DetectObjectFunction(&positionX, &positionY, invalidImageId, 0.8),
        MatchingResult::MatchingResult_InvalidImageId);

    EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectFunction(&positionX, &positionY, invalidImageId, 0.8, 1000),
        MatchingResult::MatchingResult_InvalidImageId);


    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

