﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Collections.Generic;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Xml.Serialization;
using System.Xml;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.AuthoringTool;
using Nintendo.Authoring.ETicketLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ExcecMultipleNspTest : ExcecutionTestBase
    {
        [TestInitialize]
        public void ForceGC()
        {
            ForceGCImpl();
        }

        [TestMethod]
        public void TestExecutionProdEncryptionMultiApplicationCard()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var uppV201327042 = env.SourceDir + "\\UppTest\\dummyUpp_v201327042.nsp";
            var uppV268435906 = env.SourceDir + "\\UppTest\\dummyUpp_v268435906.nsp";
            var metaFileSource = env.SourceDir + "\\UppTest\\describe_all_required_system_version.nmeta";

            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            var aocMetaFile = env.OutputDir + "\\multiple_aoc_required_version.nmeta";
            File.Copy(env.SourceDir + "\\UppTest\\multiple_aoc_required_version.nmeta", aocMetaFile);

            var appIds = new ulong[]
            {
                0x01004b9000490000UL,
                0x01004b90004A0000UL,
                0x01004b90004B0000UL,
                0x01004b90004C0000UL,
                0x01004b90004D0000UL,
                0x01004b90004E0000UL,
                0x01004b90004F0000UL,
                0x01004b9000500000UL,
            };

            Func<ulong, string, string> getNspPath = (appId, suffix) =>
            {
                return env.OutputDir + "\\0x" + appId.ToString("x16") + suffix + ".nsp";
            };

            foreach (var appId in appIds)
            {
                var metaFileV0 = env.OutputDir + "\\" + Path.GetFileName(metaFileSource).Replace(".nmeta", "_v0.nmeta");
                var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFileSource).Replace(".nmeta", "_v1.nmeta");
                MakeSpecifiedVersionMetaFile(metaFileSource, metaFileV0, 0, appId);
                MakeSpecifiedVersionMetaFile(metaFileSource, metaFileV1, 65536, appId);

                var v0 = getNspPath(appId, "_v0");
                var v1 = getNspPath(appId, "_v1");
                var v1patch = getNspPath(appId, "_v1_patch");

                // アプリ・パッチ作成
                CreateNsp(v0, env, metaFileV0, string.Format(" --keygeneration 2 --accessible-urls {0} --legal-information {1}", accessibleUrlsDir, legalInformationZip));
                CreateNsp(v1, env, metaFileV1, string.Format(" --keygeneration 2 --accessible-urls {0} --legal-information {1}", accessibleUrlsDir, legalInformationZip));
                CreatePatch(v1patch, env, v0, v1, string.Empty);
                var error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --upp {2} --original {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, uppV201327042, v0, v1patch));
                Assert.IsTrue(error == string.Empty, error);
            }

            // Digest 不一致テストのため 0x01004b9000490000 はもう一組作成
            {
                var appId = 0x01004b9000490000UL;
                var metaFileV0 = env.OutputDir + "\\" + Path.GetFileName(metaFileSource).Replace(".nmeta", "_v0.nmeta");
                var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFileSource).Replace(".nmeta", "_v1.nmeta");
                var v0 = getNspPath(appId, "_v0_alt");
                var v1 = getNspPath(appId, "_v1_alt");
                var v1patch = getNspPath(appId, "_v1_patch_alt");
                // アプリ・パッチ作成（legal-information を抜く）
                CreateNsp(v0, env, metaFileV0, string.Format(" --accessible-urls {0}", accessibleUrlsDir));
                CreateNsp(v1, env, metaFileV1, string.Format(" --accessible-urls {0}", accessibleUrlsDir));
                CreatePatch(v1patch, env, v0, v1, string.Empty);
            }

            Func<string, string, string> bundleup = (nsp, args) =>
            {
                return ExecuteProgram(string.Format("bundleup -o {0} {1}", nsp, args));
            };

            var bundleNsp = env.OutputDir + "\\bundle.nsp";

            // ID 重複でエラー（提出用 ID）
            {
                var appId = 0x01004b9000490000UL;
                var appIdStr = "0x" + appId.ToString("x16");

                var patchId = 0x01004b9000490800UL;
                var patchIdStr = "0x" + patchId.ToString("x16");

                var v0 = getNspPath(appId, "_v0");
                var v1patch = getNspPath(appId, "_v1_patch");

                var error = bundleup(bundleNsp, $"--id {appIdStr} {v0} {v1patch}");
                Assert.IsTrue(error.IndexOf("ID for multi-application card must be unique.") >= 0, error);
                error = bundleup(bundleNsp, $"--id {patchIdStr} {v0} {v1patch}");
                Assert.IsTrue(error.IndexOf("ID for multi-application card must be unique.") >= 0, error);
            }

            var id = "0x02004b9000490000";
            var fullArg = string.Join(" ", appIds.SelectMany(x => new string[] { getNspPath(x, "_v0"), getNspPath(x, "_v1_patch") }));

            // ID 重複でエラー（アプリ）
            {
                var v0 = getNspPath(0x01004b9000490000UL, "_v0");
                var error = bundleup(bundleNsp, $"--id {id} {fullArg} {v0}");
                Assert.IsTrue(error.IndexOf("Application IDs in multi-application card cannot be duplicated.") >= 0, error);
            }

            // ID 重複でエラー（パッチ）
            {
                var v1patch = getNspPath(0x01004b9000490000UL, "_v1_patch");
                var error = bundleup(bundleNsp, $"--id {id} {fullArg} {v1patch}");
                Assert.IsTrue(error.IndexOf("Patch IDs in multi-application card cannot be duplicated.") >= 0, error);
            }

            // パッチのみバンドルでエラー
            {
                var v0 = getNspPath(0x01004b9000490000UL, "_v0");
                var fullArgRemoveApp = fullArg.Replace(v0, string.Empty);
                var error = bundleup(bundleNsp, $"--id {id} {fullArgRemoveApp}");
                Assert.IsTrue(error.IndexOf("There is no Application corresponds to Patch 0x01004b9000490800.") >= 0, error);
            }

            // アプリ・パッチ以外を含めようとするとエラー
            {
                var aocNsp = env.OutputDir + "\\aoc.nsp";
                var error = ExecuteProgram(string.Format("creatensp -o {0} --type AddOnContent --meta {1}", aocNsp, aocMetaFile));
                Assert.IsTrue(error == string.Empty, error);
                error = bundleup(bundleNsp, $"--id {id} {fullArg} {aocNsp}");
                Assert.IsTrue(error.IndexOf("Only Application or Patch can be bundled.") >= 0, error);
            }

            // 各項目が正しく入力されていること
            var macAppIds = new List<string>();
            var macPatchIds = new List<string>();
            {
                var error = bundleup(bundleNsp, $"--id {id} {fullArg} --cardsize 16 --cardlaunchflags 1");
                Assert.IsTrue(error == string.Empty, error);

                using (var nspReader = new NintendoSubmissionPackageReader(bundleNsp))
                {
                    var model = nspReader.ListFileInfo().Where(x => x.Item1 == "multiapplicationcard.xml").Select(x => ArchiveReconstructionUtils.ReadXml<MultiApplicationCardInfoModel>(nspReader, x.Item1, x.Item2)).First();
                    Assert.AreEqual(id, model.Id);
                    Assert.AreEqual(appIds.Count(), model.ApplicationCount);
                    foreach (var appId in appIds)
                    {
                        ContentMetaModel appContentMeta;
                        ContentMetaModel patchContentMeta;
                        using (var appNspReader = new NintendoSubmissionPackageReader(getNspPath(appId, "_v0")))
                        {
                            appContentMeta = ArchiveReconstructionUtils.ReadContentMetaInNsp(appNspReader).Single();
                        }
                        using (var patchNspReader = new NintendoSubmissionPackageReader(getNspPath(appId, "_v1_patch")))
                        {
                            patchContentMeta = ArchiveReconstructionUtils.ReadContentMetaInNsp(patchNspReader).Single();
                        }
                        Assert.IsTrue(model.Application.ContentMetaList.Any(x => (Convert.ToUInt64(x.Id, 16) == appContentMeta.GetUInt64Id()) && x.GetDigestBytes().SequenceEqual(appContentMeta.GetDigestBytes())));
                        macAppIds = model.Application.ContentMetaList.Select(x => x.Id).ToList();
                        Assert.IsTrue(model.Patch.ContentMetaList.Any(x => (Convert.ToUInt64(x.Id, 16) == patchContentMeta.GetUInt64Id()) && x.GetDigestBytes().SequenceEqual(patchContentMeta.GetDigestBytes())));
                        macPatchIds = model.Patch.ContentMetaList.Select(x => x.Id).ToList();
                    }
                    var zeroBytes = new byte[32];
                    Assert.IsTrue(!zeroBytes.SequenceEqual(model.GetDigestBytes()));
                }
            }

            Action<string, ulong[], bool> checkXciResultXml = (nsp, refAppIds, includePatch) =>
            {
                var xml = nsp.Replace(".nsp", "_prod.xci.result.xml");
                CheckXmlEncoding(xml);
                ResultModel model;
                using (var fs = new FileStream(xml, FileMode.Open, FileAccess.Read))
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    model = (ResultModel)serializer.Deserialize(fs);
                }
                Assert.IsTrue(model.CardHeader != null);
                Assert.IsTrue(model.CardHeader.RomSize == XciUtils.GetBytesString(XciInfo.RomSize16GB));
                Assert.IsTrue(model.CardHeader.Flags == XciUtils.GetBytesString((byte)1));
                foreach (var appId in refAppIds)
                {
                    var appIdStr = "0x" + appId.ToString("x16");
                    Assert.IsTrue(model.ContentMetaList.Any(x => x.Id == appIdStr));
                    if (includePatch)
                    {
                        var patchIdStr = "0x" + (appId + 0x800).ToString("x16");
                        Assert.IsTrue(model.ContentMetaList.Any(x => x.Id == patchIdStr));
                    }
                }
                // 順番が multiapplicationcardinfo.xml の順に並び替えられていること
                for (int i = 0; i < model.ContentMetaList.Count; i++)
                {
                    if (i < macAppIds.Count)
                    {
                        Assert.IsTrue(model.ContentMetaList[i].Id == macAppIds[i]);
                    }
                    else if (i < macAppIds.Count + macPatchIds.Count)
                    {
                        Assert.IsTrue(model.ContentMetaList[i].Id == macPatchIds[i - macAppIds.Count]);
                    }
                    else
                    {
                        Assert.IsTrue(false);
                    }

                    // Patch の RequiredSystemVersion が UPP によって引き上げられていること
                    if (model.ContentMetaList[i].Type == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        Assert.AreEqual((model.ContentMetaList[i] as PatchContentMetaModel).RequiredSystemVersion, 201327042 & 0xFFFF0000);
                    }
                }
            };

            // XCI 作成
            {
                var cmdBase = string.Format("prodencryption -o {0} --keyconfig {1} --no-check --no-padding --upp {2} --multiapplicationgamecard {3} ", env.OutputDir, env.DefaultKeyConfigFile, uppV268435906, bundleNsp);

                var v0 = getNspPath(0x01004b9000490000UL, "_v0");
                var v1patch = getNspPath(0x01004b9000490000UL, "_v1_patch");
                var v0Alt = getNspPath(0x01004b9000490000UL, "_v0_alt");
                var v1patchAlt = getNspPath(0x01004b9000490000UL, "_v1_patch_alt");

                // アプリの入力不足
                var error = ExecuteProgram(cmdBase + fullArg.Replace(v0, string.Empty));
                Assert.IsTrue(error.IndexOf("There is no consistency between multi-application card indicator and input nsp files. Application count mismatch.") >= 0, error);

                // パッチの入力不足
                error = ExecuteProgram(cmdBase + fullArg.Replace(v1patch, string.Empty));
                Assert.IsTrue(error.IndexOf("There is no consistency between multi-application card indicator and input nsp files. Patch count mismatch.") >= 0, error);

                // アプリの Digest 不一致
                error = ExecuteProgram(cmdBase + fullArg.Replace(v0, v0Alt));
                Assert.IsTrue(error.IndexOf("There is no consistency between multi-application card indicator and input nsp files. Application digest mismatch.") >= 0, error);

                // パッチの Digest 不一致
                error = ExecuteProgram(cmdBase + fullArg.Replace(v1patch, v1patchAlt));
                Assert.IsTrue(error.IndexOf("There is no consistency between multi-application card indicator and input nsp files. Patch digest mismatch.") >= 0, error);

                // 成功（入力順を並び替える）
                var fullArgReplaced = fullArg.Replace(v0, string.Empty).Replace(v1patch, string.Empty) + string.Format(" {0} {1}", v1patch, v0);
                error = ExecuteProgram(cmdBase + fullArgReplaced.Replace("patch.nsp", "patch_prod.nsp"));
                Assert.IsTrue(error == string.Empty, error);

                checkXciResultXml(bundleNsp, appIds, true);
            }

            // アプリだけでも OK
            {
                var fullArgOnlyApp = string.Join(" ", appIds.SelectMany(x => new string[] { getNspPath(x, "_v0") }));
                var error = bundleup(bundleNsp, $"--id {id} {fullArgOnlyApp} --cardsize 16 --cardlaunchflags 1");
                Assert.IsTrue(error == string.Empty, error);
                var cmdBase = string.Format("prodencryption -o {0} --keyconfig {1} --no-check --no-padding --upp {2} --multiapplicationgamecard {3} ", env.OutputDir, env.DefaultKeyConfigFile, uppV268435906, bundleNsp);
                error = ExecuteProgram(cmdBase + fullArgOnlyApp);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(bundleNsp, appIds, false);
            }

            SafeDeleteDirectory(env.OutputDir);
        }

        [TestMethod]
        public void TestExecutionMergeApplication()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFileSource = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = env.DefaultKeyConfigFile;

            Func<string, string> getNspPath = (suffix) =>
            {
                return env.OutputDir + "\\" + suffix + ".nsp";
            };

            const int UnitNum = 4;
            const int FileSize = 1024;
            var fileData = new byte[FileSize];
            for (int i = 0; i < FileSize; i++)
            {
                fileData[i] = (byte)i;
            }
            var rng = new RNGCryptoServiceProvider();

            for (int i = 0; i < UnitNum; i++)
            {
                var addFile = env.TestCodeDir + string.Format("\\test{0}.dat", i);
                using (FileStream stream = File.Create(addFile))
                {
                    stream.Write(fileData, 0, FileSize);
                }
                var metaFile = env.OutputDir + string.Format("\\test{0}.nmeta", i);
                MakeMetaFileWithProgramIndex(metaFileSource, metaFile, (byte)i, 0);
                var nsp = getNspPath(string.Format("test{0}", i));
                var htmlArg = (i == 0 || i == 2) ? string.Format("--accessible-urls {0}", accessibleUrlsDir) : string.Empty;
                CreateNsp(nsp, env, metaFile, string.Format(" {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2}", htmlArg, legalInformationZip, iconPath));
                File.Delete(addFile);
            }

            for (int i = 0; i < UnitNum; i++)
            {
                var addFile = env.TestCodeDir + string.Format("\\test{0}.dat", i);
                using (FileStream stream = File.Create(addFile))
                {
                    var bytes = new byte[FileSize];
                    rng.GetBytes(bytes);
                    stream.Write(bytes, 0, FileSize);
                }
                var metaFile = env.OutputDir + string.Format("\\test{0}.nmeta", i);
                MakeMetaFileWithProgramIndex(metaFileSource, metaFile, (byte)i, 65536);
                var nsp = getNspPath(string.Format("test{0}_v1", i));
                // 0: html 有 ⇒ 有
                // 1: html 無 ⇒ 有
                // 2: html 有 ⇒ 無
                // 3: html 無 ⇒ 無
                var htmlArg = (i == 0 || i == 1) ? string.Format("--accessible-urls {0}", accessibleUrlsDir) : string.Empty;
                CreateNsp(nsp, env, metaFile, string.Format(" {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2}", htmlArg, legalInformationZip, iconPath));

                if (i == 0)
                {
                    // test0 だけ v2 も作る
                    MakeMetaFileWithProgramIndex(metaFileSource, metaFile, (byte)i, 131072);
                    nsp = getNspPath(string.Format("test{0}_v2", i));
                    CreateNsp(nsp, env, metaFile, string.Format(" --accessible-urls {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2}", accessibleUrlsDir, legalInformationZip, iconPath));
                }

                File.Delete(addFile);
            }

            // ProgramIndex 4 は ApplicationId 違い
            {
                var metaFile = env.OutputDir + "\\test4.nmeta";
                MakeMetaFileWithProgramIndex(metaFileSource, metaFile, 4, 0, 0x0005000C20000001);
                var nsp = getNspPath("test4");
                CreateNsp(nsp, env, metaFile, string.Format(" --accessible-urls {0} --legal-information {1} --icon AmericanEnglish {2} Japanese {2}", accessibleUrlsDir, legalInformationZip, iconPath));
            }

            Func<string, string, string> mergeApplication = (nsp, args) =>
            {
                return ExecuteProgram(string.Format("mergeapplication -o {0} {1}", nsp, args));
            };

            // multi-program application
            var mpa = env.OutputDir + "\\mpa.nsp";
            var mpa2 = env.OutputDir + "\\mpa_v1.nsp";
            var mpa3 = env.OutputDir + "\\mpa_v2.nsp";
            var fullArg = string.Join(" ", Enumerable.Range(0, 4).Select(x => getNspPath(string.Format("test{0}", x))));

            // ApplicationId 不一致
            {
                var error = mergeApplication(mpa, $"{fullArg} {getNspPath("test4")}");
                Assert.IsTrue(error.IndexOf("Only one Application ID can be used in multi-program application.") >= 0, error);
            }
            // ProgramIndex 重複
            {
                var error = mergeApplication(mpa, $"{fullArg} {getNspPath("test1")}");
                Assert.IsTrue(error.IndexOf("ProgramIndex of application must be serial and start at 0 in multi-program application.") >= 0, error);
            }
            // ProgramIndex 欠番
            {
                var fullArgRemoveIndex = fullArg.Replace(getNspPath("test2"), string.Empty);
                var error = mergeApplication(mpa, $"{fullArgRemoveIndex}");
                Assert.IsTrue(error.IndexOf("ProgramIndex of application must be serial and start at 0 in multi-program application.") >= 0, error);
            }
            // ProgramIndex 0 無し
            {
                var fullArgRemoveIndex = fullArg.Replace(getNspPath("test0"), string.Empty);
                var error = mergeApplication(mpa, $"{fullArgRemoveIndex}");
                Assert.IsTrue(error.IndexOf("ProgramIndex of application must be serial and start at 0 in multi-program application.") >= 0, error);
            }

            {
                var error = mergeApplication(mpa, $"{fullArg}");
                Assert.IsTrue(error == string.Empty, error);

                using (var nspReader = new NintendoSubmissionPackageReader(mpa))
                {
                    var contentMeta = ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader).Single();
                    var fileInfos = nspReader.ListFileInfo();

                    // ContentMeta にコンテンツが不足なく含まれている
                    Assert.AreEqual(contentMeta.ContentList.Count, fileInfos.Where(x => x.Item1.EndsWith(".nca")).Count());

                    // ProgramIndex と secureValue が対応している
                    foreach (var content in contentMeta.ContentList.Where(x => x.Type != NintendoContentMetaConstant.ContentTypeMeta))
                    {
                        var ncaReader = nspReader.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration()));
                        Assert.AreEqual(ncaReader.GetRepresentProgramIdOffset(), content.IdOffset);
                    }

                    // 個別アプリと同じ内容物（legalinfo.nca, cnmt*, legalinfo.xml, cardspec.xml, authoringtoolinfo.xml は除く）が含まれている
                    Func<string, bool> excludes = (name) =>
                    {
                        return !name.EndsWith("cnmt.nca") &&
                               !name.EndsWith("cnmt.xml") &&
                               name != "cardspec.xml" &&
                               name != "authoringtoolinfo.xml";
                    };
                    for (int i = 0; i < UnitNum; i++)
                    {
                        using (var individualNspReader = new NintendoSubmissionPackageReader(getNspPath(string.Format("test{0}", i))))
                        {
                            var legalinfoContentId = ArchiveReconstructionUtils.GetSpecifiedContentId(individualNspReader, contentMeta.GetUInt64Id(), NintendoContentMetaConstant.ContentTypeLegalInformation, (byte)i);
                            foreach (var fileName in individualNspReader.ListFileInfo().Where(x => excludes(x.Item1)).Select(x => x.Item1))
                            {
                                // idOffset == 0 以外のリーガル情報は含まれていない
                                if (fileName.StartsWith(legalinfoContentId) && i != 0)
                                {
                                    Assert.IsFalse(fileInfos.Any(x => x.Item1 == fileName));
                                }
                                else
                                {
                                    Assert.IsTrue(fileInfos.Any(x => x.Item1 == fileName));
                                }
                            }
                        }
                    }
                }
            }

            {
                // extract
                var extractDir = env.OutputDir + "\\extract";
                var error = ExecuteProgram(string.Format("extract {0} -o {1}", mpa, extractDir));
                Assert.IsTrue(error == string.Empty, error);

                // replace
                {
                    var replaceDir = env.OutputDir + "\\replace";
                    var extractDir2 = env.OutputDir + "\\extract2";
                    var replaceEntryInNspTestList = new Tuple<Regex, bool>[] {
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/test0.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/test1.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/test2.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/test3.dat"), true),
                        new Tuple<Regex, bool>(new Regex(@".*\.nca/fs0/Application_0005000c10000001.cnmt"), false),
                    };

                    TestReplaceWithNspOrNca(mpa, replaceDir, extractDir, extractDir2, replaceEntryInNspTestList, new List<int[]>() { new int[] { 0, 1, 2, 3 } });
                    // Index が維持されること
                    using (var nspReader = new NintendoSubmissionPackageReader(mpa.Replace("mpa.nsp", "\\replace\\mpa_replaced.nsp")))
                    {
                        var contentMeta = ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader).Single();
                        foreach (var content in contentMeta.ContentList.Where(x => x.Type == NintendoContentMetaConstant.ContentTypeProgram))
                        {
                            var ncaReader = nspReader.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration()));
                            var fileList = ncaReader.OpenFileSystemArchiveReader(0).ListFileInfo();
                            if (fileList.Any(x => x.Item1.EndsWith("test0.dat")))
                            {
                                Assert.AreEqual(ncaReader.GetRepresentProgramIdOffset(), 0);
                            }
                            else if (fileList.Any(x => x.Item1.EndsWith("test1.dat")))
                            {
                                Assert.AreEqual(ncaReader.GetRepresentProgramIdOffset(), 1);
                            }
                            else if (fileList.Any(x => x.Item1.EndsWith("test2.dat")))
                            {
                                Assert.AreEqual(ncaReader.GetRepresentProgramIdOffset(), 2);
                            }
                            else if (fileList.Any(x => x.Item1.EndsWith("test3.dat")))
                            {
                                Assert.AreEqual(ncaReader.GetRepresentProgramIdOffset(), 3);
                            }
                            else
                            {
                                Assert.IsTrue(false);
                            }
                        }
                    }

                    TestReplaceWithNspOrNca(mpa, replaceDir, extractDir, extractDir2, replaceEntryInNspTestList, new List<int[]>() { new int[] { 4 } });

                    // nca 差し替え
                    var entryList = GetEntryFilePathList(extractDir2);
                    var targetName = "test0.dat";
                    var ncaName = entryList.Find(entry => entry.EndsWith(targetName)).Replace("/fs0/" + targetName, "");
                    var replaceDir2 = env.OutputDir + "\\replace_nca";
                    TestReplaceNcaInNsp(mpa, replaceDir2, ncaName, targetName);
                }
            }

            // patch
            var patchV1 = mpa2.Replace(".nsp", "_patch.nsp");
            var patchV2 = mpa3.Replace(".nsp", "_patch.nsp");
            {

                // ProgramIndex 3 は v2 で追加する
                var fullArgV0 = fullArg.Replace(getNspPath("test3"), string.Empty);
                var error = mergeApplication(mpa, $"{fullArgV0}");
                Assert.IsTrue(error == string.Empty, error);

                var fullArgV1 = fullArgV0.Replace("test0.nsp", "test0_v1.nsp").Replace("test2.nsp", "test2_v1.nsp");
                error = mergeApplication(mpa2, $"{fullArgV1}");
                Assert.IsTrue(error == string.Empty, error);

                var fullArgV2 = fullArgV1.Replace("test0_v1.nsp", "test0_v2.nsp").Replace("test1.nsp", "test1_v1.nsp") + getNspPath("test3_v1");
                error = mergeApplication(mpa3, $"{fullArgV2}");
                Assert.IsTrue(error == string.Empty, error);

                error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", patchV1, env.DefaultDescFile, mpa, mpa2));
                Assert.IsTrue(error == string.Empty, error);

                error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --previous {3} --current {4}", patchV2, env.DefaultDescFile, mpa, patchV1, mpa3));
                Assert.IsTrue(error == string.Empty, error);

                var delta = patchV2.Replace("patch", "delta");
                error = ExecuteProgram(string.Format("makedelta -o {0} --source {1} --destination {2}", delta, patchV1, patchV2));
                Assert.IsTrue(error == string.Empty, error);

                // 中身が修正版個別アプリと一致していることの確認
                Action<string, string, string> checkPatchIntegrity = (original, current, currentPatch) =>
                {
                    using (var originalReader = new NintendoSubmissionPackageReader(original))
                    using (var currentReader = new NintendoSubmissionPackageReader(current))
                    using (var currentPatchReader = new PatchedNintendoSubmissionPackageReader(currentPatch, originalReader))
                    {
                        var contentMeta = ArchiveReconstructionUtils.ReadContentMetaInNsp(currentReader).Single();
                        var contentMetaPatch = ArchiveReconstructionUtils.ReadContentMetaInNsp(currentPatchReader).Single();
                        foreach (var patchedContent in contentMetaPatch.ContentList.Where(x => x.Type != NintendoContentMetaConstant.ContentTypeMeta))
                        {
                            var content = contentMeta.ContentList.Single(x => x.Type == patchedContent.Type && x.IdOffset == patchedContent.IdOffset);
                            var ncaReader = currentReader.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration()));
                            var patchedNcaReaders = currentPatchReader.OpenPatchableNintendoContentArchiveReader(patchedContent.Id + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration()));
                            foreach (var i in patchedNcaReaders.Item1.GetExistentFsIndices())
                            {
                                using (var fsReader = ncaReader.OpenFileSystemArchiveReader(i))
                                using (var patchedFsReader = patchedNcaReaders.Item1.OpenFileSystemArchiveReader(i, patchedNcaReaders.Item2))
                                {
                                    const long BufferSize = 8 * 1024 * 1024;
                                    long restSize = fsReader.GetBaseSize();
                                    long offset = 0;
                                    while (restSize > 0)
                                    {
                                        long readSize = Math.Min(restSize, BufferSize);
                                        var bytesA = fsReader.ReadBase(offset, readSize);
                                        var bytesB = patchedFsReader.ReadBase(offset, readSize);
                                        Assert.IsTrue(bytesA.SequenceEqual(bytesB));
                                        restSize -= readSize;
                                        offset += readSize;
                                    }
                                }
                            }
                        }
                    }
                };

                checkPatchIntegrity(mpa, mpa2, patchV1);
                checkPatchIntegrity(mpa, mpa3, patchV2);

                // 差分パッチが意図通り作成されていることの確認（cnmt.xml での確認）
                var deltaMeta = (DeltaContentMetaModel)ArchiveReconstructionUtils.ReadContentMetaInNsp(delta).Single();
                var patchV1Meta = (PatchContentMetaModel)ArchiveReconstructionUtils.ReadContentMetaInNsp(patchV1).Single();
                var patchV2Meta = (PatchContentMetaModel)ArchiveReconstructionUtils.ReadContentMetaInNsp(patchV2).Single();
                {
                    // 0: 変更無
                    // 1: 変更有（Html 追加）
                    // 2: 変更無
                    // 3: 新規追加
                    foreach (var contentV2 in patchV2Meta.ContentList.Where(x => x.Type != NintendoContentMetaConstant.ContentTypeMeta))
                    {
                        var srcContentId = string.Empty;
                        var dstContentId = contentV2.Id;
                        if ((contentV2.IdOffset == 1 && contentV2.Type == NintendoContentMetaConstant.ContentTypeHtmlDocument) ||
                            contentV2.IdOffset == 3)
                        {
                            srcContentId = "00000000000000000000000000000000";
                        }
                        else
                        {
                            srcContentId = patchV1Meta.ContentList.Single(x => x.IdOffset == contentV2.IdOffset && x.Type == contentV2.Type).Id;
                        }
                        Assert.IsTrue(deltaMeta.FragmentSetList.Any(x => x.FragmentTargetContentType == contentV2.Type && x.Source.ContentId == srcContentId && x.Destination.ContentId == dstContentId));
                    }
                }
            }

            // prodencryption
            {
                var error = ExecuteProgram(string.Format("prodencryption -o {0} --keyconfig {1} --no-check --no-nspu {2}", env.OutputDir, env.DefaultKeyConfigFile, mpa));
                Assert.IsTrue(error == string.Empty, error);
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --keyconfig {1} --no-check --no-nspu --original {2} --original-prod {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, mpa, mpa.Replace(".nsp", "_prod.nsp"), patchV1));
                Assert.IsTrue(error == string.Empty, error);
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --keyconfig {1} --no-check --no-nspu --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, mpa, mpa.Replace(".nsp", "_prod.nsp"), patchV1.Replace(".nsp", "_prod.nsp"), patchV2));
                Assert.IsTrue(error == string.Empty, error);
                error = ExecuteProgram(string.Format("prodencryption -o {0} --gamecard --keyconfig {1} --no-check --no-xcie --patch {2} {3}", env.OutputDir, env.DefaultKeyConfigFile, patchV2.Replace(".nsp", "_prod.nsp"), mpa));
                Assert.IsTrue(error == string.Empty, error);
            }

            // SafeDeleteDirectory(env.OutputDir);
        }

        [TestMethod]
        public void TestExecutionMergeApplicationErrorUnpublishableNacpEqualCheck()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFileSource = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";

            // 指定されたnmetaの情報毎にnspを作成して、mergeapplicationを実行した時のエラー出力を返す
            Func<string[], string> getMergeErrorString = (string[] nmetaStrings) =>
            {
                List<string> nspFileList = new List<string>();
                // マージ用のnspを作成
                for (int index = 0; index < nmetaStrings.Count(); index++)
                {
                    var metaFile = env.OutputDir + "\\" + Path.GetFileName(metaFileSource) + $"_{index}.nmeta";
                    MakeFileImpl(metaFile, nmetaStrings[index]);

                    var nsp = env.OutputDir + "\\" + Path.GetFileName(metaFileSource) + $"_{index}.nsp";
                    CreateNsp(nsp, env, metaFile, " --ignore-unpublishable-error");
                    nspFileList.Add(nsp);
                }

                // nspをマージ
                var mergeNspFile = env.OutputDir + "\\" + Path.GetFileName(metaFileSource) + "_merge.nsp";
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format("mergeapplication -o {0} {1}", mergeNspFile, string.Join(" ", nspFileList.ToArray())));
                // 例外が発生して nsp の生成に失敗する
                Assert.IsTrue(standardError.IndexOf("[Error] Found one or more unpublishable error in nsp. The nsp will be deleted automatically.") >= 0, standardError);
                Assert.IsFalse(File.Exists(mergeNspFile));

                return standardOut;
            };

            // ProgramIndex 0～2 の nmeta の中身が一致しない
            {
                string[] metaStrings =
                {
                    // ProgramIndex0 用
                    @"<?xml version=""1.0""?>
                      <NintendoSdkMeta>
                        <Core>
                          <ProgramId>0x01004b9000490000</ProgramId>
                        </Core>
                        <Application>
                          <Title>
                            <Language>Japanese</Language>
                            <Name>Title</Name>
                            <Publisher>Publisher</Publisher>
                          </Title>
                          <Title>
                            <Language>AmericanEnglish</Language>
                            <Name>Title</Name>
                            <Publisher>Publisher</Publisher>
                          </Title>
                          <DisplayVersion>1.0.0</DisplayVersion>
                          <SupportedLanguage>Japanese</SupportedLanguage>
                          <SupportedLanguage>AmericanEnglish</SupportedLanguage>
                          <ApplicationId>0x01004b9000490000</ApplicationId>
                          <ProgramIndex>0</ProgramIndex>
                        </Application>
                      </NintendoSdkMeta>",
                    // ProgramIndex1 用
                    @"<?xml version=""1.0""?>
                      <NintendoSdkMeta>
                        <Core>
                          <ProgramId>0x01004b9000490001</ProgramId>
                        </Core>
                        <Application>
                          <Title>
                            <Language>Japanese</Language>
                            <Name>Title1</Name>
                            <Publisher>Publisher</Publisher>
                          </Title>
                          <Title>
                            <Language>AmericanEnglish</Language>
                            <Name>Title2</Name>
                            <Publisher>Publisher</Publisher>
                          </Title>
                          <DisplayVersion>1.0.0</DisplayVersion>
                          <SupportedLanguage>Japanese</SupportedLanguage>
                          <SupportedLanguage>AmericanEnglish</SupportedLanguage>
                          <Rating>
                              <Organization>CERO</Organization>
                              <Age>18</Age>
                          </Rating>
                          <Hdcp>None</Hdcp>
                          <ApplicationId>0x01004b9000490000</ApplicationId>
                          <ProgramIndex>1</ProgramIndex>
                        </Application>
                      </NintendoSdkMeta>",
                    // ProgramIndex2 用
                    @"<?xml version=""1.0""?>
                      <NintendoSdkMeta>
                        <Core>
                          <ProgramId>0x01004b9000490002</ProgramId>
                        </Core>
                        <Application>
                          <Title>
                            <Language>Japanese</Language>
                            <Name>Title</Name>
                            <Publisher>Publisher1</Publisher>
                          </Title>
                          <Title>
                            <Language>AmericanEnglish</Language>
                            <Name>Title</Name>
                            <Publisher>Publisher</Publisher>
                          </Title>
                          <DisplayVersion>1.0.0</DisplayVersion>
                          <Hdcp>Required</Hdcp>
                          <ApplicationId>0x01004b9000490000</ApplicationId>
                          <ProgramIndex>2</ProgramIndex>
                        </Application>
                      </NintendoSdkMeta>",
                };

                var errorMessage = getMergeErrorString(metaStrings);

                // 下記のエラーが出力されることを確認する
                // UnpublishableError:
                //     ProgramIndex:0
                //       このコンテンツにはエラーはありません。
                //     ProgramIndex:1
                //       [Error] (Issue 10-816) アプリケーション管理データの不一致
                //         アプリケーション管理データに不一致の項目があります。特定の項目以外は全て共通の設定にしてください。
                //         <Title> が一致していません。
                //         <Rating> が一致していません。
                //     ProgramIndex:2
                //       [Error] (Issue 10-816) アプリケーション管理データの不一致
                //         アプリケーション管理データに不一致の項目があります。特定の項目以外は全て共通の設定にしてください。
                //         <Title> が一致していません。
                //         <SupportedLanguage> が一致していません。
                //         <Hdcp> が一致していません。

                // 期待値(実行環境によって英語で出力されるので日本語部分はワイルドカードで設定)
                var expectList = new List<string>
                {
                    "UnpublishableError:",
                    // ProgramIndex = 0 ではエラーが発生しない
                    " +ProgramIndex:0",
                    ".*",
                    // ProgramIndex = 1 ではエラーが発生する
                    " +ProgramIndex:1",
                    @" +\[Error\] \(Issue 10-816\).*",
                    ".*",
                    // ProgramIndex = 1 の Title/Name が一致しない
                    // (複数の Title が一致しない場合でもエラーは1つだけ表示される)
                    " +<Title>.*",
                    // ProgramIndex = 0 で指定されていない Rating が ProgramIndex = 1 に指定されている
                    " +<Rating>.*",
                    // ProgramIndex = 2 ではエラーが発生する
                    " +ProgramIndex:2",
                    @" +\[Error\] \(Issue 10-816\).*",
                    ".*",
                    // ProgramIndex = 2 の Title/Publisher が一致しない
                    " +<Title>.*",
                    // ProgramIndex = 0 で指定されている SupportedLanguage が ProgramIndex = 2 に指定されていない
                    " +<SupportedLanguage>.*",
                    // ProgramIndex = 0 で指定されていない Hdcp が ProgramIndex = 2 にデフォルト値ではない値で指定されている
                    // (ProgramIndex = 1 では Hdcp のデフォルト値で指定されているのでエラーにならない)
                    " +<Hdcp>.*",
                };

                // エラーメッセージを行単位に分割
                var errorMessageList = errorMessage.Split(new string[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries).ToList();
                // 「UnpublishableError:」の前に出力されたメッセージを削除
                errorMessageList = errorMessageList.Skip(errorMessageList.FindIndex(x => x == "UnpublishableError:")).ToList();

                // エラーメッセージのチェック
                Assert.IsTrue(expectList.Count() == errorMessageList.Count());
                for (int i = 0; i < expectList.Count(); i++)
                {
                    Assert.IsTrue((new Regex(expectList[i])).IsMatch(errorMessageList[i]), errorMessageList[i]);
                }
            }

            SafeDeleteDirectory(env.OutputDir);
        }

        [TestMethod]
        public void TestExecutionMergeApplicationErrorUnpublishable()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFileSource = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var legalInfoZipDir = env.SourceDir + "\\ErrorUnpublishable";
            var okAccessibleUrl = "^https://www.nintendo.co.jp";
            var ngAccessibleUrl = "https://www.nintendo.co.jp";
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");

            // 指定されたnspのUnpublishableErrorメッセージを取得
            Func<string, Tuple<string, string>> getUnpublishableErrorMessage = (string nspFile) =>
            {
                // get-unpublishable-error を実行してテキスト形式の出力を取得
                string standardError;
                string standardOutText;
                ExecuteProgram(
                    out standardError,
                    out standardOutText,
                    "get-unpublishable-error " + nspFile);
                Assert.IsTrue(string.IsNullOrWhiteSpace(standardError));

                // get-unpublishable-error を実行してXML形式の出力を取得
                string standardOutXmlWithUtf8 = string.Empty;
                {
                    Process process = new Process();
                    TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

                    process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\AuthoringTool\\AuthoringTool.exe";
                    process.StartInfo.Arguments = "get-unpublishable-error --xml " + nspFile;
                    process.StartInfo.CreateNoWindow = true;
                    process.StartInfo.UseShellExecute = false;
                    process.StartInfo.RedirectStandardOutput = true;
                    process.StartInfo.RedirectStandardError = false;
                    // XML形式はUTF8で出力されるのでエンコードをUTF8に変更する
                    process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;

                    process.Start();

                    standardOutXmlWithUtf8 = process.StandardOutput.ReadToEnd();
                    process.WaitForExit();
                }

                return Tuple.Create(standardOutText, standardOutXmlWithUtf8);
            };

            // 指定された情報(ソフトリーガル情報、URLリスト)毎にnspを作成してmergeし、作成されたnspパスを返す(元nsp1,元nsp2,...,結合後nsp)
            Func<string[], string[], List<string>> mergeNspAndGetUnpublishableError = (string[] legalInfoFiles, string[] accessibleUrls) =>
            {
                List<string> nspFileList = new List<string>();
                // マージ用のnspを作成
                for (int index = 0; index < legalInfoFiles.Count(); index++)
                {
                    var metaFile = env.OutputDir + "\\" + Path.GetFileName(metaFileSource) + $"_{index}.nmeta";
                    MakeMetaFileWithProgramIndex(metaFileSource, metaFile, (byte)index, 0);

                    MakeFileImpl(accessibleUrlsFilePath, accessibleUrls[index]);

                    var nsp = env.OutputDir + "\\" + Path.GetFileName(metaFileSource) + $"_{index}.nsp";
                    var legalInfoFile = legalInfoZipDir + "\\" + legalInfoFiles[index];
                    CreateNsp(nsp, env, metaFile, string.Format(" --legal-information {0} --accessible-urls {1}  --ignore-unpublishable-error", legalInfoFile, accessibleUrlsDir));
                    nspFileList.Add(nsp);
                }

                var mergeNspFile = env.OutputDir + "\\" + Path.GetFileName(metaFileSource) + "_merge.nsp";
                {
                    // nspをマージ
                    string standardError;
                    string standardOut;
                    ExecuteProgram(
                        out standardError,
                        out standardOut,
                        string.Format("mergeapplication -o {0} {1} --ignore-unpublishable-error",
                        mergeNspFile,
                        string.Join(" ", nspFileList.ToArray())));
                    Assert.IsTrue(File.Exists(mergeNspFile));
                    nspFileList.Add(mergeNspFile);
                }

                return nspFileList;
            };

            string issueString10812 = "Issue 10-812"; // 改変されたソフトリーガル情報のID
            string issueString10048 = "Issue 10-048"; // https接続でないアクセス可能URL
            string issueString12002 = "Issue 12-002"; // ソフトリーガル情報の内容確認(不要なソフトリーガル情報の表記)
                                                      // "legal_info.zip" では発生せずに、"legal_info_invalid_hash.zip" で発生する
            string ProgramIndexMarker = "ProgramIndex";

            // 1つ目のソフトリーガル情報のハッシュが正しく、2つ目のソフトリーガル情報のハッシュが不正
            // すべてのAccessibleUrlが正しい
            {
                string[] legalInfoFiles =
                {
                    "legal_info.zip",
                    "legal_info_invalid_hash.zip"
                };
                string[] accessibleUrls =
                {
                    okAccessibleUrl,
                    okAccessibleUrl
                };
                var nspFileList = mergeNspAndGetUnpublishableError(legalInfoFiles, accessibleUrls);

                Tuple<string, string> unpublishableErrorResult;

                // マージ前の1つ目のnspのメッセージをチェック
                unpublishableErrorResult = getUnpublishableErrorMessage(nspFileList[0]);
                // 10-812 と 12-002 が見つからないことを確認
                Assert.IsTrue((new Regex(issueString10812)).Matches(unpublishableErrorResult.Item1).Count == 0);
                Assert.IsTrue((new Regex(issueString12002)).Matches(unpublishableErrorResult.Item1).Count == 0);

                // マージ前の2つ目のnspのメッセージをチェック
                unpublishableErrorResult = getUnpublishableErrorMessage(nspFileList[1]);
                // 10-812 と 12-002 が1つ見つかることを確認
                Assert.IsTrue((new Regex(issueString10812)).Matches(unpublishableErrorResult.Item1).Count == 1);
                Assert.IsTrue((new Regex(issueString12002)).Matches(unpublishableErrorResult.Item1).Count == 1);

                // マージ後のnspのメッセージをチェック
                unpublishableErrorResult = getUnpublishableErrorMessage(nspFileList.Last());
                // 10-812 と 12-002 が1つもないことを確認
                // (マージ時に idOffset = 0 のものだけになるのでエラー無し)
                Assert.IsTrue((new Regex(issueString10812)).Matches(unpublishableErrorResult.Item1).Count == 0);
                Assert.IsTrue((new Regex(issueString12002)).Matches(unpublishableErrorResult.Item1).Count == 0);
                // 10-048 が1つもないことを確認
                Assert.IsTrue((new Regex(issueString10048)).Matches(unpublishableErrorResult.Item1).Count == 0);
            }

            // 1つ目のソフトリーガル情報のハッシュが不正
            // 2つ目のAccessibleUrlが不正
            {
                string[] legalInfoFiles =
                {
                    "legal_info_invalid_hash.zip",
                    "legal_info.zip",
                };
                string[] accessibleUrls =
                {
                    okAccessibleUrl,
                    ngAccessibleUrl
                };

                var nspFileList = mergeNspAndGetUnpublishableError(legalInfoFiles, accessibleUrls);

                Tuple<string, string> unpublishableErrorResult;

                // マージ前の1つ目のnspのメッセージをチェック
                unpublishableErrorResult = getUnpublishableErrorMessage(nspFileList[0]);
                // ProgramIndexが0のみのコンテンツでは "ProgramIndex" が表示されないことを確認
                Assert.IsTrue((new Regex(ProgramIndexMarker)).Matches(unpublishableErrorResult.Item1).Count == 0);
                Assert.IsTrue((new Regex(ProgramIndexMarker)).Matches(unpublishableErrorResult.Item2).Count == 0);
                // 10-812 と 12-002 が1つ見つかることを確認
                Assert.IsTrue((new Regex(issueString12002)).Matches(unpublishableErrorResult.Item1).Count == 1);
                Assert.IsTrue((new Regex(issueString10812)).Matches(unpublishableErrorResult.Item1).Count == 1);
                // 10-048 が見つからないことを確認
                Assert.IsTrue((new Regex(issueString10048)).Matches(unpublishableErrorResult.Item1).Count == 0);

                // マージ前の2つ目のnspのメッセージをチェック
                unpublishableErrorResult = getUnpublishableErrorMessage(nspFileList[1]);
                // ProgramIndexが1のみのコンテンツでは "ProgramIndex" が表示されることを確認
                Assert.IsTrue((new Regex(ProgramIndexMarker)).Matches(unpublishableErrorResult.Item1).Count == 1);
                Assert.IsTrue((new Regex("<" + ProgramIndexMarker + ">")).Matches(unpublishableErrorResult.Item2).Count == 1);
                // 10-812 と 12-002 が見つからないことを確認
                Assert.IsTrue((new Regex(issueString10812)).Matches(unpublishableErrorResult.Item1).Count == 0);
                Assert.IsTrue((new Regex(issueString12002)).Matches(unpublishableErrorResult.Item1).Count == 0);
                // 10-048 が1つ見つかることを確認
                Assert.IsTrue((new Regex(issueString10048)).Matches(unpublishableErrorResult.Item1).Count == 1);

                // マージ後のnspのメッセージをチェック
                unpublishableErrorResult = getUnpublishableErrorMessage(nspFileList.Last());
                // 10-812 と 12-002 が2つ見つかることを確認
                // (リーガル情報はマージ時に idOffset = 0 のものだけになり、 UnpublishableError のチェックは全てのコンテンツのチェックを先頭のリーガル情報で実施するのでエラーは 2 つ)
                Assert.IsTrue((new Regex(issueString10812)).Matches(unpublishableErrorResult.Item1).Count == 2);
                Assert.IsTrue((new Regex(issueString12002)).Matches(unpublishableErrorResult.Item1).Count == 2);
                // 10-048 が1つ見つかることを確認
                Assert.IsTrue((new Regex(issueString10048)).Matches(unpublishableErrorResult.Item1).Count == 1);

                // テキストフォーマットの確認
                // コンテンツ毎に正しいエラーが表示されることを確認
                var offsetOfProgramIndex0 = unpublishableErrorResult.Item1.IndexOf(ProgramIndexMarker + ":0");
                var offsetOfProgramIndex1 = unpublishableErrorResult.Item1.IndexOf(ProgramIndexMarker + ":1");
                var offsetOfIssueString10812 = unpublishableErrorResult.Item1.IndexOf(issueString10812);
                var offsetOfIssueString10048 = unpublishableErrorResult.Item1.IndexOf(issueString10048);
                var offsetOfIssueString12002 = unpublishableErrorResult.Item1.IndexOf(issueString12002);
                Assert.IsTrue((offsetOfProgramIndex0 > 0) && (offsetOfProgramIndex1 > 0) && (offsetOfIssueString10812 > 0) && (offsetOfIssueString10048 > 0) && (offsetOfIssueString12002 > 0));
                // 10-812 と 12-002 が1つ目のコンテンツで発生していることを確認
                Assert.IsTrue((offsetOfIssueString10812 > offsetOfProgramIndex0) && (offsetOfIssueString10812 < offsetOfProgramIndex1));
                Assert.IsTrue((offsetOfIssueString12002 > offsetOfProgramIndex0) && (offsetOfIssueString12002 < offsetOfProgramIndex1));
                // 10-048 が2つ目のコンテンツで発生していることを確認
                Assert.IsTrue((offsetOfIssueString10048 > offsetOfProgramIndex0) && (offsetOfIssueString10048 > offsetOfProgramIndex1));
                // 10-812 と 12-002 が2つ目のコンテンツでも発生していることを確認
                offsetOfIssueString10812 = unpublishableErrorResult.Item1.IndexOf(issueString10812, offsetOfProgramIndex1);
                Assert.IsTrue((offsetOfIssueString10812 > offsetOfProgramIndex0) && (offsetOfIssueString10812 > offsetOfProgramIndex1));
                offsetOfIssueString12002 = unpublishableErrorResult.Item1.IndexOf(issueString12002, offsetOfProgramIndex1);
                Assert.IsTrue((offsetOfIssueString12002 > offsetOfProgramIndex0) && (offsetOfIssueString12002 > offsetOfProgramIndex1));

                // XMLフォーマットの確認
                var reader = XmlReader.Create(new StringReader(unpublishableErrorResult.Item2));
                var document = new XmlDocument();
                document.Load(reader);
                var nodes = document.SelectNodes("UnpublishableError//Content");
                Assert.IsTrue(nodes.Count == 2);

                // Xml内の指定したタグの内容が一致する要素数を取得
                Func<XmlNode, string, string, int> getXmlNodeCount = (XmlNode node, string tag, string searchText) =>
                {
                    int elementCount = 0;
                    foreach (XmlNode id in node.SelectNodes(tag))
                    {
                        if (id.InnerText == searchText)
                        {
                            elementCount++;
                        }
                    }
                    return elementCount;
                };

                // 先頭のコンテンツの ProgramIndex が0で、 10-812,12-002 が発生して 10-048 が発生していないことの確認
                var node1 = nodes[0];
                Assert.IsTrue(node1.SelectSingleNode("ProgramIndex").InnerText == "0");
                Assert.IsTrue(getXmlNodeCount(node1, "Error//Message//Id", issueString10812) == 1);
                Assert.IsTrue(getXmlNodeCount(node1, "Error//Message//Id", issueString10048) == 0);
                Assert.IsTrue(getXmlNodeCount(node1, "Error//Message//Id", issueString12002) == 1);

                // 2つ目のコンテンツの ProgramIndex が1で、 10-048 と 10-812 と 12-002 が発生していることの確認
                var node2 = nodes[1];
                Assert.IsTrue(node2.SelectSingleNode("ProgramIndex").InnerText == "1");
                Assert.IsTrue(getXmlNodeCount(node2, "Error//Message//Id", issueString10812) == 1);
                Assert.IsTrue(getXmlNodeCount(node2, "Error//Message//Id", issueString10048) == 1);
                Assert.IsTrue(getXmlNodeCount(node2, "Error//Message//Id", issueString12002) == 1);
            }

            SafeDeleteDirectory(env.OutputDir);
        }

        [TestMethod]
        public void TestExecutionMultiApplicationCardErrorUnpublishable()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var metaFileSource = env.SourceDir + "\\UppTest\\describe_all_required_system_version.nmeta";

            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            var appIds = new ulong[]
            {
                0x01004b9000490000UL,
                0x01004b90004A0000UL,
                0x01004b90004B0000UL,
                0x01004b90004C0000UL,
                0x01004b90004D0000UL,
            };

            Func<ulong, string, string> getNspPath = (appId, suffix) =>
            {
                return env.OutputDir + "\\0x" + appId.ToString("x16") + suffix + ".nsp";
            };

            foreach (var appId in appIds)
            {
                var metaFileV0 = env.OutputDir + "\\" + Path.GetFileName(metaFileSource).Replace(".nmeta", "_v0.nmeta");
                var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFileSource).Replace(".nmeta", "_v1.nmeta");
                MakeSpecifiedVersionMetaFile(metaFileSource, metaFileV0, 0, appId);
                MakeSpecifiedVersionMetaFile(metaFileSource, metaFileV1, 65536, appId);

                var v0 = getNspPath(appId, "_v0");
                var v1 = getNspPath(appId, "_v1");
                var v1patch = getNspPath(appId, "_v1_patch");

                // アプリ・パッチ作成
                CreateNsp(v0, env, metaFileV0, string.Format(" --keygeneration 2 --accessible-urls {0} --legal-information {1}", accessibleUrlsDir, legalInformationZip));
                CreateNsp(v1, env, metaFileV1, string.Format(" --keygeneration 2 --accessible-urls {0} --legal-information {1}", accessibleUrlsDir, legalInformationZip));
                CreatePatch(v1patch, env, v0, v1, string.Empty);
            }

            Func<string, string, Tuple<string, string>> bundleup = (nsp, args) =>
            {
                string standardError;
                string standardOut;
                ExecuteProgram(
                    out standardError,
                    out standardOut,
                    string.Format("bundleup -o {0} {1}", nsp, args));
                return Tuple.Create(standardError, standardOut);
            };

            var bundleNsp = env.OutputDir + "\\bundle.nsp";
            var id = "0x02004b9000490000";
            var issueString10050 = "[Warning] (Issue 10-050)"; // マルチアプリケーションカードの確認（規定数以上のアプリケーション数）

            // アプリケーションとパッチを 5 つずつまとめる
            var fullArg = string.Join(" ", appIds.SelectMany(x => new string[] { getNspPath(x, "_v0"), getNspPath(x, "_v1_patch") }));
            // --error-unpublishable なし
            {
                var result = bundleup(bundleNsp, $"--id {id} {fullArg} --cardsize 16 --cardlaunchflags 1");
                Assert.IsTrue(result.Item1 == string.Empty, result.Item1);
                Assert.IsTrue(result.Item2 == string.Empty, result.Item2); // メッセージなし
            }
            // --error-unpublishable あり
            {
                var result = bundleup(bundleNsp, $"--id {id} {fullArg} --cardsize 16 --cardlaunchflags 1 --error-unpublishable");
                Assert.IsTrue(result.Item1 == string.Empty, result.Item1);
                Assert.IsTrue(result.Item2.Contains(issueString10050)); // 警告が表示される
            }

            // アプリケーションとパッチを 4 つずつまとめる
            fullArg = string.Join(" ", appIds.Take(4).SelectMany(x => new string[] { getNspPath(x, "_v0"), getNspPath(x, "_v1_patch") }));
            // --error-unpublishable あり
            {
                var result = bundleup(bundleNsp, $"--id {id} {fullArg} --cardsize 16 --cardlaunchflags 1 --error-unpublishable");
                Assert.IsTrue(result.Item1 == string.Empty, result.Item1);
                Assert.IsTrue(result.Item2.Contains("UnpublishableError:")); // UnpublishableError のメッセージが表示される
                Assert.IsFalse(result.Item2.Contains(issueString10050)); // 警告は表示されない
            }

            SafeDeleteDirectory(env.OutputDir);
        }
    }
}
