﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace AuthoringToolsTest
{
    [TestClass]
    public class CryptoTest
    {
        private void InitializeCounter(byte[] ctr, ulong value)
        {
            // big endian
            int i, j;
            if (ctr.Length < 8)
            {
                throw new ArgumentException();
            }
            for (i = ctr.Length - 1, j = 0; j < 8; --i, j++)
            {
                ctr[i] = (byte)((value >> j * 8) & 0xFF);
            }
        }
        private void UpdateCounter(byte[] ctr)
        {
            for (int i = ctr.Length - 1; i >= 0; --i)
            {
                if (++ctr[i] != 0)
                {
                    break;
                }
            }
        }

        [TestMethod]
        public void RsaEncryptDecryptTest()
        {
            {
                RsaCryptoDriver cryptor = new RsaCryptoDriver(RsaCryptoDriver.CryptoMode.RSA2048, RsaCryptoDriver.EncryptionKeyData);
                byte[] src = new byte[128];
                byte[] encrypted = cryptor.EncryptBlock(src, 0, src.Length);
                byte[] decrypted = cryptor.DecryptBlock(encrypted, 0, encrypted.Length);
                for (int i = 0; i < src.Length; i++)
                {
                    Assert.AreEqual(src[i], decrypted[i]);
                }
            }
            {
                RsaCryptoDriver cryptor = new RsaCryptoDriver(RsaCryptoDriver.CryptoMode.RSA2048, RsaCryptoDriver.SignKeyData);
                byte[] src = new byte[1024];

                SHA256CryptoServiceProvider sha256 = new SHA256CryptoServiceProvider();
                byte[] signed = cryptor.SignBlock(src, 0, src.Length);
                Assert.IsTrue(cryptor.VerifyBlock(src, signed));
            }
        }

        [TestMethod]
        public void Rsa2048OaepSha256CryptoDriverTest()
        {
            var encryptor = new Rsa2048OaepSha256CryptoDriver(Rsa2048OaepSha256KeyIndex.XcieKeyArea);
            byte[] src = new byte[32];
            byte[] encrypted = new byte[encryptor.KeySize];
            encryptor.EncryptBlock(src, 0, src.Length, encrypted, 0);
            Assert.IsTrue(true);
        }

        [TestMethod]
        public void Aes128CbcCryptoDriverTest()
        {
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();
            var encryptor = new Aes128CbcCryptoDriver(Aes128KeyIndex.XciHeader);
            const int LoopNum = 64;
            byte[] src = new byte[112];
            byte[] encrypted = new byte[src.Length];
            byte[] decrypted = new byte[src.Length];
            byte[] iv = new byte[encryptor.KeySize];
            for (int i = 0; i < LoopNum; i++)
            {
                rng.GetBytes(src);
                rng.GetBytes(iv);
                encryptor.EncryptBlock(iv, src, 0, src.Length, encrypted, 0);
                encryptor.DecryptBlock(iv, encrypted, 0, encrypted.Length, decrypted, 0);
                Assert.IsTrue(src.SequenceEqual(decrypted));
            }
        }

        [TestMethod]
        public void Aes128CcmCryptoDriverTest()
        {
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();
            const int LoopNum = 64;
            byte[] key = new byte[16];
            byte[] src = new byte[16];
            byte[] encrypted = new byte[src.Length];
            byte[] decrypted = new byte[src.Length];

            byte[] aad = new byte[0];
            byte[] nonce = new byte[12];

            byte[] mac = new byte[src.Length];
            byte[] macVerify = new byte[src.Length];
            for (int i = 0; i < LoopNum; i++)
            {
                rng.GetBytes(key);
                rng.GetBytes(src);
                rng.GetBytes(nonce);
                var encryptor = new Aes128CcmCryptoDriver(key);
                encryptor.EncryptBlock(nonce, aad, src, 0, src.Length, mac, 0, mac.Length, encrypted, 0);
                encryptor.DecryptBlock(nonce, aad, encrypted, 0, encrypted.Length, macVerify, 0, macVerify.Length, decrypted, 0);
                Assert.IsTrue(src.SequenceEqual(decrypted));
                Assert.IsTrue(mac.SequenceEqual(macVerify));
            }
        }

        [TestMethod]
        public void HmacSha256CryptoDriverTest()
        {
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();
            var hashCalculator = new HmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);

            const int LoopNum = 64;
            byte[] src = new byte[8];
            byte[] hash = new byte[32];

            for (int i = 0; i < LoopNum; i++)
            {
                rng.GetBytes(src);

                hashCalculator.CalculateHash(src, 0, src.Length, hash, 0);
            }
        }

#if false // 自動テスト不可
        [TestMethod]
        public void HsmRsaCryptoDriverTest()
        {
            const int LoopNum = 4;
            var hsmCryptoDriver1 = new HsmRsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NcaHeader1);
//            var hsmCryptoDriver2 = new HsmRsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.Acid);
            var hsmCryptoDriver2 = new HsmRsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NrrCertificate);
            var hsmCryptoDriver3 = new HsmRsa2048Pkcs1Sha256SignCryptoDriver(Rsa2048Pkcs1Sha256KeyIndex.XciHeader);
            var cryptoDriver1 = new Rsa2048PssSha256SignCryptoDriver(hsmCryptoDriver1.GetKeyModulus(), hsmCryptoDriver1.GetKeyPublicExponent(), null);
            var cryptoDriver2 = new Rsa2048PssSha256SignCryptoDriver(hsmCryptoDriver2.GetKeyModulus(), hsmCryptoDriver2.GetKeyPublicExponent(), null);
            var cryptoDriver3 = new Rsa2048Pkcs1Sha256SignCryptoDriver(hsmCryptoDriver3.GetKeyModulus(), hsmCryptoDriver3.GetKeyPublicExponent(), null);
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();

            // RSA-PSS-SHA256
            var t1 = Task.Factory.StartNew( () =>
            {
                byte[] src = new byte[1024];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM で署名
                    byte[] sign = hsmCryptoDriver1.SignBlock(src, 0, src.Length);

                    // HSM で検証
                    // Assert.IsTrue(hsmCryptoDriver1.VerifyBlock(src, 0, src.Length, sign));

                    // libnn_crypto で検証
                    Assert.IsTrue(cryptoDriver1.VerifyBlock(src, 0, src.Length, sign));

                    Console.WriteLine("#1 done: " + i);
                }
            });

            var t2 = Task.Factory.StartNew( () =>
            {
                byte[] src = new byte[1024];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM で署名
                    byte[] sign = hsmCryptoDriver2.SignBlock(src, 0, src.Length);

                    // HSM で検証
                    Assert.IsTrue(hsmCryptoDriver2.VerifyBlock(src, 0, src.Length, sign));

                    // libnn_crypto で検証
                    Assert.IsTrue(cryptoDriver2.VerifyBlock(src, 0, src.Length, sign));

                    Console.WriteLine("#2 done: " + i);
                }
            });

            // RSA-PKCS-SHA256
            var t3 = Task.Factory.StartNew( () =>
            {
                byte[] src = new byte[1024];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM で署名
                    byte[] sign = hsmCryptoDriver3.SignBlock(src, 0, src.Length);

                    // HSM で検証
                    // Assert.IsTrue(hsmCryptoDriver3.VerifyBlock(src, 0, src.Length, sign));

                    // libnn_crypto で検証
                    Assert.IsTrue(cryptoDriver3.VerifyBlock(src, 0, src.Length, sign));

                    Console.WriteLine("#3 done: " + i);
                }
            });

            t1.Wait();
            t2.Wait();
            t3.Wait();
        }

        [TestMethod]
        public void HsmAesCryptoDriverTest()
        {
            const int LoopNum = 4;
            var hsmCryptoDriver1 = new HsmAes128CryptoDriver(Aes128KeyIndex.NcaContentKey);
            var hsmCryptoDriver2 = new HsmAes128CryptoDriver(Aes128KeyIndex.NcaHeader);
            // TODO: libnn_crypto で暗復号化した結果と比較
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();

            // AES-ECB
            var t1 = Task.Factory.StartNew( () =>
            {
                byte[] src = new byte[1024];
                byte[] encrypted = new byte[src.Length];
                byte[] decrypted = new byte[src.Length];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM で暗復号化
                    hsmCryptoDriver1.EncryptBlock(src, 0, src.Length, encrypted, 0);
                    hsmCryptoDriver1.DecryptBlock(encrypted, 0, encrypted.Length, decrypted, 0);
                    Assert.IsTrue(!src.SequenceEqual(encrypted));
                    Assert.IsTrue(src.SequenceEqual(decrypted));

                    Console.WriteLine("#1 done: " + i);
                }
            });

            var t2 = Task.Factory.StartNew( () =>
            {
                byte[] src = new byte[1024];
                byte[] encrypted = new byte[src.Length];
                byte[] decrypted = new byte[src.Length];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM で暗復号化
                    hsmCryptoDriver2.EncryptBlock(src, 0, src.Length, encrypted, 0);
                    hsmCryptoDriver2.DecryptBlock(encrypted, 0, encrypted.Length, decrypted, 0);
                    Assert.IsTrue(!src.SequenceEqual(encrypted));
                    Assert.IsTrue(src.SequenceEqual(decrypted));

                    Console.WriteLine("#2 done: " + i);
                }
            });

            t1.Wait();
            t2.Wait();
        }

        [TestMethod]
        public void HsmAesCbcCryptoDriverTest()
        {
            const int LoopNum = 4;
            var hsmCryptoDriver1 = new HsmAes128CbcCryptoDriver(Aes128KeyIndex.ETicketCommonKey);
            // TODO: libnn_crypto で暗復号化した結果と比較
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();

            // AES-CBC
            var t1 = Task.Factory.StartNew(() =>
            {
                byte[] src = new byte[1024];
                byte[] iv = new byte[16];
                byte[] encrypted = new byte[src.Length];
                byte[] decrypted = new byte[src.Length];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM で暗復号化
                    hsmCryptoDriver1.EncryptBlock(iv, src, 0, src.Length, encrypted, 0);
                    hsmCryptoDriver1.DecryptBlock(iv, encrypted, 0, encrypted.Length, decrypted, 0);
                    Assert.IsTrue(!src.SequenceEqual(encrypted));
                    Assert.IsTrue(src.SequenceEqual(decrypted));

                    Console.WriteLine("#1 done: " + i);
                }
            });
        }

        [TestMethod]
        public void HsmHmacSha256CryptoDriverTest()
        {
            const int LoopNum = 4;
            var hashCalculator = new HmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
            // TODO: libnn_crypto で暗復号化した結果と比較
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();

            // HMAC-SHA256
            var t1 = Task.Factory.StartNew(() =>
            {
                byte[] src = new byte[8];
                byte[] hash = new byte[32];
                for (int i = 0; i < LoopNum; ++i)
                {
                    rng.GetBytes(src);

                    // HSM でハッシュを計算
                    hashCalculator.HashBlock(src, 0, src.Length, hash, 0);

                    Console.WriteLine("#1 done: " + i);
                }
            });
        }
#endif
    }
}
