﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Xml.Linq;
using System.Xml.XPath;
using Livet.Messaging;
using Xunit;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Test
{
    public class ProjectFileHelperTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void CreateNewProject_Application()
        {
            const ContentMetaType type = ContentMetaType.Application;

            using (var app = _context.DiContainer.GetInstance<App>())
            {
                app.Project.Meta.Application.DisplayVersion = "ABC";

                ProjectFileHelper.CreateNewProject(app, type);

                Assert.Equal(string.Empty, app.Project.Meta.Application.DisplayVersion);
            }
        }

        [Fact]
        public void CreateNewProject_Aoc()
        {
            const ContentMetaType type = ContentMetaType.Application;

            using (var app = _context.DiContainer.GetInstance<App>())
            {
                app.Project.AocMeta.Contents.Add(new AocContent(_context.DiContainer, app.Project.AocMeta));

                ProjectFileHelper.CreateNewProject(app, type);

                Assert.NotNull(app.Project.AocMeta.Contents);
                Assert.Empty(app.Project.AocMeta.Contents);
            }
        }

        [Theory]
        [InlineData("InvalidValue")]
        [InlineData("Allow")]
        [InlineData("Deny")]
        public void CheckInvalidVideoCaptureValue(string value)
        {
            using (var d = new DisposableDirectory())
            using (var app = _context.DiContainer.GetInstance<App>())
            {
                if (app.Project.AppCapability.IsSupportVideoCapture == false)
                    return;

                app.Project.Meta.Application.VideoCapture = VideoCaptureType.Enable;

                var xml = app.Project.MakeAppMetaXmlForAuthoringTool();
                var doc = XDocument.Parse(xml);

                {
                    var videoCapture = doc.Root.XPathSelectElement("Application/VideoCapture");
                    videoCapture.Value = value;
                }

                var xmlPath = d.CreateFile("app.nmeta", doc.ToString());

                ProjectFileHelper.Open(
                    _context.DiContainer,
                    new InteractionMessenger(),
                    ImportableFileType.Meta,
                    _context.DiContainer.GetInstance<Config>(),
                    xmlPath,
                    null,
                    (f, original) =>
                    {
                        Assert.Equal(xmlPath, f);
                        Assert.Throws<InvalidApplicationMetaItemException>(() =>
                        {
                            app.Open(ImportableFileType.Meta, f);
                        });
                    },
                    null);

            }

        }
    }
}
