﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.ProjectEditPanel.Pages
{
    public class AocContentVmTest
    {
        private readonly TestContext _context = new TestContext();

        [Fact]
        public void DefaultCtor()
        {
            var project = GetProject();
            project.AocMeta.AddContent();
            var content = project.AocMeta.Contents.First();
            using (var vm = new AocContentVm(_context.DiContainer, content, project))
            {
                Assert.True(vm.HasErrors.Value);
            }
        }

        [Theory]
        [InlineData(2001, true)]
        [InlineData(0, true)]
        [InlineData(Constants.AocIndexMinimum, false)]
        [InlineData(Constants.AocIndexMaximum, false)]
        public void IndexRangeCheck(ulong index, bool hasErrors)
        {
            var project = GetProject();
            project.AocMeta.AddContent();
            var content = project.AocMeta.Contents.First();
            using (var vm = new AocContentVm(_context.DiContainer, content, project))
            {
                content.DataPath.Path = _context.TempDirPath;
                content.Tag = "Test";
                Assert.False(vm.HasErrors.Value);

                content.Index = index;
                Assert.Equal(hasErrors, vm.HasErrors.Value);
            }
        }

        [Fact]
        public void RelativePath()
        {
            var project = GetProject();
            project.AocMeta.AddContent();
            var content = project.AocMeta.Contents.First();
            using (var vm = new AocContentVm(_context.DiContainer, content, project))
            {
                _context.CreateFolder("aoc");
                _context.CreateFile(@"aoc\empty.txt");
                content.DataPath.Path = Path.Combine(_context.TempDirPath, "aoc");
                content.Tag = "Test";
                Assert.False(vm.HasErrors.Value);

                _context.DiContainer.GetInstance<App>().Save(
                    ExportableFileType.AocMeta,
                    Path.Combine(_context.TempDirPath, "aoc.nmeta"));
                Assert.False(vm.HasErrors.Value);

            }
        }

        [Fact]
        public void TagLength()
        {
            var project = GetProject();
            project.AocMeta.AddContent();
            var content = project.AocMeta.Contents.First();
            using (var vm = new AocContentVm(_context.DiContainer, content, project))
            {
                Assert.True(vm.HasErrors.Value);
                content.Tag = "a";
                Assert.False(vm.HasErrors.Value);
                content.Tag = string.Join(string.Empty, Enumerable.Repeat('a', AocContent.MaxTagLength + 1));
                Assert.True(vm.HasErrors.Value);
            }
        }

        private Project GetProject()
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.AocMeta = new AocMeta
            {
                DiContainer = _context.DiContainer,
                Contents = new ObservableCollection<AocContent>()
            };
            return project;
        }
    }
}
