﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Helper.Test
{
    public class NspReaderTest : IClassFixture<BuildTestAppAndPatchNspFixture>, IDisposable
    {
        private readonly BuildTestAppAndPatchNspFixture _fixture;
        private readonly DisposableDirectory _tempDir = new DisposableDirectory();

        // パッチも含めた複数 nsp の生成には十数秒の時間が必要なので、ビルド済みキャッシュがあればそれを使う
        private static string CachedTempDirPath
            => Environment.ExpandEnvironmentVariables(@"%TEMP%\Nintendo\AuthoringEditor\_" + nameof(NspReaderTest));

        public NspReaderTest(BuildTestAppAndPatchNspFixture fixture)
        {
            _fixture = fixture;
            if (Directory.Exists(CachedTempDirPath))
            {
                _fixture.TempDirPath = CachedTempDirPath;
            }
            else
            {
                _fixture.Initialize();
                _fixture.InitializePatch();
            }
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var reader = new NspReader(_fixture.NspFilePath))
            {

            }
        }

        [Fact]
        public void ReadContentMeta()
        {
            using (var source = new NspReader(_fixture.NspFilePath))
            {
                var contentMetaXmlText = source.ExtractAllTextFromRoot(s => s.EndsWith(".cnmt.xml"));
                Assert.NotEmpty(contentMetaXmlText);
            }
        }

        [Fact]
        public void IsIdenticalNsp()
        {
            using (var original = new NspReader(_fixture.NspFilePath).GetNcaReader(ContentType.Program))
            using (var revisedV1 = new NspReader(_fixture.RevisedV1NspFilePath).GetNcaReader(ContentType.Program))
            using (var revisedV2 = new NspReader(_fixture.RevisedV2NspFilePath).GetNcaReader(ContentType.Program))
            using (var revisedV3 = new NspReader(_fixture.RevisedV3NspFilePath).GetNcaReader(ContentType.Program))
            {
                // V1 は ndpm を変更しているのでプログラムに変更したもの
                Assert.False(original.Equals(revisedV1));
                // V2 は V1 からプログラムコード自体を変更したもの
                Assert.False(revisedV1.Equals(revisedV2));
                // V3 は V2 からアプリケーション管理プロパティのみを変更したもの
                Assert.True(revisedV2.Equals(revisedV3));
            }
        }

        [Fact]
        public void IsIdenticalPatchedNsp()
        {
            using (var original = new NspReader(_fixture.NspFilePath).GetNcaReader(ContentType.Program))
            using (var patchV1 = new NspReader(_fixture.PatchV1NspFilePath, _fixture.NspFilePath).GetNcaReader(ContentType.Program))
            using (var patchV2 = new NspReader(_fixture.PatchV2NspFilePath, _fixture.NspFilePath).GetNcaReader(ContentType.Program))
            using (var patchV3 = new NspReader(_fixture.PatchV3NspFilePath, _fixture.NspFilePath).GetNcaReader(ContentType.Program))
            using (var revisedV2 = new NspReader(_fixture.RevisedV2NspFilePath).GetNcaReader(ContentType.Program))
            {
                // V1, V2 は Original からプログラムに変更あり
                Assert.False(original.Equals(patchV1));
                Assert.False(original.Equals(patchV2));
                // パッチ間のプログラム差分 (V1 => V2 は変更あり)
                Assert.False(patchV1.Equals(patchV2));
                // パッチ V2 => V3 はプログラムに変更なし
                Assert.True(patchV2.Equals(patchV3));
                // パッチ V3 と V2 アプリは、プログラムの内容に変更はなし
                Assert.True(revisedV2.Equals(patchV3));
            }
        }

        [Fact]
        public void IsIdenticalNspPartial()
        {
            using (var original = new NspReader(_fixture.NspFilePath).GetNcaReader(ContentType.Program))
            using (var revisedV1 = new NspReader(_fixture.RevisedV1NspFilePath).GetNcaReader(ContentType.Program))
            {
                // v1 は main.npdm を変更しているものの、プログラム本体には変更なし
                Assert.True(original.Compare(revisedV1, x => x == "fs0/main").HasFlag(NcaCompareResult.Identical));
                Assert.False(original.Compare(revisedV1, x => x == "fs0/main.npdm").HasFlag(NcaCompareResult.Identical));
            }
        }

        [Fact]
        public void IsContainsFileNca()
        {
            using (var html = new NspReader(_fixture.HtmlContainsNspFilePath).GetNcaReader(ContentType.HtmlDocument))
            using (var invalidOriginal = new NspReader(_fixture.NspFilePath).GetNcaReader(ContentType.HtmlDocument))
            using (var original = new NspReader(_fixture.NspFilePath).GetNcaReader(ContentType.Program))
            {
                // nca がある && ファイルも存在する
                Assert.True(html.IsContainsFile(x => x.StartsWith("fs0/html-document/")));
                // nca がある && 該当するファイルはない
                Assert.False(html.IsContainsFile(x => x.StartsWith("fs0/accessible-urls/")));
                // nca がない
                Assert.Null(invalidOriginal);
                Assert.False(original.IsContainsFile(x => x.StartsWith("fs0/accessible-urls/")));
            }
        }

        [Fact]
        public void CompareHtmlDocumentNca()
        {
            using (var html = new NspReader(_fixture.HtmlContainsNspFilePath).GetNcaReader(ContentType.HtmlDocument))
            using (var accessibleUrls = new NspReader(_fixture.AccessibleUrlContainsNspFilePath).GetNcaReader(ContentType.HtmlDocument))
            using (var htmlAndAccessibleUrls = new NspReader(_fixture.HtmlAndAccessibleUrlContainsNspFilePath).GetNcaReader(ContentType.HtmlDocument))
            {
                {
                    var r = html.Compare(accessibleUrls, x => x.StartsWith("fs0/accessible-urls/"));
                    Assert.False(r.HasFlag(NcaCompareResult.Identical));
                    Assert.False(r.HasFlag(NcaCompareResult.SourceFileExists));
                    Assert.True(r.HasFlag(NcaCompareResult.TargetNcaExists));
                    Assert.True(r.HasFlag(NcaCompareResult.TargetFileExists));
                }

                {
                    var r = html.Compare(accessibleUrls, x => x.StartsWith("fs0/html-document/"));
                    Assert.False(r.HasFlag(NcaCompareResult.Identical));
                    Assert.False(r.HasFlag(NcaCompareResult.TargetFileExists));

                    Assert.True(r.HasFlag(NcaCompareResult.SourceNcaExists));
                    Assert.True(r.HasFlag(NcaCompareResult.TargetNcaExists));
                    Assert.True(r.HasFlag(NcaCompareResult.SourceFileExists));
                }

                Assert.True(accessibleUrls.Compare(htmlAndAccessibleUrls, x => x.StartsWith("fs0/accessible-urls/")).HasFlag(NcaCompareResult.Identical));
                Assert.True(html.Compare(htmlAndAccessibleUrls, x => x.StartsWith("fs0/html-document/")).HasFlag(NcaCompareResult.Identical));
            }
        }

        [Fact]
        public void NotFoundNca()
        {
            Assert.Null(new NspReader(_fixture.NspFilePath).GetNcaReader(ContentType.HtmlDocument));
        }

        [Fact]
        public void ExtractFiles()
        {
            using (var original = new NspReader(_fixture.NspFilePath))
            {
                ExtractXmlFlatten(original);
            }

            using (var patchV3 = new NspReader(_fixture.PatchV3NspFilePath, _fixture.NspFilePath))
            {
                ExtractXmlFlatten(patchV3);
            }
        }

        [Fact]
        public void ExtractFilesFromListText()
        {
            using (var original = new NspReader(_fixture.NspFilePath))
            {
                var extractFiles = new List<string>();
                var extractedFiles = original.GetExtractFiles(x =>
                {
                    extractFiles.Add(x);
                    return null;
                }).ToArray(); // 正格評価しないとメソッドが実行されない

                Assert.Empty(extractedFiles);
                Assert.NotEmpty(extractFiles);

                var listTextPath = Path.Combine(_tempDir.RootPath, Path.GetRandomFileName());
                File.WriteAllLines(listTextPath, extractedFiles.Select(x => $"{x}\t{x}"));

                var temporaryRoot = _tempDir.CreateFolder(Path.GetRandomFileName());
                foreach (var e in original.GetExtractFilesFromListFile(listTextPath))
                {
                    var outputPath = e.WriteFile(temporaryRoot);
                    Assert.True(File.Exists(outputPath));
                }
            }
        }

        [Fact]
        public void ExtractAll()
        {
            using (var original = new NspReader(_fixture.NspFilePath))
            {
                var temporaryRoot = _tempDir.CreateFolder(Path.GetRandomFileName());
                foreach (var e in original.GetExtractFiles(x => x))
                {
                    var outputPath = e.WriteFile(temporaryRoot);
                    Assert.True(File.Exists(outputPath));
                }
            }
        }

        [Theory]
        [InlineData(ContentType.Control)]
        [InlineData(ContentType.Program)]
        public void ExtractSpecificContentTypeNca(ContentType contentType)
        {
            using (var original = new NspReader(_fixture.NspFilePath))
                ExtractSpecificContentNca(original, contentType);

            using (var patchV1 = new NspReader(_fixture.PatchV1NspFilePath, _fixture.NspFilePath))
                ExtractSpecificContentNca(patchV1, contentType);
        }

        private void ExtractSpecificContentNca(NspReader reader, ContentType contentType)
        {
            var temporaryRoot = _tempDir.CreateFolder(Path.GetRandomFileName());
            foreach (var e in reader.GetExtractFiles(x => x, contentType))
            {
                var outputPath = e.WriteFile(temporaryRoot);
                Assert.True(File.Exists(outputPath));
            }
        }

        private void ExtractXmlFlatten(NspReader reader)
        {
            foreach (var file in reader.GetExtractFiles(x => x.EndsWith(".xml") ? x.Replace('/', '_') : null))
            {
                Assert.True(file.Name.EndsWith(".xml"));
                Assert.True(file.OutputPath.EndsWith(".xml"));
                var outputPath = file.WriteFile(_tempDir.RootPath);
                Assert.True(File.Exists(outputPath));
                Assert.True(outputPath.Contains(_tempDir.RootPath));
            }
        }

        public void Dispose()
        {
            _tempDir.Dispose();
        }
    }
}
