﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class TypeHelperTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        public static IEnumerable<object> FilePathToImportableFileTypeTestData
        {
            get
            {
                yield return new object[]
                {
                    null, "ABC"
                };

                yield return new object[]
                {
                    ImportableFileType.Project,
                    Path.Combine(TestContext.TestDataDirPath, @"project_testdata\test.narp")
                };

                yield return new object[]
                {
                    ImportableFileType.Meta,
                    Path.Combine(TestContext.TestDataDirPath, @"project_testdata\exported.nmeta")
                };

                yield return new object[]
                {
                    ImportableFileType.Meta,
                    Path.Combine(TestContext.TestDataDirPath, @"project_testdata\exported.meta")
                };

                yield return new object[]
                {
                    ImportableFileType.Nsp,
                    Path.Combine(TestContext.TestDataDirPath, @"project_testdata\broken.nsp")
                };
            }
        }

        [Theory]
        [MemberData(nameof(FilePathToImportableFileTypeTestData))]
        public void FilePathToImportableFileType(ImportableFileType? actual, string filePath)
        {
            if (actual.HasValue == false)
                Assert.Equal(actual, TypeHelper.FilePathToImportableFileType(filePath));
            else
                // ReSharper disable once PossibleInvalidOperationException
                Assert.Equal(actual.Value, TypeHelper.FilePathToImportableFileType(filePath).Value);
        }

        [Fact]
        public void FilePathToImportableFileType_Unknown()
        {
            var path = Path.Combine(TestContext.RootPath, @"dummydata\unknownfile.xyz");

            Assert.Null(TypeHelper.FilePathToImportableFileType(path));
        }

        public static IEnumerable<object> DetectMetaKindTestData
        {
            get
            {
                yield return new object[]
                {
                    MetaKind.Application,
                    Path.Combine(TestContext.TestDataDirPath, @"metafile_test\Application.nmeta")
                };

                yield return new object[]
                {
                    MetaKind.AddOnContent,
                    Path.Combine(TestContext.TestDataDirPath, @"metafile_test\AddOnContent.nmeta")
                };

                yield return new object[]
                {
                    MetaKind.Unknown,
                    null
                };

                yield return new object[]
                {
                    MetaKind.Unknown,
                    string.Empty
                };

                yield return new object[]
                {
                    MetaKind.Unknown,
                    Path.Combine(TestContext.TestDataDirPath, @"metafile_test\broken.nmeta")
                };

                yield return new object[]
                {
                    MetaKind.Unknown,
                    "AAA"
                };
            }
        }

        [Theory]
        [MemberData(nameof(DetectMetaKindTestData))]
        public void DetectMetaKind(MetaKind kind, string filePath)
        {
            Assert.Equal(kind, TypeHelper.DetectMetaKind(filePath));
        }

        public static IEnumerable<object> MakeMetaAppModeTypeTestData
        {
            get
            {
                yield return new object[]
                {
                    AppModeType.ApplicationMeta,
                    Path.Combine(TestContext.TestDataDirPath, @"metafile_test\Application.nmeta")
                };

                yield return new object[]
                {
                    AppModeType.AocMeta,
                    Path.Combine(TestContext.TestDataDirPath, @"metafile_test\AddOnContent.nmeta")
                };
            }
        }

        [Theory]
        [MemberData(nameof(MakeMetaAppModeTypeTestData))]
        public void MakeMetaAppModeType(AppModeType type, string filePath)
        {
            Assert.Equal(type, TypeHelper.MakeMetaAppModeType(filePath));
        }

        [Theory]
        [InlineData(ImportableFileType.Nsp, OutputTypes.Nsp)]
        public void MakeMetaAppModeType(ImportableFileType dst, OutputTypes src)
        {
            Assert.Equal(dst, TypeHelper.ToImportable(src));
        }

        [Theory]
        [InlineData("abc.narp")]
        [InlineData("abc.nrp")]
        public void ToAppMode_Project(string path)
        {
            Assert.Equal(AppModeType.Project, TypeHelper.ToAppMode(path));
        }

        [Theory]
        [MemberData(nameof(MakeMetaAppModeTypeTestData))]
        public void ToAppMode_Meta(AppModeType type, string path)
        {
            Assert.Equal(type, TypeHelper.ToAppMode(path));
        }
    }
}
