﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using SimpleInjector;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures
{
    public class BuildTestAppNspFixture
    {
        public string TempDirPath
        {
            get { return AppBuilder.TempDirPath; }
            set { AppBuilder.TempDirPath = value; }
        }

        public string CodeDirPath => AppBuilder.CodeDirPath;

        public string DataDirPath => AppBuilder.DataDirPath;

        public string NspFilePath => AppBuilder.NspFilePath;

        public string RevisedV1NspFilePath => Path.Combine(
            Path.GetDirectoryName(NspFilePath),
            $"{Path.GetFileNameWithoutExtension(NspFilePath)}.v1.nsp");

        private static readonly SemaphoreSlim _semaphore = new SemaphoreSlim(1, 1);
        protected static readonly Container _diContainer = new Container();

        public static readonly TestAppBuilder AppBuilder = new TestAppBuilder();

        public BuildTestAppNspFixture Initialize()
        {
            try
            {
                _semaphore.Wait();
                if (AppBuilder.IsTestAppBuilt)
                    return this;
                AppBuilder.CloneTestApp();
                AppBuilder.BuildTestApp();

                Task.Run(async () =>
                {
                    var project = Project.Import(_diContainer,
                        ImportableFileType.Meta,
                        TestContext.DefaultMetaFilePath);
                    var meta = project.Meta;

                    // 初版アプリケーション
                    {
                        meta.Application.ReleaseVersion = 0;

                        var output = await AppBuilder.BuildNspAsync(NspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }

                    // 修正版アプリケーション
                    {
                        meta.Application.ReleaseVersion = 1;
                        // Core の変更を反映するには MakeMeta で .npdm を変更する必要あり
                        meta.Core.FsAccessControlData.SaveDataOwnerIds.Add(new SaveDataOwnerId
                        {
                            Accessibility = AccessibilityType.Read,
                            ApplicationId = meta.Core.ApplicationId + 1ul
                        });

                        var output = await AppBuilder.BuildNspAsync(RevisedV1NspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }
                }).Wait();
                return this;
            }
            finally
            {
                _semaphore.Release();
            }
        }
    }
}
