﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Serialization;
using BezelEditor.Foundation.Collections;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class ContentMetaTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        private readonly BuildTestAppNspFixture _fixture;

        public ContentMetaTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        [Fact]
        public void ImportableApplicationContentMeta()
        {
            var doc = XDocument.Parse(File.ReadAllText(NintendoSdkHelper.ApplicationMetaFilePath));
            var applicationId = doc.Descendants("ApplicationId").FirstOrDefault()?.Value.ToUlong();

            var nspFile = new NspFile(_fixture.NspFilePath)
            {
                EnumerationMode = NspFileEnumeration.RootOnly
            };
            var cnmtXmlFile = nspFile.Files(new NspFileEnumerateParameter{EnumerationType = NspFileEnumeration.RootOnly}).FirstOrDefault(x => x.FilePath.EndsWith(".cnmt.xml"));
            Assert.NotNull(cnmtXmlFile);

            var cnmtXml = nspFile.ReadAllText(cnmtXmlFile.FilePath);
            Assert.NotEmpty(cnmtXml);

            var cnmt = (ApplicationContentMeta) new XmlSerializer(typeof(ApplicationContentMeta))
                .Deserialize(new StringReader(cnmtXml));
            Assert.NotNull(cnmt);

            Assert.Equal(ContentMetaType.Application, cnmt.Type);
            Assert.Equal(applicationId, cnmt.Id);
            Assert.NotEqual(applicationId, cnmt.PatchId);
            Assert.Equal(0u, cnmt.Version);
            Assert.Equal(0u, cnmt.RequiredDownloadSystemVersion);

            Assert.Equal(3, cnmt.Contents.Count);

            var contents = new[] {ContentType.Program, ContentType.Control, ContentType.Meta};
            Assert.Equal(contents, cnmt.Contents.Select(x => x.Type));
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var c = new ContentMeta())
            {
                Assert.Equal(ContentMetaType.Application, c.Type);
                Assert.Equal(0ul, c.Id);
                Assert.Equal("0x0000000000000000", c.IdHex);
                Assert.Equal(0ul, c.Version);
                Assert.Null(c.Digest);
                Assert.NotNull(c.Contents);
                Assert.Equal(0, c.Contents.Count);
            }
        }

        [Fact]
        public void GetContentId()
        {
            using (var c = new ContentMeta())
            {
                Assert.Null(c.GetContentId(ContentType.Program));

                c.Contents.Add(new Content { Type = ContentType.Program, Id = "A" });
                c.Contents.Add(new Content { Type = ContentType.Data, Id = "B" });
                c.Contents.Add(new Content { Type = ContentType.Control, Id = "C" });

                Assert.Equal("A", c.GetContentId(ContentType.Program));
                Assert.Equal("B", c.GetContentId(ContentType.Data));
                Assert.Equal("C", c.GetContentId(ContentType.Control));
                Assert.Null(c.GetContentId(ContentType.LegalInformation));
            }
        }
    }

    public class PatchContentMetaTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var c = new PatchContentMeta())
            {
               Assert.Equal(0ul, c.RequiredSystemVersion);
               Assert.Equal(0ul, c.ApplicationId);
               Assert.Equal("0x0000000000000000", c.ApplicationIdHex);
               Assert.NotNull(c.History);
               Assert.Equal(0, c.History.Count);
               Assert.Equal(0, c.ReleaseVersion);
            }
        }

        [Fact]
        public void Basic()
        {
            using (var c = new PatchContentMeta
            {
                RequiredSystemVersion = 1,
                ApplicationId = 2,
                History = new DisposableObservableCollection<ContentMeta> { new PatchContentMeta()},
                Version = 3<<16
            })
            {
                Assert.Equal(1ul, c.RequiredSystemVersion);
                Assert.Equal(2ul, c.ApplicationId);
                Assert.Equal("0x0000000000000002", c.ApplicationIdHex);
                Assert.NotNull(c.History);
                Assert.Equal(1, c.History.Count);
                Assert.Equal(3, c.ReleaseVersion);
            }
        }
    }
}
