﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class CardSpecTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var c = new CardSpec())
            {
                Assert.True(c.IsAutomaticSettingSize);
                Assert.Equal(2, c.Size);

                Assert.True(c.IsAutomaticSettingClockRate);
                Assert.Equal(25, c.ClockRate);
            }
        }

        public static IEnumerable<object> ValidationSizeRangeTestData => new[]
        {
            new object[] {CardSpec.SizeValidationType.SizeError, -2},
            new object[] {CardSpec.SizeValidationType.SizeError, 0},
            new object[] {CardSpec.SizeValidationType.SizeError, 10},
            new object[] {CardSpec.SizeValidationType.SizeError, 100},
            new object[] {CardSpec.SizeValidationType.UnsupportedSizeError, 1},
            new object[] {CardSpec.SizeValidationType.Ok, 2},
            new object[] {CardSpec.SizeValidationType.Ok, 4},
            new object[] {CardSpec.SizeValidationType.Ok, 8},
            new object[] {CardSpec.SizeValidationType.Ok, 16},
            new object[] {CardSpec.SizeValidationType.Ok, 32}
        };

        [Theory]
        [MemberData(nameof(ValidationSizeRangeTestData))]
        public void ValidationSizeRange_DisableAutomaticSetting(CardSpec.SizeValidationType result, int size)
        {
            using (var c = new CardSpec())
            {
                c.IsAutomaticSettingSize = false;

                c.Size = size;

                Assert.Equal(result, c.ValidationSize);
            }
        }


        [Theory]
        [MemberData(nameof(ValidationSizeRangeTestData))]
        public void ValidationSizeRange_EnableAutomaticSetting(CardSpec.SizeValidationType result, int size)
        {
            using (var c = new CardSpec())
            {
                c.IsAutomaticSettingSize = true;

                c.Size = size;

                Assert.Equal(CardSpec.SizeValidationType.Ok, c.ValidationSize);
            }
        }

        public static IEnumerable<object> ValidationClockRateTestData => new[]
        {
            new object[] {CardSpec.ClockRateValidationType.ClockRateError, 10, 1},
            new object[] {CardSpec.ClockRateValidationType.ClockRateError, 10, 2},
            new object[] {CardSpec.ClockRateValidationType.ClockRateError, 10, 4},
            new object[] {CardSpec.ClockRateValidationType.ClockRateError, 10, 8},
            new object[] {CardSpec.ClockRateValidationType.ClockRateError, 10, 16},
            new object[] {CardSpec.ClockRateValidationType.ClockRateError, 10, 32},
            new object[] {CardSpec.ClockRateValidationType.Ok, 25, 1},
            new object[] {CardSpec.ClockRateValidationType.Ok, 25, 2},
            new object[] {CardSpec.ClockRateValidationType.Ok, 25, 4},
            new object[] {CardSpec.ClockRateValidationType.Ok, 25, 8},
            new object[] {CardSpec.ClockRateValidationType.Ok, 25, 16},
            new object[] {CardSpec.ClockRateValidationType.Ok, 25, 32},
            new object[] {CardSpec.ClockRateValidationType.Under8GB, 50, 1},
            new object[] {CardSpec.ClockRateValidationType.Under8GB, 50, 2},
            new object[] {CardSpec.ClockRateValidationType.Under8GB, 50, 4},
            new object[] {CardSpec.ClockRateValidationType.Ok, 50, 8},
            new object[] {CardSpec.ClockRateValidationType.Ok, 50, 16},
            new object[] {CardSpec.ClockRateValidationType.Ok, 50, 32},
        };

        [Theory]
        [MemberData(nameof(ValidationClockRateTestData))]
        public void ValidationClockRate_DisableAutomaticSetting(CardSpec.ClockRateValidationType result, int rate, int size)
        {
            using (var c = new CardSpec())
            {
                c.IsAutomaticSettingClockRate = false;
                c.Size = size;
                c.ClockRate = rate;

                Assert.Equal(result, c.ValidationClockRate);
            }
        }

        [Theory]
        [MemberData(nameof(ValidationClockRateTestData))]
        public void ValidationClockRate_EnableAutomaticSetting(CardSpec.ClockRateValidationType result, int rate, int size)
        {
            using (var c = new CardSpec())
            {
                c.IsAutomaticSettingClockRate = true;
                c.Size = size;
                c.ClockRate = rate;

                Assert.Equal(CardSpec.ClockRateValidationType.Ok, c.ValidationClockRate);
            }
        }
    }
}
