﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class AppModeTypeExtensionTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        public static IEnumerable<object[]> AllAppModeType =>
            Enum.GetValues(typeof(AppModeType)).Cast<AppModeType>().Select(t => new object[] {t});

        [Theory]
        [MemberData(nameof(AllAppModeType))]
        public void IsAoc(AppModeType type)
        {
            var sign = type.ToString().StartsWith("Aoc");

            Assert.Equal(sign, type.IsAoc());
        }

        [Theory]
        [MemberData(nameof(AllAppModeType))]
        public void IsMeta(AppModeType type)
        {
            var sign = type.ToString().EndsWith("Meta");

            Assert.Equal(sign, type.IsMeta());
        }
    }

    public class AppProfileTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor_SpecificTitleBarAttribute()
        {
            {
                var p = new SpecificTitleBarAttribute(true);
                Assert.True(p.IsDark);
            }

            {
                var p = new SpecificTitleBarAttribute(false);
                Assert.False(p.IsDark);
            }
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var p = new AppProfile())
            {
                Assert.False(p.CanEditCardSpec);
                Assert.False(p.IsEnableCorePageGroup);
                Assert.False(p.CanImportProject);
                Assert.False(p.CanExportProject);
                Assert.False(p.CanImportMeta);
                Assert.False(p.CanSaveMeta);
                Assert.False(p.CanSaveAsMeta);
                Assert.False(p.CanImportNsp);
                Assert.False(p.CanSaveAsNsp);
                Assert.Equal(ImportableFileType.Project, p.TargetFileType);
                Assert.Equal(AppModeType.Startup, p.AppMode);
            }
        }

        [Fact]
        public void Project()
        {
            using (var p = new AppProfile { AppMode = AppModeType.Project })
            {
                Assert.True(p.CanEditCardSpec);
                Assert.True(p.IsEnableCorePageGroup);
                Assert.True(p.CanImportProject);
                Assert.True(p.CanExportProject);
                Assert.False(p.CanImportMeta);
                Assert.False(p.CanSaveMeta);
                Assert.True(p.CanSaveAsMeta);
                Assert.False(p.CanImportNsp);
                Assert.False(p.CanSaveAsNsp);
                Assert.Equal(ImportableFileType.Project, p.TargetFileType);
                Assert.Equal(AppModeType.Project, p.AppMode);
            }
        }

        [Fact]
        public void Basic()
        {
            using (var p = new AppProfile())
            {
                p.CanEditCardSpec = true;
                p.CanImportProject = true;
                p.CanExportProject = true;
                p.CanImportMeta = true;
                p.CanSaveMeta = true;
                p.CanSaveAsMeta = true;
                p.CanImportNsp = true;
                p.CanSaveAsNsp = true;
                p.TargetFileType = ImportableFileType.Meta;
                p.AppMode = AppModeType.ApplicationMeta;
            }
        }

        [Theory]
        [InlineData(ImportableFileType.Project, AppModeType.Project)]
        [InlineData(ImportableFileType.Meta, AppModeType.ApplicationMeta)]
        [InlineData(ImportableFileType.Nsp, AppModeType.ApplicationNsp)]
        public void AppMode(ImportableFileType importable, AppModeType appMode)
        {
            using (var p = new AppProfile())
            {
                p.AppMode = appMode;

                Assert.Equal(importable, p.TargetFileType);
            }
        }
    }
}
