﻿using System;
using System.Linq;
using _3dToolsTestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;
using nw.g3d.iflib;

namespace G3dLibraryTest
{
    [TestClass]
    public class ShapeTest
    {
        [TestMethod]
        public void CopyShapeWithReferences()
        {
            string modelPath = System.IO.Path.Combine(_3dToolsTestUtility.IoUtility.GetG3dDemoRootPath(), "Resources/human/human.fmdb");
            var file = IfReadUtility.ReadIntermediateFile(modelPath, null);
            file.AutoCalc = true;
            var model = file.GetRootEntity<Model>();
            var copiedShape = model.Shapes.First().DeepCopyWithReferences(model);
            copiedShape.Name = "hoge";
            foreach (var shape in model.Shapes)
            {
                Assert.IsTrue(model.Vertexes.Contains(shape.ShapeInfo.Vertex));
                foreach (var attr in shape.ShapeInfo.Vertex.VtxAttribs)
                {
                    Assert.IsNotNull(attr.Stream);
                    Assert.IsTrue(model.Streams.Contains((attr.Stream)));
                }

                Assert.AreEqual(1, model.Shapes.Count(x => x.Name == shape.Name));
            }

            foreach (var bone in model.Skeleton.EnumerateBones())
            {
                Assert.AreEqual(1, model.Skeleton.EnumerateBones().Count(x => x.Name == bone.Name));
            }

            foreach (var mat in model.Materials)
            {
                Assert.AreEqual(1, model.Materials.Count(x => x.Name == mat.Name));
            }

            string outputPath = IoUtility.GetTempFilePath("human.fmdb");
            try
            {
                IfWriteUtility.WriteIntermediateFile(file, outputPath, IoUtility.GetXsdBasePath());
            }
            finally
            {
                if (System.IO.File.Exists(outputPath))
                {
                    System.IO.File.Delete(outputPath);
                }
            }
        }

        [TestMethod]
        public void TestReferenceProperty()
        {
            Mesh mesh = new Mesh();
            Bone boneA = new Bone() { Name = "boneA" };
            Shape shapeA = new Shape() { Name = "shapeA" };
            shapeA.ShapeInfo.Bone = boneA;
            shapeA.Meshes.Add(mesh);
            Assert.AreEqual(shapeA, mesh.Parent);

            Model model = new Model();
            model.Skeleton.RootBone = boneA;
            model.Shapes.Add(shapeA);
            Assert.AreEqual(model, shapeA.Parent);

            // Bone.Shapes プロパティの更新テスト
            {
                Assert.AreEqual(1, boneA.Shapes.Count);
                Assert.IsTrue(boneA.Shapes.Contains(shapeA));
                Assert.AreEqual(boneA, shapeA.ShapeInfo.Bone);

                // ボーンを削除したら参照が消える
                model.Skeleton.RemoveBone(boneA);
                Assert.AreEqual(0, boneA.Shapes.Count);
                Assert.IsNull(shapeA.ShapeInfo.Bone);
                // ボーン名は維持される
                Assert.AreEqual(boneA.Name, shapeA.ShapeInfo.BoneName);

                // 元に戻す
                model.Skeleton.RootBone = boneA;
                Assert.AreEqual(1, boneA.Shapes.Count);
                Assert.IsTrue(boneA.Shapes.Contains(shapeA));
                Assert.AreEqual(boneA, shapeA.ShapeInfo.Bone);

                // シェイプを削除したら参照が消える
                model.Shapes.Remove(shapeA);
                Assert.AreEqual(0, boneA.Shapes.Count);
                Assert.IsNull(shapeA.ShapeInfo.Bone);

                // 元に戻す
                model.Shapes.Add(shapeA);
                Assert.AreEqual(boneA, shapeA.ShapeInfo.Bone);

                // シェイプのボーンを変更したら反映される
                Bone boneB = new Bone() { Name = "boneB" };
                Shape shapeB = new Shape() { Name = "shapeB" };
                model.Shapes.Add(shapeB);
                boneA.ChildBones.Add(boneB);
                shapeA.ShapeInfo.Bone = boneB;
                shapeB.ShapeInfo.Bone = boneB;
                Assert.AreEqual(0, boneA.Shapes.Count);
                Assert.AreEqual(2, boneB.Shapes.Count);
                Assert.IsTrue(boneB.Shapes.Contains(shapeA));
                Assert.AreEqual(boneB, shapeA.ShapeInfo.Bone);
                Assert.IsTrue(boneB.Shapes.Contains(shapeB));
                Assert.AreEqual(boneB, shapeB.ShapeInfo.Bone);

                shapeB.ShapeInfo.Bone = boneA;
                Assert.AreEqual(1, boneA.Shapes.Count);
                Assert.AreEqual(1, boneB.Shapes.Count);
                Assert.IsTrue(boneB.Shapes.Contains(shapeA));
                Assert.AreEqual(boneB, shapeA.ShapeInfo.Bone);
                Assert.IsTrue(boneA.Shapes.Contains(shapeB));
                Assert.AreEqual(boneA, shapeB.ShapeInfo.Bone);

                // 元に戻す
                model.Shapes.Remove(shapeB);
                shapeA.ShapeInfo.Bone = boneA;
                Assert.AreEqual(1, boneA.Shapes.Count);
                Assert.IsTrue(boneA.Shapes.Contains(shapeA));
                Assert.AreEqual(boneA, shapeA.ShapeInfo.Bone);
            }

            {
                StreamInt indexStream = new StreamInt();
                mesh.IndexStream = indexStream;

                // 親モデルが持っていないストリームが設定されているときはシリアライズ可能データ作成時にエラー
                {
                    bool isErrorOccurred = false;
                    try
                    {
                        mesh.CreateSerializableData();
                    }
                    catch (Exception)
                    {
                        isErrorOccurred = true;
                    }
                    Assert.IsTrue(isErrorOccurred);
                }

                // 親がストリームを持つときは成功
                model.Streams.Add(indexStream);
                Assert.AreEqual(0, mesh.CreateSerializableData().stream_index);
            }

            {
                Vertex vertex = new Vertex();
                shapeA.ShapeInfo.Vertex = vertex;

                {
                    bool isErrorOccurred = false;
                    try
                    {
                        shapeA.CreateSerializableData();
                    }
                    catch (Exception)
                    {
                        isErrorOccurred = true;
                    }
                    Assert.IsTrue(isErrorOccurred);
                }

                model.Vertexes.Add(vertex);
                Assert.AreEqual(0, shapeA.CreateSerializableData().shape_info.vertex_index);
            }
        }
    }
}
