﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.Linq;
using System.ComponentModel;
using _3dToolsTestUtility;

namespace G3dLibraryTest
{
    [TestClass]
    public class ModelTest
    {
        [TestMethod]
        public void RemoveShapeFromModel()
        {
            string modelPath = System.IO.Path.Combine(_3dToolsTestUtility.IoUtility.GetG3dDemoRootPath(), "Resources/human/human.fmdb");
            var file = IfReadUtility.ReadIntermediateFile(modelPath, null);
            var model = file.GetRootEntity<Model>();

            model.Materials.Add(new Material() { Name = "hoge" });
            Assert.AreEqual(2, model.Materials.Count);
            model.RemoveUnreferencedMaterials();
            Assert.AreEqual(1, model.Materials.Count);

            var removeShape = model.Shapes[0];
            var bone = removeShape.ShapeInfo.Bone;
            model.RemoveShape(removeShape);
            model.RemoveMaterial(removeShape.ShapeInfo.MatName);
            model.RemoveBone(bone);

            string tempFile = IoUtility.GetTempFilePath("human.fmdb");
            try
            {
                IfWriteUtility.WriteIntermediateFile(file, tempFile, IoUtility.GetXsdBasePath());
            }
            finally
            {
                if (System.IO.File.Exists(tempFile))
                {
                    System.IO.File.Delete(tempFile);
                }
            }
        }

        [TestMethod]
        public void TestSkeletonReadWrite()
        {
            var sourceFile = new IntermediateFile(IntermediateFileKind.Model);
            var sourceRoot = sourceFile.GetRootEntity<Model>();
            var bone = new Bone();
            sourceRoot.Skeleton.RootBone = bone;
            bone.RigidSkinningMatrixIndex = 0;
            bone.Translate.X = 10.1004419f;
            bone.Translate.Y = 10.692132f;
            bone.Translate.Z = 1.72986f;
            bone.Rotate.X = -0.000114591559f;
            bone.Rotate.Y = -0.00538580352f;
            bone.Rotate.Z = 0;
            bone.Rotate.W = 1;

            var writeData = sourceFile.CreateSerializableData();
            string[] splitedInvMat = (writeData.Item as modelType).skeleton.bone_array.bone[0].inv_model_matrix.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            Assert.AreEqual(12, splitedInvMat.Length);

            var file = new IntermediateFile(writeData);
            var root = file.GetRootEntity<Model>();
            Assert.AreEqual(bone.Translate.X, root.Skeleton.RootBone.Translate.X);
            Assert.AreEqual(bone.Translate.Y, root.Skeleton.RootBone.Translate.Y);
            Assert.AreEqual(bone.Translate.Z, root.Skeleton.RootBone.Translate.Z);
            Assert.AreEqual(bone.Rotate.X, root.Skeleton.RootBone.Rotate.X);
            Assert.AreEqual(bone.Rotate.Y, root.Skeleton.RootBone.Rotate.Y);
            Assert.AreEqual(bone.Rotate.Z, root.Skeleton.RootBone.Rotate.Z);
            Assert.AreEqual(bone.Rotate.W, root.Skeleton.RootBone.Rotate.W);
        }

        [TestMethod]
        public void DeepCopyModel()
        {
            string modelPath = System.IO.Path.Combine(_3dToolsTestUtility.IoUtility.GetG3dDemoRootPath(), "Resources/human/human.fmdb");
            var file = IfReadUtility.ReadIntermediateFile(modelPath, null);
            file.AutoCalc = true;
            var model = file.GetRootEntity<Model>();
            Assert.IsNotNull(model.Skeleton.RootBone.Parent);
            Assert.IsNull(model.Skeleton.RootBone.ParentBone);
            var copiedModel = new Model(model);
            foreach (var bone in copiedModel.Skeleton.EnumerateBones())
            {
                Assert.IsTrue((bone.ParentBone != null) || (bone.Parent != null), $"{bone.Name}");
            }

            copiedModel.CreateSerializableData();

            // コピーしたモデルを元のモデルにマージ
            {
                copiedModel.Skeleton.EnumerateBones().Select(x => { x.Name = $"{x.Name}_Copied"; return x; });
                model.Skeleton.RootBone.ChildBones.Add(copiedModel.Skeleton.RootBone);

                model.Shapes.Add(copiedModel.Shapes.Select(x => { x.Name = $"{x.Name}_Copied"; return x; }));
                model.Vertexes.Add(copiedModel.Vertexes);
                model.Streams.Add(copiedModel.Streams);
            }

            // シェイプとボーンの削除テスト
            {
                var targetBone = model.Skeleton.EnumerateBones().ElementAt(1);
                model.Shapes.RemoveAll(x => x.ShapeInfo.Bone == targetBone);
                model.RemoveBone(targetBone);
            }

            file.CreateSerializableData();
        }

        [TestMethod]
        public void DeepCopySkeleton()
        {
            Skeleton skeleton = new Skeleton();
            var bone0 = new Bone() { Name = "hoge" };
            var bone1 = new Bone() { Name = "fuga" };
            var bone2 = new Bone() { Name = "piyo" };
            skeleton.RootBone = bone0;
            bone0.ChildBones.Add(bone1);
            bone1.ChildBones.Add(bone2);
            Skeleton copiedSkeleton = new Skeleton(skeleton);
            Assert.AreEqual(3, copiedSkeleton.CountBones());
            Assert.AreEqual("hoge", copiedSkeleton.RootBone.Name);
            Assert.AreNotEqual(bone0, copiedSkeleton.RootBone);
            Assert.AreEqual(1, copiedSkeleton.EnumerateBones().First(x => x.Name == "hoge").ChildBones.Count);
            Assert.AreEqual(1, copiedSkeleton.EnumerateBones().First(x => x.Name == "fuga").ChildBones.Count);
            Assert.AreEqual(0, copiedSkeleton.EnumerateBones().First(x => x.Name == "piyo").ChildBones.Count);
            Assert.AreNotEqual(bone1, copiedSkeleton.EnumerateBones().First(x => x.Name == "hoge").ChildBones.First());
        }


        [TestMethod]
        public void SkeletonRootBonePropertyTest()
        {
            Skeleton skeleton = new Skeleton();
            var bone0 = new Bone() { Name = "hoge" };
            var bone1 = new Bone() { Name = "fuga" };
            var bone2 = new Bone() { Name = "piyo" };

            skeleton.RootBone = bone0;
            bone0.ChildBones.Add(bone1);
            bone1.ChildBones.Add(bone2);
            Assert.AreEqual(bone0, skeleton.RootBone);

            bone0.ChildBones.Clear();
            bone1.ChildBones.Clear();
            bone2.ChildBones.Clear();
            skeleton.RootBone = bone2;
            bone2.ChildBones.Add(bone1);
            bone1.ChildBones.Add(bone0);
            Assert.AreEqual(bone2, skeleton.RootBone);

            // プロパティ変更通知のテスト
            {
                bool isRootBonePropChanged = false;
                bool isHashValuePropChanged = false;
                PropertyChangedEventHandler handler = (object sender, System.ComponentModel.PropertyChangedEventArgs e) =>
                {
                    if (e.PropertyName == "RootBone")
                    {
                        isRootBonePropChanged = true;
                    }
                    else if (e.PropertyName == "HashValue")
                    {
                        isHashValuePropChanged = true;
                    }
                };
                skeleton.PropertyChanged += handler;

                skeleton.Reset();
                Assert.AreEqual(false, isRootBonePropChanged);
                Assert.AreEqual(false, isHashValuePropChanged);
                bone2.Visibility = false;
                Assert.AreEqual(false, isRootBonePropChanged);
                Assert.AreEqual(true, isHashValuePropChanged);

                isHashValuePropChanged = false;
                Assert.AreEqual(false, isHashValuePropChanged);
                bone2.ChildBones.Remove(bone1);
                skeleton.RootBone = bone1;
                bone1.ChildBones.Add(bone2);
                Assert.AreEqual(true, isRootBonePropChanged);
                Assert.AreEqual(true, isHashValuePropChanged);
            }
        }

        [TestMethod]
        public void AddExternalModelToModel()
        {
            string modelPath = System.IO.Path.Combine(_3dToolsTestUtility.IoUtility.GetG3dDemoRootPath(), "Resources/human/human.fmdb");
            var file = IfReadUtility.ReadIntermediateFile(modelPath, null);
            var model = file.GetRootEntity<Model>();

            var copiedFile = new IntermediateFile(file);
            var copiedModel = copiedFile.GetRootEntity<Model>();

            model.Skeleton.RootBone.ChildBones.Add(copiedModel.Skeleton.RootBone);
            model.Materials.Add(copiedModel.Materials);
            model.Shapes.Add(copiedModel.Shapes);
        }

        [TestMethod]
        public void ChildBoneHashTest()
        {
            {
                var parent = new Bone { Name = "parent" };
                var child = new Bone { Name = "child" };
                parent.ChildBones.Add(child);
                Assert.AreEqual(child, parent.ChildBones.First());
                parent.AutoCalc = true;
                parent.Reset();
                Assert.IsTrue(child.AutoCalc);
                Assert.IsFalse(parent.IsDirty);
                Assert.IsFalse(child.IsDirty);

                child.Name = "fuga";

                Assert.IsTrue(child.IsDirty);
                Assert.IsTrue(parent.IsDirty);
            }

            {
                var file = new IntermediateFile(IntermediateFileKind.Model);
                file.AutoCalc = true;
                file.Reset();
                Assert.IsFalse(file.IsDirty);

                var model = file.GetRootEntity<Model>();
                model.Skeleton.RootBone = new Bone() { Name = "root" };
                Assert.IsTrue(file.IsDirty);

                file.Reset();
                Assert.IsFalse(file.IsDirty);

                var bone = new Bone() { Name = "child" };
                model.Skeleton.RootBone.ChildBones.Add(bone);
                Assert.IsTrue(file.IsDirty);

                file.Reset();
                Assert.IsFalse(file.IsDirty);
                Assert.IsTrue(bone.AutoCalc);
                Assert.IsFalse(bone.IsDirty);
                Assert.IsFalse(bone.Translate.IsDirty);

                bone.Translate.X = 1.0f;
                Assert.IsTrue(bone.Translate.IsDirty);
                Assert.IsTrue(bone.IsDirty);
                Assert.IsTrue(model.Skeleton.RootBone.IsDirty);
                Assert.IsTrue(file.IsDirty);
            }
        }
    }
}
