﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Empty.cpp,PageSampleEmpty}
 *
 * @brief
 * 画面クリアのサンプルプログラム
 */

/**
 * @page PageSampleEmpty 画面クリアのサンプルプログラム
 * @tableofcontents
 *
 * @brief
 * サンプルプログラムEmptyの解説です。
 *
 * @section PageSampleEmpty_SectionBrief 概要
 * 画面クリアのサンプルプログラムです。
 *
 * @section PageSampleEmpty_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleEmpty_SectionNecessaryEnvironment 必要な環境
 * 画面表示が利用可能である必要があります。
 *
 * @section PageSampleEmpty_SectionHowToOperate 操作方法
 * startボタン( PC の場合はスペースキー)により、メニューに戻ります。
 *
 * @section PageSampleEmpty_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleEmpty_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューからEmptyを選択し、実行してください。
 *
 * @section PageSampleEmpty_SectionDetail 解説
 * このサンプルプログラムは、フレームバッファをフレームごとに色を変えながらクリアしていきます。
 * g3d demoで用意されているユーティリティクラスの使用方法を示しています。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - コマンドバッファを初期化
 * - フレームバッファを初期化
 * - ループ開始
 * - クリアカラーを計算
 * - コマンドリスト作成
 * - コマンドリストを実行
 * - スキャンバッファをスワップ
 * - ループ開始に戻る
 *
 */

#include "g3ddemo_GfxUtility.h"

namespace g3ddemo = nn::g3d::demo;

int EmptyMain()
{
    nn::gfx::Device* pDevice = g3ddemo::GetDevice();

    // コマンドバッファを初期化
    g3ddemo::CommandBuffer commandBuffer;
    {
        commandBuffer.Setup(pDevice, 1);
    }

    // フレームバッファを初期化
    g3ddemo::FrameBuffer frameBuffer;
    {
        g3ddemo::FrameBuffer::InitArg initArg(1280, 720);
        frameBuffer.Setup(pDevice, &initArg);
    }

    // メインループ
    g3ddemo::Pad pad;
    uint32_t counter = 0;
    while (NN_STATIC_CONDITION(1))
    {
        g3ddemo::ProcessMessage();
        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        // 計算
        float sinR, sinG, sinB, cosR, cosG, cosB;
        {
            ++counter;
            float step = 3.14f / 128;
            sinR = std::sin(step * counter);
            cosR = std::cos(step * counter);
            sinG = std::sin(step * (counter + 0x50));
            cosG = std::cos(step * (counter + 0x50));
            sinB = std::sin(step * (counter + 0xA0));
            cosB = std::cos(step * (counter + 0xA0));
            sinR = sinR * 0.5f + 0.5f; cosR = cosR * 0.5f + 0.5f;
            sinG = sinG * 0.5f + 0.5f; cosG = cosG * 0.5f + 0.5f;
            sinB = sinB * 0.5f + 0.5f; cosB = cosB * 0.5f + 0.5f;
            frameBuffer.SetClearColor(sinR, sinG, sinB);
        }

        // コマンド生成
        commandBuffer.Begin();
        {
            // クリア
            frameBuffer.Clear(commandBuffer.GetCurrent());
        }
        // プリミティブレンダラのビューポートをここで設定しておく
        commandBuffer.End();

        // コマンド実行
        g3ddemo::ExecuteCommand(commandBuffer.GetCurrent());
        // スキャンバッファへコピーアウト
        g3ddemo::CopyOut(frameBuffer.GetColorBuffer()->GetColorTargetView());

        // 垂直同期待ちをし、バッファをスワップする
        g3ddemo::SwapScanBuffers(1);
    }

    // フレームバッファの破棄
    {
        frameBuffer.Cleanup(pDevice);
    }

    // コマンドバッファの破棄
    {
        commandBuffer.Cleanup(pDevice);
    }

    return 0;
}
