﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Forms;
using System.Threading;
using System.Reflection;
using NintendoWare.G3d.Edit;
using System.Diagnostics;
using G3dHioTestTool;
using NintendoWare.G3d.Edit.Math;

namespace G3dHioTestTool
{
    public partial class MainForm : Form
    {
        private Thread pollingThread;
        private readonly object syncObj = new object();
        private NintendoWare.G3d.Edit.HIO hio = null;

        private Model dummyAttachModel = new Model() { Key = 4321 };
        private Model dummyRetargetingHostModel = new Model() { Key = 2 };
        private Model dummyRetargetClientModel = new Model() { Key = 3 };

        private List<Model> models = new List<Model>();
        private List<Animation> animations = new List<Animation>();
        private List<Texture> textures = new List<Texture>();
        private FileData dummyFileData = new FileData()
        {
            // バイナリによっては変えないといけないが、とりあえず固定
            Alignment = 8192
        };

        private static uint loadModelKey = 1;
        private static uint loadAnimKey = 1;
        private static uint loadTextureKey = 1;
        private static readonly string hostModelName = "PierrotAnimDance";

        public class ComboboxItem
        {
            public string Text { get; set; }

            public override string ToString()
            {
                return Text;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainForm()
        {
            InitializeComponent();
            ConnectionMenuItem.Enabled = true;
            CloseConnectionMenuItem.Enabled = false;

            // ホストIOの初期化
            NintendoWare.G3d.Edit.HIO.InitializeHIO();
            this.hio = NintendoWare.G3d.Edit.HIO.GetInstance();
            this.hio.PropertyChanged += MainForm_ChangeConnection;
            this.hio.RecvInfo.RuntimeErrorOccured += this.OnRuntimeErrorOccured;
            this.hio.RecvInfo.FileLoaded += this.OnFileLoaded;
            this.hio.RecvInfo.FileReloaded += this.OnFileReloaded;
            this.hio.RecvInfo.ModelFileLoaded += this.OnModelFileLoaded;
            this.hio.RecvInfo.AttachModelReceived += this.OnAttachModelCommandReceived;
            this.hio.RecvInfo.DetachModelReceived += this.OnDetachModelCommandReceived;
            this.hio.RecvInfo.ShowMessageRequested += this.OnShowMessageRequested;

            // デフォルトはHTCにしておく
            this.hio.ChangeCommDevice(HIOBase.TargetType.Htc, "Generic-Windows");

            //this.hio.SelectTargetShapes(dummyModel, new int[1] { 0 });
            //this.hio.SelectTargetMaterials(dummyModel, new int[1] { 0 });

            string RelativeContentDir = "..\\Contents";
            string contentDir = System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), RelativeContentDir);
            if (System.IO.Directory.Exists(contentDir))
            {
                foreach (string filePath in System.IO.Directory.EnumerateFiles(contentDir))
                {
                    string fileName = System.IO.Path.GetFileName(filePath);
                    string relativePath = System.IO.Path.Combine(RelativeContentDir, fileName);
                    if (IsBfresFile(filePath))
                    {
                        this.BfresPathComboBox.Items.Add(new ComboboxItem() { Text = relativePath });
                        this.AnimBfresPathComboBox.Items.Add(new ComboboxItem() { Text = relativePath });
                    }
                    else if (IsBntxFile(filePath))
                    {
                        this.BntxPathComboBox.Items.Add(new ComboboxItem() { Text = relativePath });
                    }
                    else if (IsBfshaFile(filePath))
                    {
                        this.BfshaPathComboBox.Items.Add(new ComboboxItem() { Text = relativePath });
                    }
                }
            }

            if (this.BfresPathComboBox.Items.Count > 0)
            {
                this.BfresPathComboBox.Text = this.BfresPathComboBox.Items[0].ToString();
            }

            if (this.AnimBfresPathComboBox.Items.Count > 0)
            {
                this.AnimBfresPathComboBox.Text = this.AnimBfresPathComboBox.Items[0].ToString();
            }

            if (this.BntxPathComboBox.Items.Count > 0)
            {
                this.BntxPathComboBox.Text = this.BntxPathComboBox.Items[0].ToString();
            }

            if (this.BfshaPathComboBox.Items.Count > 0)
            {
                this.BfshaPathComboBox.Text = this.BfshaPathComboBox.Items[0].ToString();
            }
            this.dummyFileData.FileName = this.GetBfresFilePathFromTextBox();
        }

        private Model FindModel(uint key)
        {
            foreach (Model model in this.models)
            {
                if (model.Key == key)
                {
                    return model;
                }
            }

            return null;
        }

        private string GetTargetTypeAsString(HIOBase.TargetType targetType)
        {
            switch (targetType)
            {
                case HIOBase.TargetType.Htc: return "Htc";
                case HIOBase.TargetType.Cafe: return "Cafe";
                case HIOBase.TargetType.OldPc: return "OldPc";
                default: return "Unknown";
            }
        }

        private void SyncInfo()
        {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.Append(GetTargetTypeAsString(hio.TargetDeviceType));

            if (this.hio.IsPingConnected)
            {
                stringBuilder.Append(", Ping 接続中");
            }
            else
            {
                stringBuilder.Append(", Ping 未接続");
            }

            if (this.hio.IsConnected)
            {
                stringBuilder.Append(", 接続中");
            }
            else
            {
                stringBuilder.Append(", 未接続");
            }

            InfoTextBox.Invoke((MethodInvoker)(() =>
                {
                    this.InfoTextBox.Text = stringBuilder.ToString();
                })
            );

        }

        /// <summary>
        /// ポーリングスレッドの実行処理です。
        /// </summary>
        private void PollingTask()
        {
            while (true)
            {
                lock (this.syncObj)
                {
                    this.hio.Poll();
                    this.hio.PollPing();
                }

                SyncInfo();
            }
        }

        private string GetRuntimeErrorMessage(RuntimeError error)
        {
            switch (error)
            {
                case RuntimeError.NoError: return "エラーはありません";
                case RuntimeError.InvalidMaterialCount: return "ランタイム側のリソースとマテリアル数が異なります";
                default: return string.Format("不明なエラー({0})です", error);
            }
        }

        /// <summary>
        /// モデル読み込み完了時のコールバックです。
        /// </summary>
        private void OnModelFileLoaded(object sender, ModelFileLoadedEventArgs arg)
        {
            this.WriteInfoLog(string.Format(
                "モデルファイル読み込み完了通知受信: ToolKey = {0}, ModelObjKey = {1}, ResFileKey = {2}, ResModelKey = {3}",
                arg.ToolKey, arg.ModelObjKey, arg.ResFileKey, arg.ResModelKey));

            Model targetModel = this.models.FirstOrDefault(model => model.Key == arg.ToolKey);
            Debug.Assert(targetModel != null);

            targetModel.ModelObjKey = arg.ModelObjKey;
            targetModel.ResFileKey = arg.ResFileKey;
            targetModel.ResModelKey = arg.ResModelKey;

            // ロードしたモデルを適当な位置に移動する
            {
                hio.ClearSelectedModels();
                hio.SelectTargetModel(targetModel);
                Random random = new Random();
                Vector3 scale = new Vector3(1.0f, 1.0f, 1.0f);
                Vector3 rotate = new Vector3(0.0f, 0.0f, 0.0f);
                Vector3 translate = new Vector3((float)(random.NextDouble() * 20.0 - 10.0), 0.0f, 0.0f);
                hio.EditModelLayout(false, scale, rotate, translate);
            }

            // テクスチャのバインド情報を送信
            {
                Texture[] loadedTextures = this.textures.Where(tex => tex.ResFileKey > 0).ToArray();
                this.WriteInfoLog(string.Format("モデルにテクスチャバインド: modelObjKey = {0}, texture count = {1}", arg.ModelObjKey, loadedTextures.Length));
                HIO.Result result = this.hio.SetTextureBindingsForModel(targetModel, loadedTextures);
                if (result != HIO.Result.Success)
                {
                    this.WriteErrorLog(string.Format("SetTextureBindings() 失敗: {0}", this.dummyFileData.FileName));
                    return;
                }
            }
        }

        private void OnFileLoaded(object sender, FileLoadedEventArgs arg)
        {
            this.WriteInfoLog(string.Format("ファイル読み込み完了通知受信: ToolKey = {0}, ResFileKey = {1}", arg.ToolKey, arg.ResFileKey));

            {
                Animation target = this.animations.FirstOrDefault(x => x.Key == arg.ToolKey);
                if (target != null)
                {
                    this.WriteInfoLog(string.Format("アニメーションの読み込み完了"));
                    target.ResFileKey = arg.ResFileKey;

                    // テクスチャのバインド情報を送信
                    if (target.AnimKind == AnimKind.MaterialAnim)
                    {
                        Texture[] loadedTextures = this.textures.Where(tex => tex.ResFileKey > 0).ToArray();
                        this.WriteInfoLog(string.Format("アニメにテクスチャバインド: resFileKey = {0}, texture count = {1}", arg.ResFileKey, loadedTextures.Length));
                        HIO.Result result = this.hio.SetTextureBindingsForAnimation(target, loadedTextures);
                        if (result != HIO.Result.Success)
                        {
                            this.WriteErrorLog(string.Format("SetTextureBindings() 失敗: {0}", this.dummyFileData.FileName));
                            return;
                        }
                    }
                    return;
                }
            }

            {
                Texture target = this.textures.FirstOrDefault(x => x.Key == arg.ToolKey);
                if (target != null)
                {
                    this.WriteInfoLog(string.Format("テクスチャの読み込み完了"));
                    target.ResFileKey = arg.ResFileKey;
                    return;
                }
            }
        }

        private void OnFileReloaded(object sender, FileReloadedEventArgs arg)
        {
            this.WriteInfoLog(
                string.Format(
                "ファイル再読み込み完了通知受信: ResFileKey = {0}, NewResFileKey = {1}", arg.ResFileKey, arg.NewResFileKey));

            {
                Animation target = this.animations.FirstOrDefault(x => x.ResFileKey == arg.ResFileKey);
                if (target != null)
                {
                    this.WriteInfoLog(string.Format("アニメーションの再読み込み完了"));
                    target.ResFileKey = arg.NewResFileKey;
                    return;
                }
            }

            {
                Texture target = this.textures.FirstOrDefault(x => x.ResFileKey == arg.ResFileKey);
                if (target != null)
                {
                    this.WriteInfoLog(string.Format("テクスチャの再読み込み完了"));
                    target.ResFileKey = arg.NewResFileKey;
                    return;
                }
            }
        }

        private void OnRuntimeErrorOccured(object sender, RuntimeErrorOccuredEventArgs e)
        {
            this.WriteInfoLog("ランタイムのエラー通知受信");

            RuntimeError error = e.RuntimeError;
            this.WriteErrorLog(GetRuntimeErrorMessage(error));

            this.hio.Close();
            this.hio.Open();
        }

        private void OnAttachModelCommandReceived(object sender, AttachModelReceivedArgs arg)
        {
            this.WriteInfoLog("アタッチモデルコマンド受信");

            if (!System.IO.File.Exists(this.dummyFileData.FileName))
            {
                this.WriteErrorLog(string.Format("{0}が存在しません", this.dummyFileData.FileName));
                return;
            }

            if (!string.IsNullOrEmpty(arg.AttachPath))
            {
                string path = arg.AttachPath;
                if (!System.IO.File.Exists(path))
                {
                    this.Invoke(
                        new MethodInvoker(() =>
                        {
                            OpenFileDialog dialog = new OpenFileDialog();
                            dialog.Title = "アタッチ中にデタッチするテスト用なのでファイルを選んでもアタッチできません";
                            DialogResult result = dialog.ShowDialog(this);
                            if (result != System.Windows.Forms.DialogResult.OK)
                            {
                                this.hio.CancelAttachModel(arg.ModelObjKey);
                                this.WriteInfoLog("Attach canceled");
                                return;
                            }

                            path = dialog.FileName;
                        }));
                }
            }

            this.dummyAttachModel.ModelObjKey = arg.ModelObjKey;
            bool success = this.hio.SendAttachModel(this.dummyAttachModel, this.dummyFileData);
            if (!success)
            {
                this.WriteErrorLog(string.Format("SendAttachModel() 失敗: {0}", this.dummyFileData.FileName));
                return;
            }

            this.WriteInfoLog(string.Format("アタッチモデルコマンド送信: {0}", this.dummyFileData.FileName));

            // テクスチャのバインド情報を送信
            {
                Texture[] loadedTextures = this.textures.Where(tex => tex.ResFileKey > 0).ToArray();
                HIO.Result result = this.hio.SetTextureBindingsForModel(this.dummyAttachModel, loadedTextures);
                if (result != HIO.Result.Success)
                {
                    this.WriteErrorLog(string.Format("SetTextureBindings() 失敗: {0}", this.dummyFileData.FileName));
                    return;
                }
            }
        }

        private void OnDetachModelCommandReceived(object sender, DetachModelReceivedArgs arg)
        {
            this.WriteInfoLog("デタッチモデルコマンド受信");

            this.dummyAttachModel.ModelObjKey = 0;
            this.WriteInfoLog(string.Format("デタッチモデルコマンド送信: {0}", this.dummyFileData.FileName));
        }

        private void OnShowMessageRequested(object sender, ShowMessageRequestedArgs arg)
        {
            this.WriteInfoLog(string.Format(
                "メッセージ表示コマンド受信\nMessageType = {0}\nMessageDestination = {1}\nMessage = {2}",
                arg.MessageType, arg.MessageDestination, arg.Message));
            switch (arg.MessageDestination)
            {
                case MessageDestination.Dialog:
                    {
                        string title = "不明な種類のメッセージ";
                        switch (arg.MessageType)
                        {
                            case MessageType.UserInfo:
                                {
                                    title = "情報";
                                }
                                break;
                            case MessageType.UserWarning:
                                {
                                    title = "警告";
                                }
                                break;
                            case MessageType.UserError:
                                {
                                    title = "エラー";
                                }
                                break;
                            default:
                                break;
                        }
                        MessageBox.Show(arg.Message, title, MessageBoxButtons.OK);
                    }
                    break;
                case MessageDestination.Log:
                    {
                        switch (arg.MessageType)
                        {
                            case MessageType.UserInfo:
                                {
                                    WriteInfoLog(arg.Message);
                                }
                                break;
                            case MessageType.UserWarning:
                                {
                                    WriteWarningLog(arg.Message);
                                }
                                break;
                            case MessageType.UserError:
                                {
                                    WriteErrorLog(arg.Message);
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// アプリケーション初期化完了後の処理です。
        /// </summary>
        /// <param name="sender">イベント送信元です。</param>
        /// <param name="e">イベント引数です。</param>
        private void MainForm_Load(object sender, EventArgs e)
        {
            this.WriteInfoLog("ポーリングスレッド開始");
            pollingThread = new Thread(new ThreadStart(this.PollingTask));
            pollingThread.Name = "ポーリングスレッド";
            pollingThread.IsBackground = true;
            pollingThread.Start();
        }

        /// <summary>
        /// ウィンドウクローズ時の処理です。
        /// </summary>
        /// <param name="sender">イベント送信元です。</param>
        /// <param name="e">イベント引数です。</param>
        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            this.hio.PropertyChanged -= MainForm_ChangeConnection;
            pollingThread.Abort();
            pollingThread.Join();
            NintendoWare.G3d.Edit.HIO.FinalizeHIO();
        }

        private void CloseMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        // ====================================================================
        //                             ログ関連
        // ====================================================================
        public string Log
        {
            get { return this.DebugConsole.Text; }
            set
            {
                this.DebugConsole.Text = value;
            }
        }

        public void WriteInfoLog(string message)
        {
            this.Invoke(
                new MethodInvoker(() =>
                {
                    string replacedMessage = message.Replace("\n", System.Environment.NewLine);
                    Log += "■" + replacedMessage + System.Environment.NewLine;
                }
                ));
        }

        public void WriteWarningLog(string message)
        {
            this.Invoke(
                new MethodInvoker(() =>
                {
                    Log += "Warning: " + message + System.Environment.NewLine;
                }
                ));
        }

        public void WriteErrorLog(string message)
        {
            this.Invoke(
                new MethodInvoker(() =>
                {
                    Log += "Error: " + message + System.Environment.NewLine;
                }
                ));
        }

        private void ClearLogMenuItem_Click(object sender, EventArgs e)
        {
            this.DebugConsole.Clear();
        }

        private void CafeConnectedMenuItem_Click(object sender, EventArgs e)
        {
            Disconnect();

            lock (this.syncObj)
            {
                if (!(this.hio.TargetDeviceType == HIOBase.TargetType.Cafe))
                {
                    this.hio.ChangeCommDevice(HIOBase.TargetType.Cafe, "");
                }

                HtcResult result = this.hio.Connect();
                if (result.IsSuccess)
                {
                    WriteInfoLog("Cafe接続成功");
                    hio.EnableRuntimeDebugLog();
                    WriteInfoLog("デバッグログを有効化");
                }
                else
                {
                    WriteErrorLog("Cafe接続失敗");
                }
            }
        }

        private void Disconnect()
        {
            if (this.hio.IsConnected)
            {
                this.UnloadModel();
                this.UnloadAllTextures();
                this.hio.Disconnect();
                this.WriteInfoLog("切断");
            }

            this.models.Clear();
            this.textures.Clear();
            this.animations.Clear();
        }

        private void CloseConnectionMenuItem_Click(object sender, EventArgs e)
        {
            Disconnect();
        }

        private void hTCToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Disconnect();

            lock (this.syncObj)
            {
                this.hio.ChangeCommDevice(HIOBase.TargetType.Htc, "Generic-Windows");
                if (this.hio.Connect().IsSuccess)
                {
                    WriteInfoLog("HTC接続成功");
                    hio.EnableRuntimeDebugLog();
                    WriteInfoLog("デバッグログを有効化");
                }
                else
                {
                    WriteErrorLog("HTC接続失敗");
                }
            }
        }

        private void PCConnectedMenuItem_Click(object sender, EventArgs e)
        {
            Disconnect();

            lock (this.syncObj)
            {
                this.hio.ChangeCommDevice(HIOBase.TargetType.OldPc, "");
                if (this.hio.Connect().IsSuccess)
                {
                    WriteInfoLog("PC接続成功");
                    hio.EnableRuntimeDebugLog();
                    WriteInfoLog("デバッグログを有効化");
                }
                else
                {
                    WriteErrorLog("PC接続失敗");
                }
            }
        }

        private void MainForm_ChangeConnection(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "IsConnected")
            {
                var instance = this.hio;
                if (instance.IsConnected)
                {
                    this.Invoke(
                        new MethodInvoker(() =>
                        {
                            CloseConnectionMenuItem.Enabled = true;
                        }));
                }
                else
                {
                    this.Invoke(
                        new MethodInvoker(() =>
                        {
                            CloseConnectionMenuItem.Enabled = false;
                        }));
                    Disconnect();
                }
            }
            else if (e.PropertyName == "IsPingConnected")
            {
            }
            else if (e.PropertyName == "IsAttachCommandReceived")
            {
            }
        }

        private void ロードモデルToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (!this.hio.IsConnected)
            {
                this.WriteErrorLog("接続されていません");
                return;
            }

            if (!System.IO.File.Exists(this.dummyFileData.FileName))
            {
                this.WriteErrorLog(string.Format("{0}が存在しません", this.dummyFileData.FileName));
                return;
            }

            LoadModelFile(this.GetBfresFilePathFromTextBox());
        }

        private Model LoadModelFile(string bfresFilePath)
        {
            if (!IsBfresFile(bfresFilePath))
            {
                this.WriteErrorLog(string.Format("{0}はbfresファイルではありません", bfresFilePath));
                return null;
            }

            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };
            Model model = new Model() { Key = loadModelKey };
            ++loadModelKey;

            this.models.Add(model);
            bool success = this.hio.LoadModel(model, fileData);
            if (!success)
            {
                this.WriteErrorLog(string.Format("LoadModel()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("ロードモデルコマンド送信: {0}", fileData.FileName));
            return model;
        }

        private Animation LoadSkeletalAnimFile(string bfresFilePath)
        {
            if (!IsBfresFile(bfresFilePath))
            {
                this.WriteErrorLog(string.Format("{0}はbfresファイルではありません", bfresFilePath));
                return null;
            }

            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };
            Animation anim = new Animation();
            anim.Key = loadAnimKey;
            anim.AnimKind = AnimKind.SkeletalAnim;
            ++loadAnimKey;
            this.animations.Add(anim);
            bool success = hio.LoadSkeletalAnimation(anim, fileData, hostModelName);
            if (!success)
            {
                this.WriteErrorLog(string.Format("LoadSkeletalAnimation()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("ロードスケルタルアニメ: {0}", fileData.FileName));
            return anim;
        }

        private Animation LoadMaterialAnimFile(string bfresFilePath)
        {
            if (!IsBfresFile(bfresFilePath))
            {
                this.WriteErrorLog(string.Format("{0}はbfresファイルではありません", bfresFilePath));
                return null;
            }

            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };
            Animation anim = new Animation();
            anim.Key = loadAnimKey;
            anim.AnimKind = AnimKind.MaterialAnim;
            ++loadAnimKey;
            this.animations.Add(anim);
            bool success = hio.LoadMaterialAnimation(anim, fileData);
            if (!success)
            {
                this.WriteErrorLog(string.Format("LoadMaterialAnimation()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("ロードマテリアルアニメ: {0}", fileData.FileName));
            return anim;
        }

        private Texture LoadTextureFile(string bntxFilePath)
        {
            if (!IsBntxFile(bntxFilePath))
            {
                this.WriteErrorLog(string.Format("{0}はbntxファイルではありません", bntxFilePath));
                return null;
            }

            // アライメントは本当はヘッダーから取得する。とりあえず適当
            FileData fileData = new FileData() { FileName = bntxFilePath, Alignment = 8192 };
            Texture texture = new Texture();
            texture.Key = loadTextureKey;
            ++loadTextureKey;
            this.textures.Add(texture);
            HIO.Result result = hio.LoadTexture(texture, fileData);
            if (result != HIO.Result.Success)
            {
                this.WriteErrorLog(string.Format("LoadTexture()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("ロードテクスチャ: {0}", fileData.FileName));
            return texture;
        }

        private bool IsBntxFile(string filePath)
        {
            return System.IO.Path.GetExtension(filePath).Equals(".bntx");
        }

        private bool IsBfresFile(string filePath)
        {
            return System.IO.Path.GetExtension(filePath).Equals(".bfres");
        }

        private bool IsBfshaFile(string filePath)
        {
            return System.IO.Path.GetExtension(filePath).Equals(".bfsha");
        }

        private Texture ReloadTextureFile(Texture texture, string bntxFilePath)
        {
            if (!IsBntxFile(bntxFilePath))
            {
                this.WriteErrorLog(string.Format("{0}はbntxファイルではありません", bntxFilePath));
                return null;
            }

            // アライメントは本当はヘッダーから取得する。とりあえず適当
            FileData fileData = new FileData() { FileName = bntxFilePath, Alignment = 8192 };
            HIO.Result result = hio.ReloadTexture(texture, fileData);
            if (result != HIO.Result.Success)
            {
                this.WriteErrorLog(string.Format("ReloadTexture()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("リロードテクスチャ: {0}(ResFileKey {1})", fileData.FileName, texture.ResFileKey));
            return texture;
        }

        private void UnloadAllTextures()
        {
            foreach (Texture texture in this.textures)
            {
                HIO.Result result = this.hio.UnloadTexture(texture);
                if (result != HIO.Result.Success)
                {
                    this.WriteErrorLog("UnloadTexture() 失敗");
                    return;
                }

                this.WriteInfoLog(string.Format("アンロードテクスチャコマンド送信: key = {0}, resFileKey = {1}", texture.Key, texture.ResFileKey));
                texture.ResFileKey = 0;
            }

            this.textures.Clear();
        }

        private void テクスチャロードToolStripMenuItem_Click(object sender, EventArgs e)
        {
            LoadTextureFile(this.GetBntxFilePathFromTextBox());
        }

        private void テクスチャリロードToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (this.textures.Count == 0)
            {
                return;
            }

            ReloadTextureFile(this.textures[0], this.GetBntxFilePathFromTextBox());
        }

        private void テクスチャアンロードToolStripMenuItem_Click(object sender, EventArgs e)
        {
            UnloadAllTextures();
        }

        private void ロードスケルタルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            LoadSkeletalAnimFile(this.GetAnimBfresFilePathFromTextBox());
        }

        private void FrameTrackBar_Scroll(object sender, EventArgs e)
        {
            TrackBar trackBar = sender as TrackBar;
            this.hio.StopFrameCtrl(trackBar.Value);
        }

        private void バインドスケルタルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Animation anim = this.animations.FirstOrDefault(x => x.Key > 0 && x.AnimKind == AnimKind.SkeletalAnim);
            BindAnim(anim);
        }

        private void ロードリターゲットテストモデルToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (!this.hio.IsConnected)
            {
                this.WriteErrorLog("接続されていません");
                return;
            }

            {
                string hostFilePath = System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), "..\\Contents\\RetargetHostModel.bfres");
                Model hostModel = LoadModelFile(hostFilePath);
                Debug.Assert(hostModel != null);
            }

            {
                string clientFilePath = System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), "..\\Contents\\RetargetClientModel.bfres");
                Model clientModel = LoadModelFile(clientFilePath);
                Debug.Assert(clientModel != null);
            }
        }

        private void リロードモデルToolStripMenuItem1_Click(object sender, EventArgs e)
        {
            if (!this.hio.IsConnected)
            {
                this.WriteErrorLog("接続されていません");
                return;
            }

            foreach (Model model in this.models)
            {
                bool success = this.hio.ReloadModel(model, this.dummyFileData, null, null);
                if (!success)
                {
                    this.WriteErrorLog(string.Format("ReloadModel() 失敗: {0}", this.dummyFileData.FileName));
                    return;
                }

                Texture[] loadedTextures = this.textures.Where(tex => tex.ResFileKey > 0).ToArray();
                HIO.Result result = this.hio.SetTextureBindingsForModel(model, loadedTextures);
                if (result != HIO.Result.Success)
                {
                    this.WriteErrorLog(string.Format("SetTextureBindings() 失敗: {0}", this.dummyFileData.FileName));
                    return;
                }

                this.WriteInfoLog(string.Format("リロードモデルコマンド送信: {0}", this.dummyFileData.FileName));
            }
        }

        private void アンロードモデルToolStripMenuItem_Click(object sender, EventArgs e)
        {
            foreach (Model model in this.models)
            {
                bool result = this.hio.UnloadModel(model);
                if (!result)
                {
                    this.WriteErrorLog("UnloadModel() 失敗");
                    return;
                }

                model.ResModelKey = 0;
                model.ResFileKey = 0;
                model.ModelObjKey = 0;

                this.WriteInfoLog(string.Format("アンロードモデルコマンド送信: {0}", model.Key));
            }

            this.models.Clear();
        }

        private void UnloadMenuItem_Click(object sender, EventArgs e)
        {
            UnloadModel();
        }

        private void UnloadModel()
        {
            foreach (Model model in this.models)
            {
                bool result = this.hio.UnloadModel(model);
                if (!result)
                {
                    this.WriteErrorLog("UnloadModel() 失敗");
                    return;
                }

                model.ResModelKey = 0;
                model.ResFileKey = 0;
                model.ModelObjKey = 0;

                this.WriteInfoLog(string.Format("アンロードモデルコマンド送信: {0}", model.Key));
            }

            this.models.Clear();
        }

        private void SendBfresPathTextBox_TextChanged(object sender, EventArgs e)
        {
            this.dummyFileData.FileName = this.GetBfresFilePathFromTextBox();
        }

        private string GetFullPath(string path)
        {
            if (this.IsAbsolutePath(path))
            {
                return path;
            }
            else
            {
                return System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), path);
            }
        }

        private string GetBfresFilePathFromTextBox()
        {
            return GetFullPath(this.BfresPathComboBox.Text);
        }

        private string GetAnimBfresFilePathFromTextBox()
        {
            return GetFullPath(this.AnimBfresPathComboBox.Text);
        }

        private string GetBntxFilePathFromTextBox()
        {
            return GetFullPath(this.BntxPathComboBox.Text);
        }

        private string GetBfshaFilePathFromTextBox()
        {
            return GetFullPath(this.BfshaPathComboBox.Text);
        }

        private bool IsAbsolutePath(string path)
        {
            if (System.IO.Path.IsPathRooted(path))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        private void DummyUserCommandMenuItem_Click(object sender, EventArgs e)
        {
            string scriptText = "Apply Blur\nApply Glow";
            this.hio.SelectTargetBones(dummyAttachModel, new int[3] { 3, 1, 2 });
            this.hio.SelectTargetShapes(dummyAttachModel, new int[2] { 1, 0 });
            this.hio.SelectTargetMaterials(dummyAttachModel, new int[2] { 1, 0 });

            try
            {
                this.hio.ExecuteModelUserScript(dummyAttachModel, scriptText);
            }
            catch (Exception exception)
            {
                this.WriteErrorLog(exception.Message);
                return;
            }

            this.WriteInfoLog(string.Format("ユーザコマンド送信: \nscriptText = \n{0}\n", scriptText));
        }

        private void SelectMaterialMenuItem_Click(object sender, EventArgs e)
        {
            this.hio.EditSelectTarget(SelectTargetKind.Material);
            this.WriteInfoLog("マテリアル選択");
        }

        private void SelectShapeMenuItem_Click(object sender, EventArgs e)
        {
            this.hio.EditSelectTarget(SelectTargetKind.Shape);
            this.WriteInfoLog("シェイプ選択");
        }

        private void リロードスケルタルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            string bfresFilePath = this.GetAnimBfresFilePathFromTextBox();
            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };

            foreach (var anim in this.animations)
            {
                if (anim.AnimKind != AnimKind.SkeletalAnim)
                {
                    continue;
                }

                bool success = hio.ReloadSkeletalAnimation(anim, fileData);
                if (!success)
                {
                    this.WriteErrorLog(string.Format("LoadSkeletalAnimation()失敗: {0}", fileData.FileName));
                    continue;
                }

                this.WriteInfoLog(string.Format("リロードスケルタルアニメ: {0}", fileData.FileName));
            }
        }

        private void PauseCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            CheckBox checkbox = sender as CheckBox;
            bool isPaused = checkbox.Checked;
            float frame = FrameTrackBar.Value;
            if (this.models.Count > 0)
            {
                // 最初のモデルをテスト対象にしてアニメーション変更
                Model targetModel = this.models[0];
                foreach (var skeltalAnim in this.animations)
                {
                    bool success = hio.EditAnimationPause(targetModel, skeltalAnim, frame, isPaused);
                    if (!success)
                    {
                        this.WriteErrorLog(string.Format("EditAnimationPause()失敗: {0}"));
                        continue;
                    }

                    this.WriteInfoLog(string.Format("モデル {0} のアニメーション {1} を {2}", targetModel.Key, skeltalAnim.ResFileKey, isPaused ? "停止" : "再生"));
                }

                PrintModelInfo(targetModel);
            }
            else
            {
                this.WriteErrorLog("モデルがロードされていません\n");
            }
        }

        private void PrintModelInfo(Model model)
        {
            this.WriteInfoLog(
                string.Format(
                "Key = {0}, ModelObjKey = {1}, ResFileKey = {2}, ResModelKey = {3}, IsSendAttached = {4}\n",
                model.Key,
                model.ModelObjKey,
                model.ResFileKey,
                model.ResModelKey,
                model.IsSendAttached));
        }

        private void RetargetEnabledCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            UpdateRetargetingState();
        }

        private void UpdateRetargetingState()
        {
            int modelIndex = (int)retargetHostModelIndexNumericUpDown.Value;
            if (modelIndex >= this.models.Count)
            {
                return;
            }

            if (this.retargetingEnabledCheckBox.Checked)
            {
                Model hostModel = this.models[modelIndex];
                if (hostModel != null)
                {
                    foreach (var anim in this.animations)
                    {
                        hio.SetRetargetingHostModel(anim, hostModel);
                    }
                }
            }
            else
            {
                foreach (var anim in this.animations)
                {
                    hio.SetRetargetingHostModel(anim, null);
                }
            }
        }

        private void RetargetingHostModelIndexUpDown_ValueChanged(object sender, EventArgs e)
        {
            UpdateRetargetingState();
        }

        private void ログ有効化ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            WriteInfoLog("ランタイムデバッグログ有効化コマンド送信");
            HIO.Result result = this.hio.EnableRuntimeDebugLog();
            if (result != HIO.Result.Success)
            {
                WriteErrorLog(string.Format("EnableRuntimeDebugLog failed"));
            }
        }

        private void ログ無効化ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            WriteInfoLog("ランタイムデバッグログ無効化コマンド送信");
            HIO.Result result = this.hio.DisableRuntimeDebugLog();
            if (result != HIO.Result.Success)
            {
                WriteErrorLog(string.Format("DisableRuntimeDebugLog failed"));
            }
        }

        private void ロードマテリアルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            LoadMaterialAnimFile(this.GetAnimBfresFilePathFromTextBox());
        }

        private void バインドマテリアルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Animation anim = this.animations.FirstOrDefault(x => x.Key > 0 && x.AnimKind == AnimKind.MaterialAnim);
            BindAnim(anim);
        }

        private void BindAnim(Animation anim)
        {
            if (anim == null)
            {
                this.WriteErrorLog("アニメーションが読み込まれていません");
                return;
            }

            foreach (Model model in this.models)
            {
                if (model.Key == 0)
                {
                    continue;
                }

                bool success = hio.BindAnimations(model, new Animation[] { anim });
                if (!success)
                {
                    this.WriteErrorLog(string.Format("BindAnimations()失敗"));
                    return;
                }

                this.WriteInfoLog(string.Format("バインドアニメ [key = {0}, resFileKey = {1}]", anim.Key, anim.ResFileKey));
            }
        }

        private void アンロードマテリアルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Animation anim = this.animations.FirstOrDefault(x => x.Key > 0 && x.AnimKind == AnimKind.MaterialAnim);
            bool success = this.hio.UnloadMaterialAnimation(anim);
            if (!success)
            {
                this.WriteErrorLog(string.Format("UnloadMaterialAnimation()失敗 key = {0}, resFileKey = {1}", anim.Key, anim.ResFileKey));
                return;
            }

            this.WriteInfoLog(string.Format("アンロードマテリアルアニメ [key = {0}, resFileKey = {1}]", anim.Key, anim.ResFileKey));
        }

        private void リロードマテリアルアニメToolStripMenuItem_Click(object sender, EventArgs e)
        {
            string bfresFilePath = this.GetAnimBfresFilePathFromTextBox();
            if (!IsBfresFile(bfresFilePath))
            {
                return;
            }

            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };

            foreach (var anim in this.animations)
            {
                if (anim.AnimKind != AnimKind.MaterialAnim)
                {
                    continue;
                }

                bool success = hio.ReloadMaterialAnimation(anim, fileData);
                if (!success)
                {
                    this.WriteErrorLog(string.Format("ReloadMaterialAnimation()失敗: {0}", fileData.FileName));
                    continue;
                }

                this.WriteInfoLog(string.Format("リロードマテリアルアニメ: {0}[key = {1}, resFileKey = {2}]",
                    fileData.FileName, anim.Key, anim.ResFileKey));
            }
        }
    }
}
