﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Forms;
using System.Threading;
using System.Reflection;
using NintendoWare.G3d.Edit;
using System.Diagnostics;
using G3dHioTestTool;

namespace G3dHioTestTool
{
    public partial class MainForm : Form
    {
        private void CafeConnectedMenuItem_Click(object sender, EventArgs e)
        {
            if (this.hio.IsConnected)
            {
                WriteInfoLog("既に接続済み");
                return;
            }

            lock (this.syncObj)
            {
                if (!this.hio.IsCafeTargeted)
                {
                    this.hio.ChangeCommDevice(true);
                }

                if (this.hio.Connect())
                {
                    WriteInfoLog("Cafe接続成功");
                }
                else
                {
                    WriteErrorLog("Cafe接続失敗");
                }
            }
        }

        private void CloseConnectionMenuItem_Click(object sender, EventArgs e)
        {
            this.UnloadModel();
            this.hio.Disconnect();
            this.WriteInfoLog("切断");
        }

        private void PCConnectedMenuItem_Click(object sender, EventArgs e)
        {
            if (this.hio.IsConnected)
            {
                WriteInfoLog("既に接続済み");
                return;
            }

            lock (this.syncObj)
            {
                this.hio.ChangeCommDevice(false);
                if (this.hio.Connect())
                {
                    WriteInfoLog("PC接続成功");
                }
                else
                {
                    WriteErrorLog("PC接続失敗");
                }
            }
        }

        private void MainForm_ChangeConnection(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "IsConnected")
            {
                var instance = this.hio;
                if (instance.IsConnected)
                {
                    ConnectionMenuItem.Enabled = false;
                    CloseConnectionMenuItem.Enabled = true;
                }
                else
                {
                    ConnectionMenuItem.Enabled = true;
                    CloseConnectionMenuItem.Enabled = false;
                }
            }
            else if (e.PropertyName == "IsAttachCommandReceived")
            {
                var instance = this.hio;

                if (instance.RecvInfo.IsAttachModelReceived)
                {
                    //Log += "AttachedKey : " + instance.AttachedKey.ToString() + System.Environment.NewLine;
                }
            }
        }

        private void LoadModelMenuItem_Click(object sender, EventArgs e)
        {
            if (!this.hio.IsConnected)
            {
                this.WriteErrorLog("接続されていません");
                return;
            }

            if (!System.IO.File.Exists(this.dummyFileData.FileName))
            {
                this.WriteErrorLog(string.Format("{0}が存在しません", this.dummyFileData.FileName));
                return;
            }

            LoadModelFile(this.GetFilePathFromTextBox());
        }

        private Model LoadModelFile(string bfresFilePath)
        {
            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };
            Model model = new Model() { Key = loadModelKey };
            ++loadModelKey;
            this.models.Add(model);
            bool success = this.hio.LoadModel(model, fileData);
            if (!success)
            {
                this.WriteErrorLog(string.Format("LoadModel()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("ロードモデルコマンド送信: {0}", fileData.FileName));
            return model;
        }

        private SkeltalAnimation LoadSkeltalAnimFile(string bfresFilePath)
        {
            FileData fileData = new FileData() { FileName = bfresFilePath, Alignment = 8192 };
            SkeltalAnimation anim = new SkeltalAnimation();
            anim.Key = loadSkeltalAnimKey;
            ++loadSkeltalAnimKey;
            this.skeltalAnims.Add(anim);
            bool success = hio.LoadSkeletalAnimation(anim, fileData, hostModelName);
            if (!success)
            {
                this.WriteErrorLog(string.Format("LoadSkeletalAnimation()失敗: {0}", fileData.FileName));
                return null;
            }

            this.WriteInfoLog(string.Format("ロードスケルタルアニメ: {0}", fileData.FileName));
            return anim;
        }

        private void LoadSkeltalAnimMenuItem_Click(object sender, EventArgs e)
        {
            LoadSkeltalAnimFile(this.GetFilePathFromTextBox());
        }

        private void FrameTrackBar_Scroll(object sender, EventArgs e)
        {
            TrackBar trackBar = sender as TrackBar;
            this.hio.StopFrameCtrl(trackBar.Value);
        }

        private void BindSkeltalAnimMenuItem_Click(object sender, EventArgs e)
        {
            SkeltalAnimation anim = this.skeltalAnims.FirstOrDefault(x => x.Key > 0);
            if (anim == null)
            {
                this.WriteErrorLog("アニメーションが読み込まれていません");
                return;
            }

            foreach (Model model in this.models)
            {
                if (model.Key == 0)
                {
                    continue;
                }

                bool success = hio.BindAnimations(model, new SkeltalAnimation[] { anim });
                if (!success)
                {
                    this.WriteErrorLog(string.Format("BindAnimations()失敗: {0}", this.dummyFileData.FileName));
                    return;
                }

                this.WriteInfoLog(string.Format("バインドスケルタルアニメ [{0}]", model.Key));
            }
        }

        private void RetargetTestModelLoadMenuItem_Click(object sender, EventArgs e)
        {
            if (!this.hio.IsConnected)
            {
                this.WriteErrorLog("接続されていません");
                return;
            }

            {
                string hostFilePath = System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), "RetargetHostModel.bfres");
                Model hostModel = LoadModelFile(hostFilePath);
                Debug.Assert(hostModel != null);
            }

            {
                string clientFilePath = System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), "RetargetClientModel.bfres");
                Model clientModel = LoadModelFile(clientFilePath);
                Debug.Assert(clientModel != null);
            }
        }

        private void ReloadModelMenuItem_Click(object sender, EventArgs e)
        {
            if (!this.hio.IsConnected)
            {
                this.WriteErrorLog("接続されていません");
                return;
            }

            foreach (Model model in this.models)
            {
                bool success = this.hio.ReloadModel(model, this.dummyFileData, null, null);
                if (!success)
                {
                    this.WriteErrorLog(string.Format("ReloadModel() 失敗: {0}", this.dummyFileData.FileName));
                    return;
                }

                this.WriteInfoLog(string.Format("リロードモデルコマンド送信: {0}", this.dummyFileData.FileName));
            }
        }

        private void UnloadModel()
        {
            foreach (Model model in this.models)
            {
                bool result = this.hio.UnloadModel(model);
                if (!result)
                {
                    this.WriteErrorLog("UnloadModel() 失敗");
                    return;
                }

                model.ResModelKey = 0;
                model.ResFileKey = 0;
                model.ModelObjKey = 0;

                this.WriteInfoLog(string.Format("アンロードモデルコマンド送信: {0}", model.Key));
            }

            this.models.Clear();
        }

        private void UnloadMenuItem_Click(object sender, EventArgs e)
        {
            UnloadModel();
        }

        private void SendBfresPathTextBox_TextChanged(object sender, EventArgs e)
        {
            this.dummyFileData.FileName = this.GetFilePathFromTextBox();
        }

        private string GetFilePathFromTextBox()
        {
            if (this.IsAbsolutePath(this.SendBfresPathTextBox.Text))
            {
                return this.SendBfresPathTextBox.Text;
            }
            else
            {
                return System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), this.SendBfresPathTextBox.Text);
            }
        }

        private bool IsAbsolutePath(string path)
        {
            if (System.IO.Path.IsPathRooted(path))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        private void DummyUserCommandMenuItem_Click(object sender, EventArgs e)
        {
            string scriptText = "Apply Blur\nApply Glow";
            this.hio.SelectTargetBones(dummyAttachModel, new int[3] { 3, 1, 2 });
            this.hio.SelectTargetShapes(dummyAttachModel, new int[2] { 1, 0 });
            this.hio.SelectTargetMaterials(dummyAttachModel, new int[2] { 1, 0 });

            try
            {
                this.hio.ExecuteModelUserScript(dummyAttachModel, scriptText);
            }
            catch (Exception exception)
            {
                this.WriteErrorLog(exception.Message);
                return;
            }

            this.WriteInfoLog(string.Format("ユーザコマンド送信: \nscriptText = \n{0}\n", scriptText));
        }

        private void SelectMaterialMenuItem_Click(object sender, EventArgs e)
        {
            this.hio.EditSelectTarget(SelectTargetKind.Material);
            this.WriteInfoLog("マテリアル選択");
        }

        private void SelectShapeMenuItem_Click(object sender, EventArgs e)
        {
            this.hio.EditSelectTarget(SelectTargetKind.Shape);
            this.WriteInfoLog("シェイプ選択");
        }
    }
}
