﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <vector>
#include <nnt.h>

#include "testTime_TimeZoneTestUtil.h"

using namespace nn::time;

namespace
{

    const char* TestLocationName = "Europe/London";
    const TimeZoneType StartTimeZone("GMT", false, 0);

    // tz database のルールに合わせてテストケースを作る
    const TestRuleType TestRule[] =
    {
        //
        // 月日が昇順になるよう定義すること！
        // 年は順番に関係なく、そのルールが適応される西暦を指定してください。
        //

        {
            // GMT -> BST
            // 3月最終日曜日の 00:59:59(GMT) の1秒後が 02:00:00(BST) に
            2000, 2100,
            3, GettingWeekType::Last, DayOfWeek_Sunday, 1, // 1時に saveHour が1に
            TimeZoneType("BST", true, 1)
        },
        {
            // BST -> GMT
            // 10月最終日曜日の 01:59:59(BST) の1秒後が 01:00:00(GMT) に
            2000, 2100,
            10, GettingWeekType::Last, DayOfWeek_Sunday, 2, // 2時に saveHourが0に
            TimeZoneType("GMT", false, 0)
        }
    };

    std::vector<std::pair<TestCalendarType, TimeZoneType>> g_TestCase;


    // NX Addon 0.5.0 では 2010年～2050年 までテストしておく

    // Europe/London
    // 3月の最終日曜日に BST 開始. 1時台スキップ
    // 10月最終日曜日に BST 終了. 1時台重複
    const TestCalendarType TestCalendarAnswer[] = // 夏時間の開始・終了、の順にカレンダーを決め打ち
    {
        TestCalendarType(2000,  3, 26, 1, 0, 0),
        TestCalendarType(2000, 10, 29, 2, 0, 0),

        TestCalendarType(2001,  3, 25, 1, 0, 0),
        TestCalendarType(2001, 10, 28, 2, 0, 0),

        TestCalendarType(2002,  3, 31, 1, 0, 0),
        TestCalendarType(2002, 10, 27, 2, 0, 0),

        TestCalendarType(2003,  3, 30, 1, 0, 0),
        TestCalendarType(2003, 10, 26, 2, 0, 0),

        TestCalendarType(2004,  3, 28, 1, 0, 0),
        TestCalendarType(2004, 10, 31, 2, 0, 0),

        TestCalendarType(2005,  3, 27, 1, 0, 0),
        TestCalendarType(2005, 10, 30, 2, 0, 0),

        TestCalendarType(2006,  3, 26, 1, 0, 0),
        TestCalendarType(2006, 10, 29, 2, 0, 0),

        TestCalendarType(2007,  3, 25, 1, 0, 0),
        TestCalendarType(2007, 10, 28, 2, 0, 0),

        TestCalendarType(2008,  3, 30, 1, 0, 0),
        TestCalendarType(2008, 10, 26, 2, 0, 0),

        TestCalendarType(2009,  3, 29, 1, 0, 0),
        TestCalendarType(2009, 10, 25, 2, 0, 0),

        TestCalendarType(2010,  3, 28, 1, 0, 0),
        TestCalendarType(2010, 10, 31, 2, 0, 0),

        TestCalendarType(2011,  3, 27, 1, 0, 0),
        TestCalendarType(2011, 10, 30, 2, 0, 0),

        TestCalendarType(2012,  3, 25, 1, 0, 0),
        TestCalendarType(2012, 10, 28, 2, 0, 0),

        TestCalendarType(2013,  3, 31, 1, 0, 0),
        TestCalendarType(2013, 10, 27, 2, 0, 0),

        TestCalendarType(2014,  3, 30, 1, 0, 0),
        TestCalendarType(2014, 10, 26, 2, 0, 0),

        TestCalendarType(2015,  3, 29, 1, 0, 0),
        TestCalendarType(2015, 10, 25, 2, 0, 0),

        TestCalendarType(2016,  3, 27, 1, 0, 0),
        TestCalendarType(2016, 10, 30, 2, 0, 0),

        TestCalendarType(2017,  3, 26, 1, 0, 0),
        TestCalendarType(2017, 10, 29, 2, 0, 0),

        TestCalendarType(2018,  3, 25, 1, 0, 0),
        TestCalendarType(2018, 10, 28, 2, 0, 0),

        TestCalendarType(2019,  3, 31, 1, 0, 0),
        TestCalendarType(2019, 10, 27, 2, 0, 0),

        TestCalendarType(2020,  3, 29, 1, 0, 0),
        TestCalendarType(2020, 10, 25, 2, 0, 0),

        TestCalendarType(2021,  3, 28, 1, 0, 0),
        TestCalendarType(2021, 10, 31, 2, 0, 0),

        TestCalendarType(2022,  3, 27, 1, 0, 0),
        TestCalendarType(2022, 10, 30, 2, 0, 0),

        TestCalendarType(2023,  3, 26, 1, 0, 0),
        TestCalendarType(2023, 10, 29, 2, 0, 0),

        TestCalendarType(2024,  3, 31, 1, 0, 0),
        TestCalendarType(2024, 10, 27, 2, 0, 0),

        TestCalendarType(2025,  3, 30, 1, 0, 0),
        TestCalendarType(2025, 10, 26, 2, 0, 0),

        TestCalendarType(2026,  3, 29, 1, 0, 0),
        TestCalendarType(2026, 10, 25, 2, 0, 0),

        TestCalendarType(2027,  3, 28, 1, 0, 0),
        TestCalendarType(2027, 10, 31, 2, 0, 0),

        TestCalendarType(2028,  3, 26, 1, 0, 0),
        TestCalendarType(2028, 10, 29, 2, 0, 0),

        TestCalendarType(2029,  3, 25, 1, 0, 0),
        TestCalendarType(2029, 10, 28, 2, 0, 0),

        TestCalendarType(2030,  3, 31, 1, 0, 0),
        TestCalendarType(2030, 10, 27, 2, 0, 0),

        TestCalendarType(2031,  3, 30, 1, 0, 0),
        TestCalendarType(2031, 10, 26, 2, 0, 0),

        TestCalendarType(2032,  3, 28, 1, 0, 0),
        TestCalendarType(2032, 10, 31, 2, 0, 0),

        TestCalendarType(2033,  3, 27, 1, 0, 0),
        TestCalendarType(2033, 10, 30, 2, 0, 0),

        TestCalendarType(2034,  3, 26, 1, 0, 0),
        TestCalendarType(2034, 10, 29, 2, 0, 0),

        TestCalendarType(2035,  3, 25, 1, 0, 0),
        TestCalendarType(2035, 10, 28, 2, 0, 0),

        TestCalendarType(2036,  3, 30, 1, 0, 0),
        TestCalendarType(2036, 10, 26, 2, 0, 0),

        TestCalendarType(2037,  3, 29, 1, 0, 0),
        TestCalendarType(2037, 10, 25, 2, 0, 0),

        TestCalendarType(2038,  3, 28, 1, 0, 0),
        TestCalendarType(2038, 10, 31, 2, 0, 0),

        TestCalendarType(2039,  3, 27, 1, 0, 0),
        TestCalendarType(2039, 10, 30, 2, 0, 0),

        TestCalendarType(2040,  3, 25, 1, 0, 0),
        TestCalendarType(2040, 10, 28, 2, 0, 0),

        TestCalendarType(2041,  3, 31, 1, 0, 0),
        TestCalendarType(2041, 10, 27, 2, 0, 0),

        TestCalendarType(2042,  3, 30, 1, 0, 0),
        TestCalendarType(2042, 10, 26, 2, 0, 0),

        TestCalendarType(2043,  3, 29, 1, 0, 0),
        TestCalendarType(2043, 10, 25, 2, 0, 0),

        TestCalendarType(2044,  3, 27, 1, 0, 0),
        TestCalendarType(2044, 10, 30, 2, 0, 0),

        TestCalendarType(2045,  3, 26, 1, 0, 0),
        TestCalendarType(2045, 10, 29, 2, 0, 0),

        TestCalendarType(2046,  3, 25, 1, 0, 0),
        TestCalendarType(2046, 10, 28, 2, 0, 0),

        TestCalendarType(2047,  3, 31, 1, 0, 0),
        TestCalendarType(2047, 10, 27, 2, 0, 0),

        TestCalendarType(2048,  3, 29, 1, 0, 0),
        TestCalendarType(2048, 10, 25, 2, 0, 0),

        TestCalendarType(2049,  3, 28, 1, 0, 0),
        TestCalendarType(2049, 10, 31, 2, 0, 0),

        TestCalendarType(2050,  3, 27, 1, 0, 0),
        TestCalendarType(2050, 10, 30, 2, 0, 0),
    };
}

TEST(TimeZone_EuropeLondon, GenerateTestCase)
{
    for(int y = TestBeginYear ; y <= TestEndYear ; ++y)
    {
        for(auto& rule : TestRule)
        {
            if(rule.yearBegin <= y && y <= rule.yearEnd)
            {
                int day = GetDayOfSpecific(y, rule.month, rule.weekType, rule.dayOfWeek);
                g_TestCase.push_back(
                    std::make_pair(
                        TestCalendarType(y, rule.month, day, rule.startHour, 0, 0),
                        rule.timeZoneType)
                    );
            }
        }
    }

    // TestCalendarAnswer を手動で2050年までWindowsのカレンダーを目視で作ったので一致するか確認
    int i = 0;
    for(auto& answer : TestCalendarAnswer)
    {
        EXPECT_EQ(answer.c, g_TestCase[ i ].first.c) << answer.ToString() << " " << g_TestCase[ i ].first.ToString();
        ++i;
    }
}

TEST(TimeZone_EuropeLondon, ExecutionTestCase)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );
    RuleLoader rule(TestLocationName);
    NNT_ASSERT_RESULT_SUCCESS(rule.Load());

    const int64_t testSkipSeconds = nn::TimeSpan::FromHours(18).GetSeconds() + 1;

    TestCalendarType begin(TestBeginYear, 1, 1, 0, 0, 0);

    TimeZoneType currentTimeZone = StartTimeZone;

    for(auto &test : g_TestCase)
    {
        const auto& calendar = test.first;
        const auto& timeZone = test.second;

        if(currentTimeZone.saveHour == timeZone.saveHour)
        {
            // 時刻はずれない. 表示の名称とかだけが変わる、など。

            // 切り替わり直前まで
            TestCalendarType end = calendar;
            end.Shift(-1);
            NNT_TIME_LOOPBACK_1(rule, begin, end, currentTimeZone, testSkipSeconds);

            // 切り替わり後からテスト続行
            begin = calendar;
        }
        else if(currentTimeZone.saveHour == 0 && timeZone.saveHour != 0) // 時間が進むとき
        {
            // スキップ直前まで
            TestCalendarType end = calendar;
            end.Shift(-1);
            NNT_TIME_LOOPBACK_1(rule, begin, end, currentTimeZone, testSkipSeconds);

            begin = calendar; // スキップ開始時刻
            end = begin;
            end.Shift(nn::TimeSpan::FromHours(timeZone.saveHour).GetSeconds() - 1); // スキップ終了の1秒前
            NNT_TIME_LOOPBACK_0(rule, begin, end, 1);

            // 次のテストはスキップ終わってから
            begin = calendar;
            begin.Shift(nn::TimeSpan::FromHours(timeZone.saveHour).GetSeconds());

            // スキップ終わってから少しは細かいskipでテストしておく
            end = begin;
            end.Shift(60 * 30);
            NNT_TIME_LOOPBACK_1(rule, begin, end, timeZone, 1);
        }
        else if(currentTimeZone.saveHour != 0 && timeZone.saveHour == 0) // 時間が巻き戻るとき
        {
            // 重複直前まで
            TestCalendarType end = calendar;
            end.Shift(nn::TimeSpan::FromHours(-currentTimeZone.saveHour).GetSeconds() - 1); //重複1秒前
            NNT_TIME_LOOPBACK_1(rule, begin, end, currentTimeZone, testSkipSeconds);

            begin = calendar; // 重複終了時刻
            begin.Shift(nn::TimeSpan::FromHours(-currentTimeZone.saveHour).GetSeconds()); // 重複開始時刻
            end = calendar;
            end.Shift(-1); // 重複終了の1秒前
            NNT_TIME_LOOPBACK_2(
                rule, begin, end,
                currentTimeZone, timeZone,
                1);

            // 次のテストは重複終わってから
            begin = calendar;
            begin.Shift(nn::TimeSpan::FromHours(currentTimeZone.saveHour).GetSeconds());

            // 重複終わってから少しは細かいskipでテストしておく
            end = begin;
            end.Shift(nn::TimeSpan::FromMinutes(30).GetSeconds());
            NNT_TIME_LOOPBACK_1(rule, begin, end, timeZone, 1);
        }
        currentTimeZone = timeZone;

    }

    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}
