﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClockPrivilegeApi.h>// InitializeForMenu
#include <nn/time/detail/time_TimeZonePrivateApi.h>

#include "testTime_TimeZoneTestUtil.h"

#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>

using namespace nn::time;

// テスト用に作った API のテスト
TEST(TimeZone_Basic, TestOfTestApi)
{

    EXPECT_EQ( 6, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Sunday));
    EXPECT_EQ(13, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Sunday));
    EXPECT_EQ(27, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Sunday));
    EXPECT_EQ(DayOfWeek_Sunday, GetDayOfWeek(2016, 3, 6));
    EXPECT_EQ(DayOfWeek_Sunday, GetDayOfWeek(2016, 3, 13));
    EXPECT_EQ(DayOfWeek_Sunday, GetDayOfWeek(2016, 3, 27));

    EXPECT_EQ( 7, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Monday));
    EXPECT_EQ(14, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Monday));
    EXPECT_EQ(28, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Monday));
    EXPECT_EQ(DayOfWeek_Monday, GetDayOfWeek(2016, 3, 7));
    EXPECT_EQ(DayOfWeek_Monday, GetDayOfWeek(2016, 3, 14));
    EXPECT_EQ(DayOfWeek_Monday, GetDayOfWeek(2016, 3, 28));

    EXPECT_EQ( 1, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Tuesday));
    EXPECT_EQ( 8, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Tuesday));
    EXPECT_EQ(29, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Tuesday));
    EXPECT_EQ(DayOfWeek_Tuesday, GetDayOfWeek(2016, 3, 1));
    EXPECT_EQ(DayOfWeek_Tuesday, GetDayOfWeek(2016, 3, 8));
    EXPECT_EQ(DayOfWeek_Tuesday, GetDayOfWeek(2016, 3, 29));

    EXPECT_EQ( 2, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Wednesday));
    EXPECT_EQ( 9, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Wednesday));
    EXPECT_EQ(30, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Wednesday));
    EXPECT_EQ(DayOfWeek_Wednesday, GetDayOfWeek(2016, 3, 2));
    EXPECT_EQ(DayOfWeek_Wednesday, GetDayOfWeek(2016, 3, 9));
    EXPECT_EQ(DayOfWeek_Wednesday, GetDayOfWeek(2016, 3, 30));

    EXPECT_EQ( 3, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Thursday));
    EXPECT_EQ(10, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Thursday));
    EXPECT_EQ(31, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Thursday));
    EXPECT_EQ(DayOfWeek_Thursday, GetDayOfWeek(2016, 3, 3));
    EXPECT_EQ(DayOfWeek_Thursday, GetDayOfWeek(2016, 3, 10));
    EXPECT_EQ(DayOfWeek_Thursday, GetDayOfWeek(2016, 3, 31));

    EXPECT_EQ( 4, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Friday));
    EXPECT_EQ(11, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Friday));
    EXPECT_EQ(25, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Friday));
    EXPECT_EQ(DayOfWeek_Friday, GetDayOfWeek(2016, 3, 4));
    EXPECT_EQ(DayOfWeek_Friday, GetDayOfWeek(2016, 3, 11));
    EXPECT_EQ(DayOfWeek_Friday, GetDayOfWeek(2016, 3, 25));

    EXPECT_EQ( 5, GetDayOfSpecific(2016, 3, GettingWeekType::First, DayOfWeek_Saturday));
    EXPECT_EQ(12, GetDayOfSpecific(2016, 3, GettingWeekType::Second, DayOfWeek_Saturday));
    EXPECT_EQ(26, GetDayOfSpecific(2016, 3, GettingWeekType::Last, DayOfWeek_Saturday));
    EXPECT_EQ(DayOfWeek_Saturday, GetDayOfWeek(2016, 3, 5));
    EXPECT_EQ(DayOfWeek_Saturday, GetDayOfWeek(2016, 3, 12));
    EXPECT_EQ(DayOfWeek_Saturday, GetDayOfWeek(2016, 3, 26));
}

TEST(TimeZone_Basic, GetDeviceLocationName)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );
    LocationName name;
    GetDeviceLocationName(&name);
    ASSERT_STRNE(name._value, ""); // 空ではないことだけ確認しておく
    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}

TEST(TimeZone_Basic, GetDeviceLocationNameAndUpdatedTime)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );

    LocationName name;
    SteadyClockTimePoint updatedTime;

    nn::time::detail::GetDeviceLocationNameAndUpdatedTime(&name, &updatedTime);

    ASSERT_STRNE(name._value, ""); // 空ではないことだけ確認しておく

    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}

TEST(TimeZone_Basic, UsingDefaultRule)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );

    TestCalendarType begin(TestBeginYear, 1, 1, 0, 0, 0);
    const TestCalendarType end(TestEndYear, 12, 31, 23, 59, 59);

    do
    {
        // 何のルールがセットされているか分からないので TimeZone のチェックはしない

        int count;
        PosixTime posix[2];
        NNT_ASSERT_RESULT_SUCCESS(ToPosixTime(&count, posix, 2, begin.c));

        if(count > 0)
        {
            CalendarTime c;
            CalendarAdditionalInfo a;
            NNT_ASSERT_RESULT_SUCCESS(ToCalendarTime(&c, &a, posix[0]));
            EXPECT_EQ(begin.c, c);
        }

        if(count > 1)
        {
            CalendarTime c;
            CalendarAdditionalInfo a;
            NNT_ASSERT_RESULT_SUCCESS(ToCalendarTime(&c, &a, posix[1]));
            EXPECT_EQ(begin.c, c);
        }

        begin.Shift(nn::TimeSpan::FromHours(12).GetSeconds() + 5); // seconds も変わるよう適当にずらす

        if(begin > end)
        {
            break;
        }
    }
    while(NN_STATIC_CONDITION(true));

    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}

TEST(TimeZone_Basic, InputPosixMinMax)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );

    const bool print = false;

    RuleLoader rule("UTC");
    NNT_ASSERT_RESULT_SUCCESS(rule.Load());

    {
        CalendarTime c = {};
        c.year = 1999;
        c.month = 12;
        c.day = 31;
        c.hour = 0;
        c.minute = 0;
        c.second = 0;

        int count;
        PosixTime posix[2];
        NNT_ASSERT_RESULT_SUCCESS(ToPosixTime(&count, posix, 2, c, rule()));
        EXPECT_EQ(count, 1);

        if(NN_STATIC_CONDITION(print))
        {
            NN_LOG("(%04d/%02d/%02d %02d:%02d:%02d) posix:%lld\n",
                c.year, c.month, c.day, c.hour, c.minute, c.second,
                posix[0].value);
        }

        EXPECT_EQ(posix[0], InputPosixTimeMin);
    }
    {
        CalendarTime c = {};
        c.year = 2010;
        c.month = 1;
        c.day = 1;

        int count;
        PosixTime posix[2];
        NNT_ASSERT_RESULT_SUCCESS(ToPosixTime(&count, posix, 2, c, rule()));
        EXPECT_EQ(count, 1);

        if(NN_STATIC_CONDITION(print))
        {
            NN_LOG("(%04d/%02d/%02d %02d:%02d:%02d) posix:%lld\n",
                c.year, c.month, c.day, c.hour, c.minute, c.second,
                posix[0].value);
        }
    }
    {
        CalendarTime c = {};
        c.year = 2050;
        c.month = 1;
        c.day = 1;

        int count;
        PosixTime posix[2];
        NNT_ASSERT_RESULT_SUCCESS(ToPosixTime(&count, posix, 2, c, rule()));
        EXPECT_EQ(count, 1);

        if(NN_STATIC_CONDITION(print))
        {
            NN_LOG("(%04d/%02d/%02d %02d:%02d:%02d) posix:%lld\n",
                c.year, c.month, c.day, c.hour, c.minute, c.second,
                posix[0].value);
        }
    }
    {
        CalendarTime c = {};
        c.year = 2100;
        c.month = 1;
        c.day = 1;

        int count;
        PosixTime posix[2];
        NNT_ASSERT_RESULT_SUCCESS(ToPosixTime(&count, posix, 2, c, rule()));
        EXPECT_EQ(count, 1);

        if(NN_STATIC_CONDITION(print))
        {
            NN_LOG("(%04d/%02d/%02d %02d:%02d:%02d) posix:%lld\n",
                c.year, c.month, c.day, c.hour, c.minute, c.second,
                posix[0].value);
        }

        EXPECT_EQ(posix[0], InputPosixTimeMax);
    }
    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}

TEST(TimeZone_Basic, NotFoundDate)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );

    const char* TestLocationList[] =
    {
        "UTC",
        "Asia/Tokyo",
        "Europe/London",
        "America/Los_Angeles",
        "America/New_York"
    };

    for(auto& name : TestLocationList)
    {
        RuleLoader rule(name);
        NNT_ASSERT_RESULT_SUCCESS(rule.Load()) << name;

        for(int y = TestBeginYear ; y <= TestEndYear ; ++y)
        {
            if(!IsLeapYear(y))
            {
                NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 29, 0, 0, 0));
                NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 29, 12, 34, 56));
            }
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 30, 0, 0, 0));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 30, 12, 34, 56));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 30, 23, 59, 59));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 31, 0, 0, 0));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 31, 12, 34, 56));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 2, 31, 23, 59, 59));

            NNT_TIME_NOT_FOUND(TestCalendarType(y, 4, 31, 0, 0, 0));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 4, 31, 12, 34, 56));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 4, 31, 23, 59, 59));

            NNT_TIME_NOT_FOUND(TestCalendarType(y, 6, 31, 0, 0, 0));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 6, 31, 12, 34, 56));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 6, 31, 23, 59, 59));

            NNT_TIME_NOT_FOUND(TestCalendarType(y, 9, 31, 0, 0, 0));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 9, 31, 12, 34, 56));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 9, 31, 23, 59, 59));

            NNT_TIME_NOT_FOUND(TestCalendarType(y, 11, 31, 0, 0, 0));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 11, 31, 12, 34, 56));
            NNT_TIME_NOT_FOUND(TestCalendarType(y, 11, 31, 23, 59, 59));
        }
    }

    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}

TEST(TimeZone_Basic, Utc)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );
    RuleLoader rule("UTC");
    NNT_EXPECT_RESULT_SUCCESS(rule.Load());


    NNT_TIME_LOOPBACK_1(rule,
        TestCalendarType(TestBeginYear, 1, 1, 0, 0, 0),
        TestCalendarType(TestEndYear, 12, 31, 23, 59, 59),
        TimeZoneType("UTC", false), 60 * 60 * 24 + 1);


    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}
