﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

#include <cstdlib>
#include <cstring>

#include <nn/tics/tics_Api.h>

#include "CrossBar.h"
#include "HotBridgeAPI.h"
#include "BridgeAPIInternal.h"

#include "../../Common/testHtc_BridgeAddressManager.h"
#include "../testHtc_CrossBarRunThread.h"
#include "testHtc_EchoClientPlug.h"

extern "C" void nninitStartup()
{
}

// プロトタイプ宣言
namespace {
    class StackManager;
    class ChannelAgent;

    static int OnTakeover( void* ctx, int operationResult, void* operationResources );
    static int OnAddChannel( void* ctx, int operationResult, void* operationResources );
    static int OnRemoveChannel( void* ctx, int operationResult, void* operationResources );
    static void ChannelThread(void* argument);
}

namespace {
    // コンフィギュレーション
    bool g_EnableRemoveChannel = false; // true なら送受信後に RemoveChannel() を呼ぶ

    const char* g_BridgeName = "local_bridge";
    const char* g_ChannelNameBase = "TestChannel";

    const unsigned int g_RandomSeed = 0;
    const int64_t g_MaxIntervalMilliSeconds = 100;

    const size_t g_MaxTestDataSize = 10 * 1024 * 1024;

    const int g_ChannelCount = 10;
    const int g_TimeOutMilliSeconds = 5000;

    // スレッドアンセーフな AddChannel(), RemoveChannel() を呼びだす際に使う Mutex
    nn::os::MutexType g_ConfigChannelMutex;

    typedef struct Context
    {
        nn::os::EventType* pEvent;
    } Context;

    // テストデータや、通信間隔をランダムに生成するクラス
    class RandomDataProvider
    {
    public:
        explicit RandomDataProvider( unsigned int seed )
        {
            srand(seed);
        }

        int64_t GetIntervalMilliSeconds()
        {
            return 1 + rand() % g_MaxIntervalMilliSeconds;
        }

        size_t GetTestDataSize()
        {
            return 1 + rand() % g_MaxTestDataSize;
        }

        void GetTestData( uint8_t* outTestData, size_t length )
        {
            for( size_t i = 0; i < length; i++ )
            {
                outTestData[i] = static_cast<uint8_t>( rand() );
            }
        }
    };

    class StackManager
    {
    public:
        void Initialize()
        {
            NN_ASSERT( nn::os::MemoryBlockUnitSize >= nn::os::StackRegionAlignment );

            uintptr_t address;
            nn::Result result = nn::os::AllocateMemoryBlock( &address, nn::os::MemoryBlockUnitSize );
            ASSERT_TRUE( result.IsSuccess() );

            m_pMemoryBlock = reinterpret_cast<uint8_t*>( address );
            m_AllocateCount = 0;
        }

        // nn::os::StackRegionAlignment バイトの領域を確保し、そのポインタを返す
        uintptr_t Allocate()
        {
            uintptr_t address = reinterpret_cast<uintptr_t>( m_pMemoryBlock + m_AllocateCount * nn::os::StackRegionAlignment );

            if( ++m_AllocateCount >= nn::os::MemoryBlockUnitSize / nn::os::StackRegionAlignment )
            {
                this->Initialize();
            }

            return address;
        }

    private:
        uint8_t* m_pMemoryBlock;
        int m_AllocateCount;
    };

    class ChannelAgent
    {
    public:
        ChannelAgent(::tics::portability::stl::string channelName, ::tics::BridgeHandle bridgeHandle, ::tics::portability::stl::string bridgeIpAddr)
            : m_ChannelName( channelName ), m_BridgeHandle( bridgeHandle )
        {
            // セッション名は BridgeIpAddr:ChannelName
            for( int i = 0; i < g_ChannelCount; i++ )
            {
                m_SessionId.clear();
                m_SessionId += bridgeIpAddr;
                m_SessionId += ":";
                m_SessionId += m_ChannelName;
            }
        }

        void Initialize(uint8_t* testData, size_t testDataLength, StackManager *pStackManager)
        {
            // Event の初期化
            nn::os::InitializeEvent( &m_AddChannelEvent, false, nn::os::EventClearMode_ManualClear );
            nn::os::InitializeEvent( &m_RemoveChannelEvent, false, nn::os::EventClearMode_ManualClear );
            nn::os::InitializeEvent( &m_PlugEvent, false, nn::os::EventClearMode_AutoClear );

            // Echo Client Plug の作成
            m_pEchoClientPlug = new ::EchoClientPlug( &m_PlugEvent );
            m_pEchoClientPlug->SetSendData(testData, testDataLength);

            // スレッドスタックの確保
            size_t stackSize = 4096;
            void* pStack = reinterpret_cast<void*>( pStackManager->Allocate() );

            // スレッドオブジェクトの作成
            nn::Result result = nn::os::CreateThread(&m_Thread, ChannelThread, static_cast<void*>(this), pStack, stackSize, nn::os::DefaultThreadPriority);
            NN_ABORT_UNLESS( result.IsSuccess(), "nn::os::AllocateMemoryBlock() failed.\n" );
        }

        const void* GetRecvData()
        {
            return m_pEchoClientPlug->GetRecvData();
        }

        void Run()
        {
            int intResult;

            // AddChannel
            ::Context addChannelContext;
            addChannelContext.pEvent = &m_AddChannelEvent;

            nn::os::LockMutex(&g_ConfigChannelMutex);

            intResult = ::tics::BridgeChannelAdd(m_BridgeHandle, m_ChannelName.c_str(), 0, &addChannelContext, ::OnAddChannel);
            ASSERT_EQ( intResult, 0 );
            nn::os::WaitEvent( &m_AddChannelEvent );

            nn::os::UnlockMutex(&g_ConfigChannelMutex);

            // EchoClientPlug を登録
            intResult = ::tics::GetBridgeCrossBar( m_BridgeHandle )->RegisterForSessionStart(m_SessionId, m_pEchoClientPlug);
            ASSERT_EQ( intResult, 0 );

            // エコーサーバとの送受信の終了を待つ
            nn::os::WaitEvent( &m_PlugEvent );

            // RemoveChannel
            if( g_EnableRemoveChannel )
            {
                ::Context removeChannelContext;
                removeChannelContext.pEvent = &m_RemoveChannelEvent;

                nn::os::LockMutex(&g_ConfigChannelMutex);

                intResult = ::tics::BridgeChannelDelete(m_BridgeHandle, m_ChannelName.c_str(), &removeChannelContext, ::OnRemoveChannel);
                ASSERT_EQ( intResult, 0 );
                nn::os::WaitEvent( &m_RemoveChannelEvent );

                nn::os::UnlockMutex(&g_ConfigChannelMutex);
            }
        }

        void StartThread()
        {
            nn::os::StartThread(&m_Thread);
        }

        void WaitThread()
        {
            nn::os::WaitThread(&m_Thread);
        }
    private:
        ::tics::portability::stl::string m_ChannelName;
        ::tics::BridgeHandle m_BridgeHandle;

        ::tics::portability::stl::string m_SessionId;
        ::EchoClientPlug* m_pEchoClientPlug;

        nn::os::ThreadType m_Thread; // AddChannel や送受信を行うスレッド
        nn::os::EventType m_AddChannelEvent;
        nn::os::EventType m_RemoveChannelEvent;
        nn::os::EventType m_PlugEvent;
    };

    static int OnTakeover( void* ctx, int operationResult, void* operationResources )
    {
        NN_UNUSED(operationResult);
        NN_UNUSED(operationResources);

        NN_LOG("--- OnTakeover ---\n");
        nn::os::SignalEvent(static_cast<Context*>(ctx)->pEvent);

        return 0;
    }

    static int OnAddChannel( void* ctx, int operationResult, void* operationResources )
    {
        NN_UNUSED(operationResult);
        NN_UNUSED(operationResources);
        NN_UNUSED(operationResources);

        NN_LOG("--- OnAddChannel ---\n");
        nn::os::SignalEvent(static_cast<Context*>(ctx)->pEvent);

        return 0;
    }

    static int OnRemoveChannel( void* ctx, int operationResult, void* operationResources )
    {
        NN_UNUSED(operationResult);
        NN_UNUSED(operationResources);
        NN_UNUSED(operationResources);

        NN_LOG("--- OnRemoveChannel ---\n");
        nn::os::SignalEvent(static_cast<Context*>(ctx)->pEvent);

        return 0;
    }

    // スレッド関数
    static void ChannelThread(void* argument)
    {
        NN_LOG("--- ChannelThread ---\n");
        static_cast<ChannelAgent*>( argument )->Run();
    }

}

/**
    HTC ドライバライブラリを経由して、
    エコーサーバにアクセスするテスト。
*/
TEST(stressTest, stressTest)
{
    //
    // 初期化
    //
    nn::os::InitializeMutex(&g_ConfigChannelMutex, false, 0);

    // メモリ領域確保
    size_t heapSize = 16 * 1024 * 1024; // 16MB
    nn::Result result = nn::os::SetMemoryHeapSize( heapSize );
    ASSERT_TRUE( result.IsSuccess() );

    StackManager* pStackManager = new StackManager();
    pStackManager->Initialize();

    uintptr_t heapAddress;
    result = nn::os::AllocateMemoryBlock( &heapAddress, heapSize / 2 ); // StackManager が使った分を適当に減らしている
    ASSERT_TRUE( result.IsSuccess() );

    // HTC ライブラリ初期化
    nn::tics::Initialize(heapAddress, heapSize);

    // チャンネル名の決定
    ::tics::portability::stl::string channelNames[g_ChannelCount];
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        channelNames[i].clear();
        channelNames[i] += g_ChannelNameBase;
        channelNames[i] += std::to_string(i);
    }

    //
    // ランダムデータの作成
    //
    RandomDataProvider* pDataProvider = new RandomDataProvider(g_RandomSeed);

    // チャンネルの処理を開始する待ち時間
    nn::TimeSpan intervals[g_ChannelCount];
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        intervals[i] = nn::TimeSpan::FromMilliSeconds(pDataProvider->GetIntervalMilliSeconds());
    }

    // テストデータの長さ
    size_t testDataSize[g_ChannelCount];
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        testDataSize[i] = pDataProvider->GetTestDataSize();
    }

    // テストデータ
    uint8_t* testData[g_ChannelCount];
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        testData[i] = new uint8_t[testDataSize[i]];
        pDataProvider->GetTestData(testData[i], testDataSize[i]);
    }

    //
    // 通信
    //

    // Bridge の IP アドレス取得
    BridgeAddressManager bridgeAddressManager;

    char bridgeIpAddr[20];
    bridgeAddressManager.GetAddress(bridgeIpAddr, sizeof(bridgeIpAddr));

    // Takeover
    nn::os::EventType takeoverEvent;
    nn::os::InitializeEvent( &takeoverEvent, false, nn::os::EventClearMode_AutoClear );

    NN_LOG("--- ::tics::detail::BridgeTakeoverByIP() ---\n");
    ::Context takeoverContext = { &takeoverEvent };
    ::tics::TargetStatusFlags targetFlags = 0;
    targetFlags |= tics::EFlag_MainTargetPowerOn;
    ::tics::BridgeHandle bridgeHandle = ::tics::BridgeTakeoverByIP( bridgeIpAddr, true, targetFlags, g_TimeOutMilliSeconds, &takeoverContext, ::OnTakeover );
    ASSERT_NE( bridgeHandle, nullptr );

    // ChannelAgent の作成
    ::ChannelAgent* pChannelAgents[g_ChannelCount];
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        pChannelAgents[i] = new ChannelAgent(channelNames[i], bridgeHandle, bridgeIpAddr);
        pChannelAgents[i]->Initialize(testData[i], testDataSize[i], pStackManager);
    }

    // CrossBar::Run() を別スレッドで呼び出して、通信開始
    ::tics::CrossBar* pCrossBar = ::tics::GetBridgeCrossBar( bridgeHandle );
    CrossBarRunThread* pCrossBarRunThread = new CrossBarRunThread( pCrossBar );
    NN_LOG("--- CrossBarRunThread::StartThread() ---\n");
    pCrossBarRunThread->StartThread();

    // Takeover 終了を待つ
    NN_LOG("--- Wait to Takeover ---\n");
    nn::os::WaitEvent( &takeoverEvent );

    // あらかじめ設定したタイミングで、チャンネルごとにスレッドを作成
    NN_LOG("--- ChannelAgent::StartThread() ---\n");
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        pChannelAgents[i]->StartThread();
        nn::os::SleepThread( intervals[i] );
    }

    // スレッドの合流
    NN_LOG("--- ChannelAgent::WaitThread() ---\n");
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        pChannelAgents[i]->WaitThread();
    }

    // 通信終了
    pCrossBar->BeginShutdown(::tics::CrossBar::ExitDone);
    pCrossBarRunThread->WaitThread();

    //
    // 検証
    //

    // 受信データの検査。相手がエコーサーバのため、送信/受信データは等しくなるはず。
    const uint8_t* pRecvData[g_ChannelCount];
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        pRecvData[i] = reinterpret_cast<const uint8_t*>( pChannelAgents[i]->GetRecvData() );
        for( size_t j = 0; j < testDataSize[i]; j++ )
        {
            ASSERT_EQ( testData[i][j], pRecvData[i][j] );
        }
    }

    //
    // 終了処理
    //
    for( int i = 0; i < g_ChannelCount; i++ )
    {
        delete[] testData[i];
    }

    nn::tics::Finalize();
    nn::os::FinalizeMutex(&g_ConfigChannelMutex);
}
