﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

#include <cstdlib>
#include <cstring>

#include <nn/tics/tics_Api.h>
#include "CrossBar.h"
#include "TcpSlot.h"

#include "../testHtc_CrossBarRunThread.h"
#include "testHtc_EchoClientPlug.h"

namespace {
    // 1つのテストあたりのヒープサイズ
    const size_t TestHeapSize = 16 * 1024 * 1024;

    // ヒープを必要とするテストの数
    const int TestCount = 2;
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize( TestHeapSize * TestCount );
    NN_ABORT_UNLESS( result.IsSuccess(), NN_TEXT("メモリヒープの確保に失敗しました。") );
}

/**
    HTC ライブラリと HotBridgePCDeamon を経由して、
    エコーサーバにアクセスするテスト。
*/
TEST(echoClient, singleConnection)
{
    // メモリヒープ確保
    uintptr_t heapAddress;
    nn::Result result = nn::os::AllocateMemoryBlock( &heapAddress, TestHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // HTC ライブラリ初期化
    nn::tics::Initialize( heapAddress, TestHeapSize );

    // 文字列定数
    const ::tics::portability::stl::string testData = ::tics::portability::stl::string("ABCDEFGabcdefg");

    const ::tics::portability::stl::string echoClientSessionId("echoclient");
    const ::tics::portability::stl::string echoClientSlotId("tcp");
    const ::tics::portability::stl::string echoClientHost("localhost");
    const ::tics::portability::stl::string echoClientPort("6003");

    // 別スレッドとの待ち合わせ用イベント
    nn::os::EventType event;
    nn::os::InitializeEvent( &event, false, nn::os::EventClearMode_AutoClear );

    // Slot を1つだけ持つ CrossBar を作成
    ::tics::CrossBar* pCrossBar = new ::tics::CrossBar();
    ::tics::TCPSlot* pTcpSlot = new ::tics::TCPSlot(echoClientSlotId, "bridge_client", echoClientHost, echoClientPort);
    pCrossBar->AddSlot(echoClientSessionId, pTcpSlot);

    // EchoClientPlug を登録
    ::EchoClientPlug* pEchoClientPlug = new ::EchoClientPlug( &event );
    pEchoClientPlug->SetSendData(testData.c_str(), testData.size() + 1);
    pCrossBar->RegisterForSessionStart(echoClientSessionId, pEchoClientPlug);

    // CrossBar::Run() を別スレッドで呼び出し
    CrossBarRunThread* pCrossBarRunThread = new CrossBarRunThread( pCrossBar );
    pCrossBarRunThread->StartThread();

    // 処理の終了
    nn::os::WaitEvent( &event );
    pCrossBar->BeginShutdown(::tics::CrossBar::ExitDone);
    pCrossBarRunThread->WaitThread();

    // 相手がエコーサーバのため、送信/受信データは等しくなるはず
    const char* pRecvData = reinterpret_cast<const char*>( pEchoClientPlug->GetRecvData() );
    ASSERT_STREQ(testData.c_str(), pRecvData);

    nn::tics::Finalize();
}

/**
    大文字を含むセッション名で、CrossBar::AddSlot() と CrossBar::RegisterForSessionStart() を呼ぶテスト
*/
TEST(echoClient, sessionNameIncludingCapitalLetters)
{
    // メモリヒープ確保
    uintptr_t heapAddress;
    nn::Result result = nn::os::AllocateMemoryBlock( &heapAddress, TestHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // HTC ライブラリ初期化
    nn::tics::Initialize(heapAddress, TestHeapSize);

    // 文字列定数
    const ::tics::portability::stl::string echoClientSessionName("SessionNameIncludingCapitalLetters"); // セッション名に大文字を入れる
    const ::tics::portability::stl::string echoClientSlotId("tcp");
    const ::tics::portability::stl::string echoClientHost("localhost");
    const ::tics::portability::stl::string echoClientPort("6003");

    // Slot を1つだけ持つ CrossBar を作成
    ::tics::CrossBar* pCrossBar = new ::tics::CrossBar();
    ::tics::TCPSlot* pTcpSlot = new ::tics::TCPSlot(echoClientSlotId, "bridge_client", echoClientHost, echoClientPort);
    pCrossBar->AddSlot(echoClientSessionName, pTcpSlot);

    // プラグを登録
    EchoClientPlug* pEchoClientPlug = new EchoClientPlug( nullptr );
    int errorCode = pCrossBar->RegisterForSessionStart(echoClientSessionName, pEchoClientPlug);

    nn::tics::Finalize();

    // 正しく処理できた場合は、errorCode は 0 になる
    ASSERT_EQ( errorCode, 0 );
}

