﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ts.h>

#include <nn/i2c/i2c.h>
#include <nn/result/result_HandlingUtility.h>

namespace {

struct LocationData
{
    int temperatureMilliC;
    nn::ts::MeasurementMode measurementMode;
    bool errorEnabled;

    void Clear() NN_NOEXCEPT
    {
        temperatureMilliC = 0;
        measurementMode = nn::ts::MeasurementMode::Continuous;
        errorEnabled = false;
    }
};

LocationData g_LocationDataThermalSensorInternal;
LocationData g_LocationDataThermalSensorExternal;

LocationData* GetLocationData(nn::ts::Location location) NN_NOEXCEPT
{
    LocationData* pLocationData = &g_LocationDataThermalSensorInternal;

    switch ( location )
    {
    case nn::ts::Location_ThermalSensorInternal:
        pLocationData = &g_LocationDataThermalSensorInternal;
        break;
    case nn::ts::Location_ThermalSensorExternal:
        pLocationData = &g_LocationDataThermalSensorExternal;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return pLocationData;
}

} // namespace

namespace nn { namespace ts {

void Initialize() NN_NOEXCEPT
{
    g_LocationDataThermalSensorExternal.Clear();
    g_LocationDataThermalSensorInternal.Clear();
}

void Finalize() NN_NOEXCEPT
{
}

nn::Result GetTemperature(int* pOutTemperature, Location location) NN_NOEXCEPT
{
    if ( GetLocationData(location)->errorEnabled )
    {
        return nn::i2c::ResultNoAck();
    }

    *pOutTemperature = GetLocationData(location)->temperatureMilliC / 1000;
    NN_RESULT_SUCCESS;
}

nn::Result GetTemperatureMilliC(int* pOutTemperatureMilliC, Location location) NN_NOEXCEPT
{
    if ( GetLocationData(location)->errorEnabled )
    {
        return nn::i2c::ResultNoAck();
    }

    *pOutTemperatureMilliC = GetLocationData(location)->temperatureMilliC;
    NN_RESULT_SUCCESS;
}

void SetMeasurementMode(Location location, MeasurementMode measurementMode) NN_NOEXCEPT
{
    GetLocationData(location)->measurementMode = measurementMode;
}

}} // namespace nn::ts

namespace nnt { namespace ts {

void SetTemperature(nn::ts::Location location, int temperature) NN_NOEXCEPT
{
    GetLocationData(location)->temperatureMilliC = temperature * 1000;
}

void SetTemperatureMilliC(nn::ts::Location location, int temperatureMilliC) NN_NOEXCEPT
{
    GetLocationData(location)->temperatureMilliC = temperatureMilliC;
}

nn::ts::MeasurementMode GetMeasurementMode(nn::ts::Location location) NN_NOEXCEPT
{
    return GetLocationData(location)->measurementMode;
}

void SetErrorEnabled(nn::ts::Location location, bool enabled) NN_NOEXCEPT
{
    GetLocationData(location)->errorEnabled = enabled;
}

}} // namespace nnt::ts
