﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <Tasks/OsTlsTask.h>
#include <FileUtility.h>
#include <nn/nn_Log.h>

namespace nnt { namespace abuse {
    OsTlsTask::OsTlsTask(const String& typeName, const String& instanceName) :
        BaseTask(typeName, instanceName),
        m_threadArgs(nullptr),
        m_threads(nullptr),
        m_stacks(nullptr),
        m_maxThreads(64),
        m_maxGroups(16),
        m_numGroups(1),
        m_numThreads(1),
        m_numActualThreads(0),
        m_priority(nn::os::DefaultThreadPriority)
    {}

    OsTlsTask::~OsTlsTask()
    {}

    InitStatus OsTlsTask::Initialize(const String& params)
    {
        ArgVector args;
        FileUtility::ParamsToArgs(params, args);

        for(ScriptArg& arg : args)
        {
            if(arg.argName == "MaxThreads")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxThreads);
            else if(arg.argName == "MaxGroups")
                FileUtility::TryParseInt(arg, 1, 16, &m_maxGroups);
            else if(arg.argName == "Priority")
                FileUtility::TryParseInt(arg, nn::os::HighestThreadPriority, nn::os::LowestThreadPriority, &m_priority);
        }

        m_threads = (nn::os::ThreadType*) Platform::Allocate(sizeof(nn::os::ThreadType) * m_maxThreads);

        if(!m_threads)
        {
            LogError("Init Error - could not allocate threads\n");
            return INIT_ERROR;
        }

        for(int i = 0; i < m_maxThreads; ++i)
            ::new (&m_threads[i]) nn::os::ThreadType;

        LogInfo("Created thread array at %p\n", m_threads);
        m_stacks = (void**)Platform::Allocate(sizeof(void*) * m_maxThreads);

        if(!m_stacks)
        {
            LogError("Init Error - could not allocate stack array\n");
            Platform::Free(m_threads);
            m_threads = nullptr;
            return INIT_ERROR;
        }
        int i = 0;
        for(; i < m_maxThreads; ++i)
        {
            m_stacks[i] = Platform::AllocateAligned(4096, nn::os::StackRegionAlignment);
            if(!m_stacks[i])
                break;
        }

        if(i != m_maxThreads)
        {
            LogError("Init Error - could not allocate thread stacks\n");
            for(int j = 0; j <i; ++j)
            {
                Platform::Free(m_stacks[j]);
            }
            Platform::Free(m_stacks);
            Platform::Free(m_threads);
            m_threads = nullptr;
            m_stacks = nullptr;
            return INIT_ERROR;
        }

        m_threadArgs = (TlsThreadArg*)Platform::Allocate(m_maxThreads * sizeof(TlsThreadArg));

        if(!m_threadArgs)
            return INIT_ERROR;

        return INIT_OK;
    }

    StartStatus OsTlsTask::Start()
    {
        if(!Abuse::AcquireResource(this, RESOURCE_TLS_SLOTS))
            return START_RESOURCE_DENIED;

        ++m_numGroups;
        if(m_numGroups > m_maxGroups || m_numGroups > m_numThreads)
        {
            m_numGroups = 1;

            m_numThreads *= 2;
            if(m_numThreads > m_maxThreads)
                m_numThreads = 1;
        }

        LogInfo("Starting %d threads in %d groups\n", m_numThreads, m_numGroups);

        return START_OK;
    }

    RunStatus OsTlsTask::Run()
    {
        LogVerbose("++Run\n");

        m_numActualThreads = 0;

        LogVerbose("Allocating %d tls slots\n", m_numGroups);
        int numActualGroups = 0;
        for(int i = 0; i < m_numGroups; ++i)
        {
            nn::Result result = nn::os::AllocateTlsSlot(&m_TlsSlots[i], tlsDestructor);
            if(result.IsFailure())
            {
                LogWarning("Could not allocate tls slot %d. Continuing test with %d slots\n", i, numActualGroups);
                break;
            }
            ++numActualGroups;
        }


        for(int i = 0; i < m_numThreads; ++i)
        {
            m_threadArgs[i].task = this;
            m_threadArgs[i].tlsSlot = m_TlsSlots[i % numActualGroups];
            m_threadArgs[i].threadId = i;
            m_threadArgs[i].tlsDestroyed = false;
            m_threadArgs[i].tlsSet = false;

            if(! CreateThread(&m_threads[i], workerFunc, &m_threadArgs[i], m_stacks[i], 4096, m_priority) )
                break;
            ++m_numActualThreads;
        }

        if(m_numActualThreads != m_numThreads)
        {
            LogWarning("Could not create %d threads. Continuing this run with %d threads instead.\n", m_numThreads, m_numActualThreads);
        }

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::StartThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::WaitThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
        {
            DestroyThread(&m_threads[i]);
        }

        for(int i = 0; i < m_numActualThreads; ++i)
        {
            if(!m_threadArgs[i].tlsSet)
                LogWarning("Run complete, but Tls not set\n");
            if(!m_threadArgs[i].tlsDestroyed)
                LogWarning("Run complete, but Tls not destroyed\n");
        }
        LogVerbose("Freeing %d tls slots\n", numActualGroups);
        for(int i = 0; i < numActualGroups; ++i)
            nn::os::FreeTlsSlot(m_TlsSlots[i]);

        LogVerbose("--Run\n");
        return RUN_OK;
    }

    StopStatus OsTlsTask::Stop()
    {
        Abuse::ReleaseResource(this, RESOURCE_TLS_SLOTS);
        return STOP_OK;
    }

    ShutdownStatus OsTlsTask::Shutdown()
    {
        if(m_threads)
            Platform::Free(m_threads);

        if(m_stacks)
        {
            for(int i = 0; i < m_maxThreads; ++i)
                Platform::Free(m_stacks[i]);
            Platform::Free(m_stacks);
        }

        if(m_threadArgs)
            Platform::Free(m_threadArgs);

        return SHUTDOWN_OK;
    }

    const char* OsTlsTask::GetParamOptions()
    {
        return "MaxThreads=[1 - 64, default = 64] - maximum number of threads to run\n\
\tMaxGroups=[1 - 16, default = 16] - maximum number of groups\n\
\tPriority=[0 - 31, default = 16] - priority of threads";
    }

    void OsTlsTask::workerFunc(void* vTlsThreadArg)
    {
        TlsThreadArg* arg = (TlsThreadArg*)vTlsThreadArg;

        arg->task->LogVerbose("Setting tls value to 0x%x\n", arg);
        nn::os::SetTlsValue(arg->tlsSlot, (uintptr_t)arg);
        tlsCheck(arg->tlsSlot);
    }

    void OsTlsTask::tlsCheck(nn::os::TlsSlot slot)
    {
        TlsThreadArg* arg = (TlsThreadArg*)nn::os::GetTlsValue(slot);
        arg->task->LogVerbose("Got tls value 0x%x\n", arg);
        if(arg->tlsSet)
            arg->task->LogWarning("Tls set called but Tls already set\n");

        if(arg->tlsDestroyed)
            arg->task->LogWarning("Tls set called after tls destroyed\n");

        arg->tlsSet = true;
    }

    void OsTlsTask::tlsDestructor(uintptr_t value)
    {
        if(value == 0)
            return;
        TlsThreadArg* arg = (TlsThreadArg*)value;

        if(!arg->tlsSet)
            arg->task->LogWarning("Tls destruct called before Tls set\n");

        if(arg->tlsDestroyed)
            arg->task->LogWarning("Tls destruct called but Tls already destroyed\n");

        arg->tlsDestroyed = true;
    }

}    }
