﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <AbuseMath/math.h>
#include <Tasks/CpuMatrixTask.h>

namespace nnt
{
    namespace abuse
    {
        MatrixFunction s_functions[] =    {
                                        &CpuMatrixTask::TestVecAdd,
                                        &CpuMatrixTask::TestMTX4x4MultVec,
                                        };
        int            s_numFunctions = sizeof(s_functions) / sizeof(MatrixFunction);

        CpuMatrixTask::CpuMatrixTask(const String& typeName, const String& instanceName)
            : BaseTask(typeName, instanceName), m_functionIndex(0), m_testIterations(0)
        {}

        CpuMatrixTask::~CpuMatrixTask()
        {}

        InitStatus CpuMatrixTask::Initialize(const String& params)
        {
            (void)params;
            m_functionIndex = 0;
            m_testIterations = 10;
            runs = 0;
            return InitStatus::INIT_OK;
        }

        StartStatus CpuMatrixTask::Start()
        {
            LogInfo("Starting\n");
            return StartStatus::START_OK;
        }

        RunStatus CpuMatrixTask::Run()
        {

            MatrixFunction function = s_functions[m_functionIndex];

            bool result = ((*this).*(function))(ImplType::C);

            ++m_functionIndex;

            if(m_functionIndex == s_numFunctions)
                m_functionIndex = 0;

            if(result)
                return RunStatus::RUN_OK;
            else return RunStatus::RUN_ERROR;
        }

        StopStatus CpuMatrixTask::Stop()
        {
            LogInfo("Stopping\n");
            return StopStatus::STOP_OK;
        }

        ShutdownStatus CpuMatrixTask::Shutdown()
        {
            return ShutdownStatus::SHUTDOWN_OK;
        }

        static bool CmpVec(float vec[3], float ref[3])
        {
            bool ret = true;
            if(fabs(vec[0] - ref[0]) > 1.0e-6 ) ret = false;
            if(fabs(vec[1] - ref[1]) > 1.0e-6 ) ret = false;
            if(fabs(vec[2] - ref[2]) > 1.0e-6 ) ret = false;

            return ret;
        }

        static void sMultMtx44Vec(const float m[4][4], const float src[3], float dst[3])
        {
            float vTmp[3];
            f32 w;

            // a Vec has a 4th implicit 'w' coordinate of 1
            vTmp[0] = m[0][0]*src[0] + m[0][1]*src[1] + m[0][2]*src[2] + m[0][3];
            vTmp[1] = m[1][0]*src[0] + m[1][1]*src[1] + m[1][2]*src[2] + m[1][3];
            vTmp[2] = m[2][0]*src[0] + m[2][1]*src[1] + m[2][2]*src[2] + m[2][3];
            w      = m[3][0]*src[0] + m[3][1]*src[1] + m[3][2]*src[2] + m[3][3];
            w = 1.0f / w;

            // copy back
            dst[0] = vTmp[0] * w;
            dst[1] = vTmp[1] * w;
            dst[2] = vTmp[2] * w;
        }

        bool CpuMatrixTask::TestVecAdd(ImplType type)
        {
            (void)type;
            float a[3] = { 1.0f, 2.0f, 3.0f };
            float b[3] = { 3.0f, 2.0f, 1.0f };
            float dest[3];
            float ref[3] = { 4.0f, 4.0f, 4.0f };

            dest[0] = a[0] + b[0];
            dest[1] = a[1] + b[1];
            dest[2] = a[2] + b[2];

            //PerfBegin
            for(int i = 0; i < m_testIterations; ++i)
            {
                dest[0] = a[0] + b[0];
                dest[1] = a[1] + b[1];
                dest[2] = a[2] + b[2];
            }
            //PerfEnd

            if ((dest[0] == ref[0]) && (dest[1] == dest[1]) && (dest[2] == dest[2]))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        bool CpuMatrixTask::TestMTX4x4MultVec(ImplType type)
        {
            (void) type;


            float matrix[4][4] = { { 1.75f, 1.25f, 1.50f, 1.25f },
                                   { 2.50f, 2.75f, 2.25f, 2.75f },
                                   { 3.25f, 3.50f, 3.75f, 3.50f },
                                   { 1.25f, 1.50f, 1.75f, 1.50f } };
            float src[3] = { 1.25f, 2.5f, 3.75f };
            float ref[3] = { 0.911215f, 1.584112f, 2.271028f };
            float dst[3];

            sMultMtx44Vec(matrix, src, dst);

            for(int i = 0; i < m_testIterations; ++i)
            {
                sMultMtx44Vec(matrix, src, dst);
            }

            return CmpVec(dst, ref);
        }

    }
}
