﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include "BaseTask.h"
#include "AbuseString.h"


namespace nnt
{
    namespace abuse
    {
        class NetworkStressTask : public BaseTask
        {
        public:
            NetworkStressTask(const String& typeName, const String& instanceName);
            virtual ~NetworkStressTask();

            virtual InitStatus Initialize(const String& params);
            virtual StartStatus Start();
            virtual RunStatus Run();
            virtual StopStatus Stop();
            virtual ShutdownStatus Shutdown();


        private:

            enum TestType
            {
                TestType_Fixed = 0,
                TestType_Varying,
                TestType_Disconnects,
                TestType_Multiple
            };

            enum Direction
            {
                Direction_Upload = 0,
                Direction_Download,
                Direction_Both
            };

            enum Info
            {
                Info_None = 0,
                Info_Error = 1,
                Info_ThreadIndex = 2,
                Info_SocketShutdown =3
            };

            enum Command
            {
                Command_None = 0,
                Command_KillThread = 1,
                Command_LogError = 2,
                Command_LogWarning = 3,
                Command_LogInfo = 4,
                Command_LogVerbose = 5
            };

            class PacketMonitor
            {
                public:
                    NN_IMPLICIT PacketMonitor(NetworkStressTask * parent);
                    void VerifyPacket(unsigned packetId,unsigned threadID);
                private:

                    enum CountStatus
                    {
                        CountStatus_Empty = 0,
                        CountStatus_Accumulating = 1,
                        CountStatus_PacketsLost = 2
                    };
                    static const unsigned countMaxSize = 1000;
                    //packet count buffers
                    unsigned m_packetCountArray[2];
                    // status of the buffers
                    CountStatus m_packetCountStatusArray[2];
                    // RangeStart
                    unsigned m_packetRangeStart[2];
                    // index of the count buffer of the last received packet
                    unsigned m_lastPacketCountIndex;
                    // number of consecutive times the same buffer has received a packet
                    unsigned m_consecutiveCounts;
                    // pointer to the task for logging
                    NetworkStressTask * m_parent;
            };

            class NetworkThread;
            struct NetworkThreadArg
            {
                NetworkThreadArg(
                    NetworkStressTask * parent,
                    NetworkThread * networkThread,
                    RunStatus (*runningTest)(int socketDescriptor, NetworkThreadArg * arg),
                    TestType type,
                    nn::socket::Protocol protocol,
                    Direction direction,
                    bool flood,
                    nn::socket::SockAddrIn m_serverAddress,
                    int index,
                    int packetSize,
                    double idleTime);

                NetworkStressTask * parent;
                NetworkThread * networkThread;
                RunStatus (*runningTest)(int socketDescriptor, NetworkThreadArg * arg);
                TestType type;
                nn::socket::Protocol protocol;
                Direction direction;
                bool flood;
                bool firstRun;
                nn::socket::SockAddrIn serverAddress;
                int index;
                int packetSize;
                int packetId;
                int varyingPacketSize;
                double waitStart;
                double idleTime;
            };

            class NetworkThread
            {

            public:
                NetworkThread(
                    NetworkStressTask * parent,
                    RunStatus (*runningTest)(int socketDescriptor, NetworkThreadArg * arg),
                    TestType type,
                    nn::socket::Protocol protocol,
                    Direction direction,
                    bool flood,
                    nn::socket::SockAddrIn m_serverAddress,
                    int index,
                    int packetSize,
                    double idleTime);

                // creates the thread, initializes mutex
                int Initialize();
                // starts the thread
                void Start();
                // code the thread runs
                static void ThreadFunc(void * args);
                // Tells thread to stop and waits for it to do so.
                void Stop();
                // waits on the thread until it exits
                void WaitOnThread();
                // destroys the thread
                void Shutdown();

                ~NetworkThread();

                PacketMonitor packetMonitor;

            private:
                nn::os::MutexType m_threadMutex;
                nn::os::ThreadType m_thread;
                void* m_stack;
                NetworkThreadArg m_arg;
                bool m_stopping;
                bool m_exitedWithError;

            };

            struct Header
            {
                Header();
                Header(TestType type,Direction direction,int networkStressId,int networkStressThreadId, unsigned dataSize,double idleTime = .016);
                void Fill(TestType type,Direction direction,int networkStressId,int networkStressThreadId, unsigned dataSize,double idleTime = .016);
                void Fill(const Header & other);
                void Extract(char * buffer, int bufferLen, char ** dataStart);
                void ToNetwork();
                void ToLocal();

                TestType type;
                Direction direction;
                double idleTime;
                int networkStressId;
                int networkStressThreadId;
                unsigned dataSize;
            };

            struct UdpHeader
            {
                UdpHeader();
                UdpHeader(TestType type,Direction direction,int networkStressId,int networkStressThreadId,unsigned packetID, unsigned dataSize,Info info= Info_None,double idleTime = .016);
                void Fill(TestType type,Direction direction,int networkStressId,int networkStressThreadId,unsigned packetID, unsigned dataSize,Info info= Info_None,double idleTime = .016);
                void Fill(const UdpHeader & other);
                void Extract(char * buffer, int bufferLen, char ** dataStart);
                void ToNetwork();
                void ToLocal();


                TestType type;
                Direction direction;
                Info info;
                double idleTime;
                int networkStressId;
                int networkStressThreadId;
                unsigned packetID;
                unsigned dataSize;

            };

            struct CommandHeader
            {
                CommandHeader();
                CommandHeader(Command command,int networkStressId,int networkStressThreadId,unsigned dataSize);
                void Fill(Command command,int networkStressId,int networkStressThreadId,unsigned dataSize);
                void Fill(const CommandHeader & other);
                void Extract(char * buffer, int bufferLen, char ** dataStart);
                void ToNetwork();
                void ToLocal();

                Command command;
                int networkStressId;
                int networkStressThreadId;
                int dataSize;
            };

            struct CommandContainer
            {
                CommandContainer();
                CommandContainer(CommandHeader command,void * data,int dataSize);
                CommandContainer(const CommandContainer & other);
                ~CommandContainer();
                CommandContainer & operator=(const CommandContainer & rhs);

                CommandHeader command;
                int dataSize;
                void * data;
            };

            // fills out a header struct
            void FillHeader(Header& header,TestType type,Direction direction, unsigned dataSize,double idleTime = .016);
            void ExtractHeader(char * buffer, int bufferLen, Header & header, char ** dataStart);

            void FillUdpHeader(UdpHeader& header,TestType type,Direction direction,unsigned packetID, unsigned dataSize,Info info= Info_None,double idleTime = .016);
            void ExtractUdpHeader(char * buffer, int bufferLen, UdpHeader & header, char ** dataStart);

            bool ProccessParams(const String& params);
            static int HandleInfo(UdpHeader & header,const void * buffer, NetworkThreadArg * arg);
           // void ReadPacket(TestType& type,Direction& direction,char * data, unsigned size);

            int InitializeSocketLibrary();
            int FinalizeSocketLibrary();

            int CreateTcpStreamSocket();
            int CreateUdpDgramSocket();
            int Connect(nn::socket::SockAddrIn& address,int socketDesciptor);
            int ShutdownConnectedSocket(int socketDesciptor);

            static int ReceiveErrorCheck(int socketDescriptor,NetworkStressTask * task);
            static int CheckDataValidity(void* buff1, unsigned size1, void* buff2, unsigned size2,NetworkStressTask * task);

            static int SendDataTcp(int socketDescriptor, const Header& header, const char * buffer,NetworkStressTask * task);
            static int ReceiveDataTcp(int socketDescriptor, Header& header, char * buffer, int size,NetworkStressTask * task);

            static int SendDataUdp(int socketDescriptor, const UdpHeader& header, const char * buffer,NetworkStressTask * task,nn::socket::SockAddrIn * serverAddr);
            static int ReceiveDataUdp(int socketDescriptor, UdpHeader& header, char * buffer, int size,NetworkStressTask * task,bool isBlocking = false);

            int SendCommand(CommandContainer & command);
            int ReceiveCommand(CommandContainer & command);
            int HandleReceivedCommand(CommandContainer & command);
            void SetCurrentTest();

            //All the different tests
            static RunStatus RunTcpUploadDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunTcpUploadDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunTcpUploadFixedTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunTcpUploadVaryingTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunTcpDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunTcpDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg);

            static RunStatus RunUdpUploadDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunUdpUploadDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunUdpUploadFixedTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunUdpUploadVaryingTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunUdpDownloadFixedTest(int socketDescriptor, NetworkThreadArg * arg);
            static RunStatus RunUdpDownloadVaryingTest(int socketDescriptor, NetworkThreadArg * arg);


            RunStatus (*RunningTest)(int socketDescriptor, NetworkThreadArg * arg);
            bool m_connected;
            //bool m_firstRun;
            //double m_waitStart;
            double m_idleTime;


            int m_threadNumber;
            NetworkThread * m_threadArray;

            int m_commandSocketDescriptor;
            //int m_tcpSocketDescriptor;
          //  int m_udpSocketDescriptor;

            TestType m_testType;
            nn::socket::Protocol m_protocol;
            bool m_flood;
            bool m_fixed;
            Direction m_direction;
            unsigned m_packetSize;
            //unsigned m_varyingPacketSize;
            //unsigned m_packetId;
            //unsigned m_serverThreadId;

            String m_ipAddress;
            nn::socket::SockAddrIn m_commandServerAddress;
            nn::socket::SockAddrIn m_serverAddress;

            static int s_socketLibraryInitialized;

            static int s_currentNetworkTaskId;
            int m_networkTaskId;

            typedef std::list<CommandContainer, PlatformAllocator<CommandContainer>> ListCommandContainer;
            ListCommandContainer m_sendCommandList;
            nn::os::MutexType m_commandListMutex;

            PacketMonitor m_packetMonitor;
        };
    }
}
