﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/dd.h>
#include <nn/svc/svc_Dd.h>
#include <nn/nn_Abort.h>
#include "sdmmc_IHostController.h"

namespace nn { namespace sdmmc1 {
namespace detail {

class SdHostStandardController : public IHostController
{
private:
    // SD Host Standard Register
    // Specification Varsion 4.00 準拠
    struct SdHostStandardRegisters
    {
        volatile uint32_t   sdmaSystemAddress;              // 000h
        volatile uint16_t   blockSize;                      // 004h
        volatile uint16_t   blockCount;                     // 006h
        volatile uint32_t   argument1;                      // 008h
        volatile uint16_t   transferMode;                   // 00Ch
        volatile uint16_t   command;                        // 00Eh
        volatile uint32_t   response10;                     // 010h
        volatile uint32_t   response32;                     // 014h
        volatile uint32_t   response54;                     // 018h
        volatile uint32_t   response76;                     // 01Ch
        volatile uint32_t   bufferDataPort;                 // 020h
        volatile uint32_t   presentState;                   // 024h
        volatile uint8_t    hostControl1;                   // 028h
        volatile uint8_t    powerControl;                   // 029h
        volatile uint8_t    blockGapControl;                // 02Ah
        volatile uint8_t    wakeupControl;                  // 02Bh
        volatile uint16_t   clockControl;                   // 02Ch
        volatile uint8_t    timeoutControl;                 // 02Eh
        volatile uint8_t    softwareReset;                  // 02Fh
        volatile uint16_t   normalInterruptStatus;          // 030h
        volatile uint16_t   errorInterruptStatus;           // 032h
        volatile uint16_t   normalInterruptStatusEnable;    // 034h
        volatile uint16_t   errorInterruptStatusEnable;     // 036h
        volatile uint16_t   normalInterruptSignalEnable;    // 038h
        volatile uint16_t   errorInterruptSignalEnable;     // 03Ah
        volatile uint16_t   autoCmdErrorStatus;             // 03Ch
        volatile uint16_t   hostControl2;                   // 03Eh
        volatile uint32_t   capabilities10;                 // 040h
        volatile uint32_t   capabilities32;                 // 048h
        volatile uint32_t   maximumCurrentCapabilities10;   // 048h
        volatile uint32_t   maximumCurrentCapabilities32;   // 04Ch
    };

    SdHostStandardRegisters* m_pRegisters;
    nn::dd::PhysicalAddress  m_NextPhysicalAddress;
    bool m_IsSupportBusPower1_8V;
    bool m_IsSupportBusPower3_3V;

    void SetTransfer(uint32_t* pOutTransferredNumBlocks, const TransferData* pTransferData) NN_NOEXCEPT;
    uint16_t GetCommandValue(const Command* pCommand, bool isWithTransferData) NN_NOEXCEPT;
    Result AbortTransaction() NN_NOEXCEPT;
    Result WaitWhileCommandInhibit(bool isWithDatLine) NN_NOEXCEPT;
    Result WaitWhileBusy() NN_NOEXCEPT;
    Result WaitCommandComplete() NN_NOEXCEPT;
    Result WaitTransferComplete() NN_NOEXCEPT;
    void LogRegisters() NN_NOEXCEPT;

public:
    SdHostStandardController (uint32_t registersAddress, bool isSupportBusPower1_8V, bool isSupportBusPower3_3V) NN_NOEXCEPT
    {
        uintptr_t reg;
        NN_ABORT_UNLESS(nn::svc::QueryIoMapping(&reg, registersAddress, 0x10).IsSuccess());
        m_pRegisters = reinterpret_cast<SdHostStandardRegisters*>(reg);
        m_IsSupportBusPower1_8V = isSupportBusPower1_8V;
        m_IsSupportBusPower3_3V = isSupportBusPower3_3V;
    }

    virtual void Reset() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetBusWidth(BusWidth busWidth) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsSupportBusPower(BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetBusPower(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetDeviceClock(uint32_t* pOutDeviceClockFrequencyKHz, DeviceClockMode deviceClockMode) NN_NOEXCEPT NN_OVERRIDE;
    virtual void EnableDeviceClock() NN_NOEXCEPT NN_OVERRIDE;
    virtual void DisableDeviceClock() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT NN_OVERRIDE;
    virtual void GetLastResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT NN_OVERRIDE;
    virtual void GetStopTransmissionResponse(uint32_t* pOutResponse, size_t responseSize) const NN_NOEXCEPT NN_OVERRIDE;

};

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
