﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdHostStandardController.h"
#include <nn/nn_Abort.h>
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    // 004h Block Size Register ビット定義
    const uint8_t REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__SHIFT = 12;
    const uint16_t REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__MASK = 0x7 << REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__SHIFT;
    const uint16_t REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__DEF = 0x0 << REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__SHIFT;  // 4KB
    const size_t SdmaBufferBoundary = 4 * 1024;
    const uint8_t REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__SHIFT = 0;
    const uint16_t REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__MASK = 0xFFF << REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__SHIFT;

    // 006h Block Count Register ビット定義
    const uint16_t REG_BLOCK_COUNT__MAX = 0xFFFF;

    // 00Ch Transfer Mode Register ビット定義
    const uint16_t REG_TRANSFER_MODE__RESPONSE_INTERRUPT_DISABLE = 0x1 << 8;
    const uint16_t REG_TRANSFER_MODE__RESPONSE_ERROR_CHECK_ENABLE = 0x1 << 7;
    const uint16_t REG_TRANSFER_MODE__MULTI_BLOCK_SELECT = 0x1 << 5;
    const uint16_t REG_TRANSFER_MODE__DATA_TRANSFER_DIRECTION_READ = 0x1 << 4;
    const uint8_t REG_TRANSFER_MODE__AUTO_CMD_ENABLE__SHIFT = 2;
    const uint16_t REG_TRANSFER_MODE__AUTO_CMD_ENABLE__MASK = 0x3 << REG_TRANSFER_MODE__AUTO_CMD_ENABLE__SHIFT;
    const uint16_t REG_TRANSFER_MODE__AUTO_CMD_ENABLE__AUTO_CMD12_ENABLE = 0x1 << REG_TRANSFER_MODE__AUTO_CMD_ENABLE__SHIFT;
    const uint16_t REG_TRANSFER_MODE__BLOCK_COUNT_ENABLE = 0x1 << 1;
    const uint16_t REG_TRANSFER_MODE__DMA_ENABLE = 0x1 << 0;

    // 00Eh Command Register ビット定義
    const uint8_t REG_COMMAND__COMMAND_INDEX__SHIFT = 8;
    const uint16_t REG_COMMAND__COMMAND_INDEX__MASK = 0x3F << REG_COMMAND__COMMAND_INDEX__SHIFT;
    const uint16_t REG_COMMAND__DATA_PRESENT_SELECT  = 0x1 << 5;
    const uint16_t REG_COMMAND__COMMAND_INDEX_CHECK_ENABLE  = 0x1 << 4;
    const uint16_t REG_COMMAND__COMMAND_CRC_CHECK_ENABLE    = 0x1 << 3;
    const uint8_t REG_COMMAND__RESPONSE_TYPE_SELECT__SHIFT = 0;
    const uint16_t REG_COMMAND__RESPONSE_TYPE_SELECT__MASK                      = 0x3 << REG_COMMAND__RESPONSE_TYPE_SELECT__SHIFT;
    const uint16_t REG_COMMAND__RESPONSE_TYPE_SELECT__NO_RESPONSE               = 0x0 << REG_COMMAND__RESPONSE_TYPE_SELECT__SHIFT;
    const uint16_t REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_136       = 0x1 << REG_COMMAND__RESPONSE_TYPE_SELECT__SHIFT;
    const uint16_t REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48        = 0x2 << REG_COMMAND__RESPONSE_TYPE_SELECT__SHIFT;
    const uint16_t REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48_BUSY   = 0x3 << REG_COMMAND__RESPONSE_TYPE_SELECT__SHIFT;

    // 024h Present State Register ビット定義
    const uint32_t REG_PRESENT_STATE__DAT0_LINE_SIGNAL_LEVEL = 0x1ULL << 20;
    const uint32_t REG_PRESENT_STATE__BUFFER_READ_ENABLE = 0x1 << 11;
    const uint32_t REG_PRESENT_STATE__BUFFER_WRITE_ENABLE = 0x1 << 10;
    const uint32_t REG_PRESENT_STATE__COMMAND_INHIBIT_DAT = 0x1 << 1;
    const uint32_t REG_PRESENT_STATE__COMMAND_INHIBIT_CMD = 0x1 << 0;

    // 02Ch Clock Control Register ビット定義
    const uint8_t REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__SHIFT = 8;
    const uint16_t REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__MASK = 0xFF << REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__SHIFT;
    const uint8_t REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__SHIFT = 6;
    const uint16_t REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__MASK = 0x3 << REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__SHIFT;
    const uint16_t REG_CLOCK_CONTROL__CLOCK_GENERATOR_SELECT    = 0x1 << 5;
    const uint16_t REG_CLOCK_CONTROL__SD_CLOCK_ENABLE           = 0x1 << 2;
    const uint16_t REG_CLOCK_CONTROL__INTERNAL_CLOCK_STABLE     = 0x1 << 1;
    const uint16_t REG_CLOCK_CONTROL__INTERNAL_CLOCK_ENABLE     = 0x1 << 0;

    // 028h Host Control 1 Register ビット定義
    const uint8_t REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH = 0x1 << 5;
    const uint8_t REG_HOST_CONTROL_1__DATA_TRANSFER_WIDTH          = 0x1 << 1;
    const uint8_t REG_HOST_CONTROL_1__DMA_SELECT__SHIFT = 3;
    const uint8_t REG_HOST_CONTROL_1__DMA_SELECT__MASK = 0x3 << REG_HOST_CONTROL_1__DMA_SELECT__SHIFT;
    const uint8_t REG_HOST_CONTROL_1__DMA_SELECT__SDMA = 0x0 << REG_HOST_CONTROL_1__DMA_SELECT__SHIFT;

    // 029h Power Control Register ビット定義
    const uint8_t REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__SHIFT = 1;
    const uint16_t REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__MASK = 0x7 << REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__SHIFT;
    const uint16_t REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__3_3V = 0x7 << REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__SHIFT;
    const uint16_t REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__3_0V = 0x6 << REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__SHIFT;
    const uint16_t REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__1_8V = 0x5 << REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__SHIFT;
    const uint16_t REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1 = 0x1 << 0;

    // 02Eh Timeout Control Register ビット定義
    const uint8_t REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__SHIFT = 0;
    const uint8_t REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__MASK = 0xF << REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__SHIFT;
    const uint8_t REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__MAX = 0xE << REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__SHIFT;

    // 02Fh Software Reset Register ビット定義
    const uint8_t REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_DAT = 0x1 << 2;
    const uint8_t REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_CMD = 0x1 << 1;
    const uint8_t REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_ALL = 0x1 << 0;

    // 030h/034h Normal Interrupt Status/Enable Register ビット定義
    const uint16_t REG_NORMAL_INTERRUPT__ERROR_INTERRUPT        = 0x1 << 15;    // ROC
    const uint16_t REG_NORMAL_INTERRUPT__DMA_INTERRUPT          = 0x1 << 3;     // RW1C
    const uint16_t REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE      = 0x1 << 1;     // RW1C
    const uint16_t REG_NORMAL_INTERRUPT__COMMAND_COMPLETE       = 0x1 << 0;     // RW1C
    const uint16_t REG_NORMAL_INTERRUPT__ISSUE_COMMAND__MASK = ( \
        REG_NORMAL_INTERRUPT__DMA_INTERRUPT | \
        REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE | \
        REG_NORMAL_INTERRUPT__COMMAND_COMPLETE );

    // 032h/038h Error Interrupt Status/Enable Register ビット定義
    const uint16_t REG_ERROR_INTERRUPT__AUTO_CMD_ERROR          = 0x1 << 8;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__DATA_END_BIT_ERROR      = 0x1 << 6;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__DATA_CRC_ERROR          = 0x1 << 5;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__DATA_TIMEOUT_ERROR      = 0x1 << 4;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__COMMAND_INDEX_ERROR     = 0x1 << 3;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__COMMAND_END_BIT_ERROR   = 0x1 << 2;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__COMMAND_CRC_ERROR       = 0x1 << 1;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__COMMAND_TIMEOUT_ERROR   = 0x1 << 0;     // RW1C
    const uint16_t REG_ERROR_INTERRUPT__ISSUE_COMMAND__MASK = ( \
        REG_ERROR_INTERRUPT__AUTO_CMD_ERROR | \
        REG_ERROR_INTERRUPT__DATA_END_BIT_ERROR | \
        REG_ERROR_INTERRUPT__DATA_CRC_ERROR | \
        REG_ERROR_INTERRUPT__DATA_TIMEOUT_ERROR | \
        REG_ERROR_INTERRUPT__COMMAND_INDEX_ERROR | \
        REG_ERROR_INTERRUPT__COMMAND_END_BIT_ERROR | \
        REG_ERROR_INTERRUPT__COMMAND_CRC_ERROR | \
        REG_ERROR_INTERRUPT__COMMAND_TIMEOUT_ERROR );

    // 03Ch Auto CMD Error Status Register ビット定義
    const uint16_t REG_AUTO_CMD_ERROR__COMMAND_NOT_ISSUED       = 0x1 << 7;     // ROC
    const uint16_t REG_AUTO_CMD_ERROR__AUTO_CMD_INDEX_ERROR     = 0x1 << 4;     // ROC
    const uint16_t REG_AUTO_CMD_ERROR__AUTO_CMD_END_BIT_ERROR   = 0x1 << 3;     // ROC
    const uint16_t REG_AUTO_CMD_ERROR__AUTO_CMD_CRC_ERROR       = 0x1 << 2;     // ROC
    const uint16_t REG_AUTO_CMD_ERROR__AUTO_CMD_TIMEOUT_ERROR   = 0x1 << 1;     // ROC
    const uint16_t REG_AUTO_CMD_ERROR__AUTO_CMD_NOT_EXECUTED    = 0x1 << 0;     // ROC
    const uint16_t REG_AUTO_CMD_ERROR__ISSUE_COMMAND__MASK = ( \
        REG_AUTO_CMD_ERROR__COMMAND_NOT_ISSUED | \
        REG_AUTO_CMD_ERROR__AUTO_CMD_INDEX_ERROR | \
        REG_AUTO_CMD_ERROR__AUTO_CMD_END_BIT_ERROR | \
        REG_AUTO_CMD_ERROR__AUTO_CMD_CRC_ERROR | \
        REG_AUTO_CMD_ERROR__AUTO_CMD_TIMEOUT_ERROR | \
        REG_AUTO_CMD_ERROR__AUTO_CMD_NOT_EXECUTED );

    // 03Eh Host Control 2 Register ビット定義
    const uint16_t REG_HOST_CONTROL_2__PRESET_VALUE_ENABLE = 0x1 << 15;
    const uint16_t REG_HOST_CONTROL_2__HOST_VERSION_4_00_ENABLE = 0x01 << 12;

    Result CheckErrorStatus(uint16_t normalInterruptStatus, uint16_t errorInterruptStatus, uint16_t autoCmdErrorStatus) NN_NOEXCEPT
    {
        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__ERROR_INTERRUPT) != 0)
        {
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_INDEX_ERROR) != 0)
            {
                return ResultResponseIndexError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_END_BIT_ERROR) != 0)
            {
                return ResultResponseEndBitError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_CRC_ERROR) != 0)
            {
                return ResultResponseCrcError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__COMMAND_TIMEOUT_ERROR) != 0)
            {
                return ResultResponseTimeoutError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__DATA_END_BIT_ERROR) != 0)
            {
                return ResultDataEndBitError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__DATA_CRC_ERROR) != 0)
            {
                return ResultDataCrcError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__DATA_TIMEOUT_ERROR) != 0)
            {
                return ResultDataTimeoutError();
            }
            if ((errorInterruptStatus & REG_ERROR_INTERRUPT__AUTO_CMD_ERROR) != 0)
            {
                if (autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_INDEX_ERROR)
                {
                    return ResultResponseIndexError();
                }
                if ((autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_END_BIT_ERROR) != 0)
                {
                    return ResultResponseEndBitError();
                }
                if ((autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_CRC_ERROR) != 0)
                {
                    return ResultResponseCrcError();
                }
                if ((autoCmdErrorStatus & REG_AUTO_CMD_ERROR__AUTO_CMD_TIMEOUT_ERROR) != 0)
                {
                    return ResultResponseTimeoutError();
                }
                NN_DETAIL_SDMMC_ERROR_LOG("There in no error corresponding Auto CMD Error: 0x%04X\n", autoCmdErrorStatus);
                return ResultHostControllerUnexpected();
            }
            NN_DETAIL_SDMMC_ERROR_LOG("There in no error corresponding Error Interrupt: 0x%04X\n", errorInterruptStatus);
            return ResultHostControllerUnexpected();
        }
        return ResultSuccess();
    }

}

void SdHostStandardController::SetTransfer(uint32_t* pOutTransferredNumBlocks, const TransferData* pTransferData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    // 物理アドレスと連続するサイズを取得
    nn::dd::PhysicalAddress physicalAddress = reinterpret_cast<nn::dd::PhysicalAddress>(pTransferData->m_pBuffer);

    uint32_t numBlocks = pTransferData->m_NumBlocks;

    NN_DETAIL_SDMMC_DEBUG_LOG("physicalAddress 0x%08X, numBlocks %u\n", static_cast<uint32_t>(physicalAddress), numBlocks);

    // 物理アドレスは 32bit アライン想定
    NN_ABORT_UNLESS((physicalAddress % sizeof(uint32_t)) == 0);
    // 連続するサイズが 1 ブロック以上である想定
    NN_ABORT_UNLESS(numBlocks >= 1);

    // データ転送最大ブロック数で打ち切る
    numBlocks = (numBlocks < REG_BLOCK_COUNT__MAX) ? numBlocks : REG_BLOCK_COUNT__MAX;
    if (pOutTransferredNumBlocks != nullptr)
    {
        *pOutTransferredNumBlocks = numBlocks;
    }

    // SDMA アドレス設定
    m_pRegisters->sdmaSystemAddress = static_cast<uint32_t>(physicalAddress);
    m_NextPhysicalAddress = physicalAddress + SdmaBufferBoundary;

    // ブロックサイズ、ブロック数設定
    m_pRegisters->blockSize = REG_BLOCK_SIZE__SDMA_BUFFER_BOUNDARY__DEF | (pTransferData->m_BlockSize << REG_BLOCK_SIZE__TRANSFER_BLOCK_SIZE__SHIFT);
    m_pRegisters->blockCount = numBlocks;

    uint16_t transferMode = m_pRegisters->transferMode;
    if (pTransferData->m_IsMultiBlockTransfer)
    {
        transferMode |= REG_TRANSFER_MODE__MULTI_BLOCK_SELECT;
        transferMode = (transferMode & (~REG_TRANSFER_MODE__AUTO_CMD_ENABLE__MASK)) | REG_TRANSFER_MODE__AUTO_CMD_ENABLE__AUTO_CMD12_ENABLE;
        transferMode |= REG_TRANSFER_MODE__BLOCK_COUNT_ENABLE;
    }
    else
    {
        transferMode &= (~REG_TRANSFER_MODE__MULTI_BLOCK_SELECT);
    }
    if (pTransferData->m_TransferDirection == TransferDirection_ReadFromDevice)
    {
        transferMode |= REG_TRANSFER_MODE__DATA_TRANSFER_DIRECTION_READ;
    }
    else
    {
        transferMode &= (~REG_TRANSFER_MODE__DATA_TRANSFER_DIRECTION_READ);
    }
    m_pRegisters->transferMode = transferMode;
}

uint16_t SdHostStandardController::GetCommandValue(const Command* pCommand, bool isWithTransferData) NN_NOEXCEPT
{
    uint16_t command = 0;

    switch (pCommand->m_ResponseType)
    {
    case ResponseType_R0:
        command |= REG_COMMAND__RESPONSE_TYPE_SELECT__NO_RESPONSE;
        // Command Index Check は行わない
        // Command CRC Check は行わない
        break;
    case ResponseType_R1:
        NN_FALL_THROUGH;
    case ResponseType_R6:
        NN_FALL_THROUGH;
    case ResponseType_R7:
        if (pCommand->m_IsBusy)
        {
            command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48_BUSY;
        }
        else
        {
            command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48;
        }
        command |= REG_COMMAND__COMMAND_INDEX_CHECK_ENABLE;
        command |= REG_COMMAND__COMMAND_CRC_CHECK_ENABLE;
        break;
    case ResponseType_R2:
        command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_136;
        // Command Index Check は行わない
        command |= REG_COMMAND__COMMAND_CRC_CHECK_ENABLE;
        break;
    case ResponseType_R3:
        command |= REG_COMMAND__RESPONSE_TYPE_SELECT__RESPONSE_LENGTH_48;
        // Command Index Check は行わない
        // Command CRC Check は行わない
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    if (isWithTransferData)
    {
        command |= REG_COMMAND__DATA_PRESENT_SELECT;
    }
    NN_ABORT_UNLESS(pCommand->m_CommandIndex <= (REG_COMMAND__COMMAND_INDEX__MASK >> REG_COMMAND__COMMAND_INDEX__SHIFT));
    command |= (pCommand->m_CommandIndex << REG_COMMAND__COMMAND_INDEX__SHIFT);

    return command;
}

Result SdHostStandardController::AbortTransaction() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    // ステータスクリア（1 を立てたビットだけがクリアされる）
    m_pRegisters->normalInterruptStatus = m_pRegisters->normalInterruptStatus;
    m_pRegisters->errorInterruptStatus = m_pRegisters->errorInterruptStatus;

    m_pRegisters->softwareReset |= (REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_DAT | REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_CMD);
    while (true)
    {
        if ((m_pRegisters->softwareReset & (REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_DAT | REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_CMD)) == 0)
        {
            break;
        }
        // TODO: タイムアウト実装
    }

    return ResultSuccess();
}

Result SdHostStandardController::WaitWhileCommandInhibit(bool isWithDatLine) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    // CMD ライン空き待ち
    while (true)
    {
        if ((m_pRegisters->presentState & REG_PRESENT_STATE__COMMAND_INHIBIT_CMD) == 0)
        {
            break;
        }
        // TODO: タイムアウト実装
    }
    if (isWithDatLine)
    {
        // DAT ライン空き待ち
        while (true)
        {
            if ((m_pRegisters->presentState & REG_PRESENT_STATE__COMMAND_INHIBIT_DAT) == 0)
            {
                break;
            }
            // TODO: タイムアウト実装
        }
    }

    return ResultSuccess();
}

Result SdHostStandardController::WaitWhileBusy() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    // Busy 待ち
    while (true)
    {
        if ((m_pRegisters->presentState & REG_PRESENT_STATE__DAT0_LINE_SIGNAL_LEVEL) != 0)
        {
            break;
        }
        // TODO: タイムアウト実装
    }

    return ResultSuccess();
}

Result SdHostStandardController::WaitCommandComplete() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    while (true)
    {
        volatile uint16_t normalInterruptStatus =  m_pRegisters->normalInterruptStatus;
        Result result = CheckErrorStatus(normalInterruptStatus, m_pRegisters->errorInterruptStatus, m_pRegisters->autoCmdErrorStatus);
        if (result.IsFailure())
        {
            return result;
        }
        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__COMMAND_COMPLETE) != 0)
        {
            // Command Complete だけクリア（1 を立てたビットだけがクリアされる）
            m_pRegisters->normalInterruptStatus = REG_NORMAL_INTERRUPT__COMMAND_COMPLETE;
            break;
        }
        // TODO: タイムアウト実装
    }

    return ResultSuccess();
}

Result SdHostStandardController::WaitTransferComplete() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    while (true)
    {
        volatile uint16_t normalInterruptStatus =  m_pRegisters->normalInterruptStatus;
        Result result = CheckErrorStatus(normalInterruptStatus, m_pRegisters->errorInterruptStatus, m_pRegisters->autoCmdErrorStatus);
        if (result.IsFailure())
        {
            AbortTransaction();
            return result;
        }
        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__DMA_INTERRUPT) != 0)
        {
            // DMA Interrupt だけクリア（1 を立てたビットだけがクリアされる）
            m_pRegisters->normalInterruptStatus = REG_NORMAL_INTERRUPT__DMA_INTERRUPT;
            m_pRegisters->sdmaSystemAddress = static_cast<uint32_t>(m_NextPhysicalAddress);
            m_NextPhysicalAddress += SdmaBufferBoundary;
        }
        if ((normalInterruptStatus & REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE) != 0)
        {
            // Transfer Complete だけクリア（1 を立てたビットだけがクリアされる）
            m_pRegisters->normalInterruptStatus = REG_NORMAL_INTERRUPT__TRANSFER_COMPLETE;
            break;
        }
        // TODO: タイムアウト実装
    }

    return ResultSuccess();
}

void SdHostStandardController::LogRegisters() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    NN_DETAIL_SDMMC_DEBUG_LOG("sdmaSystemAddress(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->sdmaSystemAddress)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->sdmaSystemAddress);
    NN_DETAIL_SDMMC_DEBUG_LOG("blockSize(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->blockSize)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->blockSize));
    NN_DETAIL_SDMMC_DEBUG_LOG("blockCount(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->blockCount)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->blockCount));
    NN_DETAIL_SDMMC_DEBUG_LOG("argument1(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->argument1)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->argument1);
    NN_DETAIL_SDMMC_DEBUG_LOG("transferMode(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->transferMode)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->transferMode));
    NN_DETAIL_SDMMC_DEBUG_LOG("command(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->command)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->command));
    NN_DETAIL_SDMMC_DEBUG_LOG("response10(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->response10)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->response10);
    NN_DETAIL_SDMMC_DEBUG_LOG("response32(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->response32)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->response32);
    NN_DETAIL_SDMMC_DEBUG_LOG("response54(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->response54)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->response54);
    NN_DETAIL_SDMMC_DEBUG_LOG("response76(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->response76)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->response76);
    NN_DETAIL_SDMMC_DEBUG_LOG("bufferDataPort(%03Xh) skip\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->bufferDataPort)) - reinterpret_cast<uintptr_t>(m_pRegisters));
    NN_DETAIL_SDMMC_DEBUG_LOG("presentState(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->presentState)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->presentState);
    NN_DETAIL_SDMMC_DEBUG_LOG("hostControl1(%03Xh) 0x%02X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->hostControl1)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->hostControl1));
    NN_DETAIL_SDMMC_DEBUG_LOG("powerControl(%03Xh) 0x%02X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->powerControl)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->powerControl));
    NN_DETAIL_SDMMC_DEBUG_LOG("blockGapControl(%03Xh) 0x%02X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->blockGapControl)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->blockGapControl));
    NN_DETAIL_SDMMC_DEBUG_LOG("wakeupControl(%03Xh) 0x%02X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->wakeupControl)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->wakeupControl));
    NN_DETAIL_SDMMC_DEBUG_LOG("clockControl(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->clockControl)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->clockControl));
    NN_DETAIL_SDMMC_DEBUG_LOG("timeoutControl(%03Xh) 0x%02X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->timeoutControl)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->timeoutControl));
    NN_DETAIL_SDMMC_DEBUG_LOG("softwareReset(%03Xh) 0x%02X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->softwareReset)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->softwareReset));
    NN_DETAIL_SDMMC_DEBUG_LOG("normalInterruptStatus(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->normalInterruptStatus)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->normalInterruptStatus));
    NN_DETAIL_SDMMC_DEBUG_LOG("errorInterruptStatus(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->errorInterruptStatus)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->errorInterruptStatus));
    NN_DETAIL_SDMMC_DEBUG_LOG("normalInterruptStatusEnable(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->normalInterruptStatusEnable)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->normalInterruptStatusEnable));
    NN_DETAIL_SDMMC_DEBUG_LOG("errorInterruptStatusEnable(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->errorInterruptStatusEnable)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->errorInterruptStatusEnable));
    NN_DETAIL_SDMMC_DEBUG_LOG("normalInterruptSignalEnable(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->normalInterruptSignalEnable)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->normalInterruptSignalEnable));
    NN_DETAIL_SDMMC_DEBUG_LOG("errorInterruptSignalEnable(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->errorInterruptSignalEnable)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->errorInterruptSignalEnable));
    NN_DETAIL_SDMMC_DEBUG_LOG("autoCmdErrorStatus(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->autoCmdErrorStatus)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->autoCmdErrorStatus));
    NN_DETAIL_SDMMC_DEBUG_LOG("hostControl2(%03Xh) 0x%04X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->hostControl2)) - reinterpret_cast<uintptr_t>(m_pRegisters), static_cast<uint32_t>(m_pRegisters->hostControl2));
    NN_DETAIL_SDMMC_DEBUG_LOG("capabilities10(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->capabilities10)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->capabilities10);
    NN_DETAIL_SDMMC_DEBUG_LOG("capabilities32(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->capabilities32)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->capabilities32);
    NN_DETAIL_SDMMC_DEBUG_LOG("maximumCurrentCapabilities10(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->maximumCurrentCapabilities10)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->maximumCurrentCapabilities10);
    NN_DETAIL_SDMMC_DEBUG_LOG("maximumCurrentCapabilities32(%03Xh) 0x%08X\n",
        reinterpret_cast<uintptr_t>(&(m_pRegisters->maximumCurrentCapabilities32)) - reinterpret_cast<uintptr_t>(m_pRegisters), m_pRegisters->maximumCurrentCapabilities32);
}

void SdHostStandardController::Reset() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    // Host Controller リセット
    m_pRegisters->softwareReset |= REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_ALL;
    while (true)
    {
        if ((m_pRegisters->softwareReset & REG_SOFTWARE_RESET__SOFTWARE_RESET_FOR_ALL) == 0)
        {
            break;
        }
        // TODO: タイムアウト実装
    }

    // Host Controller 内部クロック供給
    m_pRegisters->clockControl |= REG_CLOCK_CONTROL__INTERNAL_CLOCK_ENABLE;
    while (true)
    {
        if ((m_pRegisters->clockControl & REG_CLOCK_CONTROL__INTERNAL_CLOCK_STABLE) != 0)
        {
            break;
        }
        // TODO: タイムアウト実装
    }

    // 手動でバス設定
    m_pRegisters->hostControl2 &= (~REG_HOST_CONTROL_2__PRESET_VALUE_ENABLE);
    m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__CLOCK_GENERATOR_SELECT);

    // 手動でレスポンスチェック
    m_pRegisters->transferMode &= (~REG_TRANSFER_MODE__RESPONSE_INTERRUPT_DISABLE);
    m_pRegisters->transferMode &= (~REG_TRANSFER_MODE__RESPONSE_ERROR_CHECK_ENABLE);

    // SDMA 転送
    m_pRegisters->hostControl1 = (m_pRegisters->hostControl1 & (~REG_HOST_CONTROL_1__DMA_SELECT__MASK)) | REG_HOST_CONTROL_1__DMA_SELECT__SDMA;
    m_pRegisters->hostControl2 &= (~REG_HOST_CONTROL_2__HOST_VERSION_4_00_ENABLE);
    m_pRegisters->transferMode |= REG_TRANSFER_MODE__DMA_ENABLE;

    // タイムアウト最大 TODO: 適切な値に変更
    m_pRegisters->timeoutControl = (m_pRegisters->timeoutControl & (~REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__MASK)) | REG_TIMEOUT_CONTROL__DATA_TIMEOUT_COUNTER_VALUE__MAX;

    // ステータスレジスタ許可
    m_pRegisters->normalInterruptStatusEnable |= REG_NORMAL_INTERRUPT__ISSUE_COMMAND__MASK;
    m_pRegisters->errorInterruptStatusEnable |= REG_ERROR_INTERRUPT__ISSUE_COMMAND__MASK;
}

void SdHostStandardController::SetBusWidth(BusWidth busWidth) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    switch (busWidth)
    {
    case BusWidth_1Bit:
        m_pRegisters->hostControl1 &= (~REG_HOST_CONTROL_1__DATA_TRANSFER_WIDTH);
        m_pRegisters->hostControl1 &= (~REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH);
        return;
    case BusWidth_4Bit:
        m_pRegisters->hostControl1 |= REG_HOST_CONTROL_1__DATA_TRANSFER_WIDTH;
        m_pRegisters->hostControl1 &= (~REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH);
        return;
    case BusWidth_8Bit:
        m_pRegisters->hostControl1 |= REG_HOST_CONTROL_1__EXTENDED_DATA_TRANSFER_WIDTH;
        return;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool SdHostStandardController::IsSupportBusPower(BusPower busPower) const NN_NOEXCEPT
{
    switch (busPower)
    {
    case BusPower_Off:
        return true;
    case BusPower_1_8V:
        return m_IsSupportBusPower1_8V;
    case BusPower_3_3V:
        return m_IsSupportBusPower3_3V;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SdHostStandardController::SetBusPower(BusPower busPower) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    NN_ABORT_UNLESS(IsSupportBusPower(busPower));
    switch (busPower)
    {
    case BusPower_Off:
        m_pRegisters->powerControl &= (~REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1);
        return;
    case BusPower_1_8V:
        m_pRegisters->powerControl = (m_pRegisters->powerControl & (~REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__MASK)) | REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__1_8V;
        m_pRegisters->powerControl |= REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1;
        return;
    case BusPower_3_3V:
        m_pRegisters->powerControl = (m_pRegisters->powerControl & (~REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__MASK)) | REG_POWER_CONTROL__SD_BUS_VOLTAGE_SELECT_FOR_VDD1__3_3V;
        m_pRegisters->powerControl |= REG_POWER_CONTROL__SD_BUS_POWER_FOR_VDD1;
        return;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SdHostStandardController::SetDeviceClock(uint32_t* pOutDeviceClockFrequencyKHz, DeviceClockMode deviceClockMode) NN_NOEXCEPT
{
    // クロックソース 408MHz 想定
    uint16_t n;
    switch (deviceClockMode)
    {
    case DeviceClockMode_Identification:
        // 400kHz 以下
        // 408MHz / 2 / 512 = 398.4375kHz
        n = 512;
        *pOutDeviceClockFrequencyKHz = 399;
        break;
    case DeviceClockMode_SdCardDefaultSpeed:
        // 25MHz 以下
        // 408MHz / 2 / 9 = 22.666...MHz
        n = 9;
        *pOutDeviceClockFrequencyKHz = 25667;
        break;
    case DeviceClockMode_MmcLegacySpeed:
        // 26MHz 以下
        // 408MHz / 2 / 8 = 25.5MHz
        n = 8;
        *pOutDeviceClockFrequencyKHz = 25500;
        break;
    case DeviceClockMode_SdCardHighSpeed:
        // 50MHz 以下
        // 408MHz / 2 / 5 = 40.8MHz
        n = 5;
        *pOutDeviceClockFrequencyKHz = 40800;
        break;
    case DeviceClockMode_MmcHighSpeed:
        // 52MHz 以下
        // 408MHz / 2 / 4 = 51MHz
        n = 4;
        *pOutDeviceClockFrequencyKHz = 51000;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    uint16_t upper_n = 0;
    if (n > 0xFF)
    {
        upper_n = n >> 8;
    }
    uint16_t mask = REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__MASK | REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__MASK;
    m_pRegisters->clockControl = (m_pRegisters->clockControl & (~mask)) | (n << REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__SHIFT) | (upper_n << REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__SHIFT);
}

void SdHostStandardController::EnableDeviceClock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    m_pRegisters->clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
}

void SdHostStandardController::DisableDeviceClock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    m_pRegisters->clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
}

Result SdHostStandardController::IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    if (pTransferData != nullptr)
    {
        if ((pTransferData->m_BlockSize == 0) || (pTransferData->m_NumBlocks == 0))
        {
            // データ転送コマンドで転送サイズが 0 ならば、何もしない
            return ResultSuccess();
        }
    }

    Result result = WaitWhileCommandInhibit((pTransferData != nullptr) || pCommand->m_IsBusy);
    if (result.IsFailure())
    {
        AbortTransaction();
        return result;
    }

    // ステータスクリア（1 を立てたビットだけがクリアされる）
    m_pRegisters->normalInterruptStatus = m_pRegisters->normalInterruptStatus;
    m_pRegisters->errorInterruptStatus = m_pRegisters->errorInterruptStatus;

    uint32_t transferredNumBlocks = 0;
    if (pTransferData != nullptr)
    {
        // データ転送設定（コマンド発行前に設定する必要がある）
        SetTransfer(&transferredNumBlocks, pTransferData);

        // pBuffer を含む領域が CPU アクセスされている可能性を考慮
        nn::dd::FlushDataCache(pTransferData->m_pBuffer, pTransferData->m_BlockSize * transferredNumBlocks);
    }

    // コマンド発行
    m_pRegisters->argument1 = pCommand->m_CommandArgument;
    m_pRegisters->command = GetCommandValue(pCommand, pTransferData != nullptr);
    NN_DETAIL_SDMMC_CMD_LOG("CMD%d 0x%08X\n", pCommand->m_CommandIndex, pCommand->m_CommandArgument);

    // コマンド発行完了待ち
    result = WaitCommandComplete();
    if (result.IsFailure())
    {
        AbortTransaction();
        return result;
    }

    if (pTransferData != nullptr)
    {
        NN_DETAIL_SDMMC_CMD_LOG("Data %u(byte) x %u\n", pTransferData->m_BlockSize, transferredNumBlocks);

        // データ転送完了待ち
        result = WaitTransferComplete();
        if (result.IsFailure())
        {
            AbortTransaction();
            return result;
        }

        if (pTransferData->m_TransferDirection == TransferDirection_ReadFromDevice)
        {
            // 最初の nn::dd::FlushDataCache 以降にキャッシュに取り込まれている可能性を考慮
            nn::dd::InvalidateDataCache(pTransferData->m_pBuffer, pTransferData->m_BlockSize * transferredNumBlocks);
        }

        if (pOutTransferredNumBlocks != nullptr)
        {
            *pOutTransferredNumBlocks = transferredNumBlocks;
        }
    }

    if (pCommand->m_IsBusy || (pTransferData != nullptr))
    {
        // Busy 完了待ち
        result = WaitWhileBusy();
        if (result.IsFailure())
        {
            AbortTransaction();
            return result;
        }
    }

    return ResultSuccess();
}

void SdHostStandardController::GetLastResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);
    NN_ABORT_UNLESS_NOT_NULL(pOutResponse);

    switch (responseType)
    {
    case ResponseType_R1:
        NN_FALL_THROUGH;
    case ResponseType_R3:
        NN_FALL_THROUGH;
    case ResponseType_R6:
        NN_FALL_THROUGH;
    case ResponseType_R7:
        NN_ABORT_UNLESS(responseSize >= sizeof(uint32_t));
        *pOutResponse = m_pRegisters->response10;
        return;
    case ResponseType_R2:
        NN_ABORT_UNLESS(responseSize >= (sizeof(uint32_t) * 4));
        pOutResponse[0] = m_pRegisters->response10;
        pOutResponse[1] = m_pRegisters->response32;
        pOutResponse[2] = m_pRegisters->response54;
        pOutResponse[3] = m_pRegisters->response76;
        return;
    case ResponseType_R0:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SdHostStandardController::GetStopTransmissionResponse(uint32_t* pOutResponse, size_t responseSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pRegisters);

    NN_ABORT_UNLESS(responseSize >= sizeof(uint32_t));
    *pOutResponse = m_pRegisters->response76;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
