﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_BaseDevice.h"
#include <nn/nn_Abort.h>
#include "sdmmc_Api.h"
#include "sdmmc_Timer.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    // Device Status ビット定義
    const uint32_t DEVICE_STATUS__ADDRESS_OUT_OF_RANGE  = 0x1 << 31;
    const uint32_t DEVICE_STATUS__ADDRESS_MISALIGN      = 0x1 << 30;
    const uint32_t DEVICE_STATUS__BLOCK_LEN_ERROR       = 0x1 << 29;
    const uint32_t DEVICE_STATUS__ERASE_SEQ_ERROR       = 0x1 << 28;
    const uint32_t DEVICE_STATUS__ERASE_PARAM           = 0x1 << 27;
    const uint32_t DEVICE_STATUS__WP_VIOLATION          = 0x1 << 26;
    const uint32_t DEVICE_STATUS__DEVICE_IS_LOCKED      = 0x1 << 25;    // Status: カードロック状態 (Clear Condition: A)
    const uint32_t DEVICE_STATUS__LOCK_UNLOCK_FAILED    = 0x1 << 24;
    const uint32_t DEVICE_STATUS__COM_CRC_ERROR         = 0x1 << 23;
    const uint32_t DEVICE_STATUS__ILLEGAL_COMMAND       = 0x1 << 22;
    const uint32_t DEVICE_STATUS__DEVICE_ECC_FAILED     = 0x1 << 21;
    const uint32_t DEVICE_STATUS__CC_ERROR              = 0x1 << 20;
    const uint32_t DEVICE_STATUS__ERROR                 = 0x1 << 19;
    const uint32_t DEVICE_STATUS__CID_CSD_OVERWRITE     = 0x1 << 16;
    const uint32_t DEVICE_STATUS__WP_ERASE_SKIP         = 0x1 << 15;
    const uint32_t DEVICE_STATUS__ERASE_RESET           = 0x1 << 13;
    const uint32_t DEVICE_STATUS__CURRENT_STATE__SHIFT  = 9;
    const uint32_t DEVICE_STATUS__CURRENT_STATE__MASK   = 0xF << DEVICE_STATUS__CURRENT_STATE__SHIFT;
    const uint32_t DEVICE_STATUS__READY_FOR_DATA        = 0x1 << 8;     // Status: 空バッファの送信 (Clear Condition: A)
    const uint32_t DEVICE_STATUS__SWITCH_ERROR          = 0x1 << 7;
    const uint32_t DEVICE_STATUS__EXCEPTIOIN_EVENT      = 0x1 << 6;     // Status: 例外発生 (Clear Condition: A)
    const uint32_t DEVICE_STATUS__APP_CMD               = 0x1 << 5;     // Status: ACMD想定 (Clear Condition: C)

    Result CheckDeviceStatus(uint32_t deviceStatus, IDevice::DeviceType deviceType) NN_NOEXCEPT
    {
        if ((deviceStatus & DEVICE_STATUS__ADDRESS_OUT_OF_RANGE) != 0)
        {
            return ResultDeviceStatusAddressOutOfRange();
        }
        if ((deviceStatus & DEVICE_STATUS__ADDRESS_MISALIGN) != 0)
        {
            return ResultDeviceStatusAddressMisalign();
        }
        if ((deviceStatus & DEVICE_STATUS__BLOCK_LEN_ERROR) != 0)
        {
            return ResultDeviceStatusBlockLenError();
        }
        if ((deviceStatus & DEVICE_STATUS__ERASE_SEQ_ERROR) != 0)
        {
            return ResultDeviceStatusEraseSeqError();
        }
        if ((deviceStatus & DEVICE_STATUS__ERASE_PARAM) != 0)
        {
            return ResultDeviceStatusEraseParam();
        }
        if ((deviceStatus & DEVICE_STATUS__WP_VIOLATION) != 0)
        {
            return ResultDeviceStatusWpViolation();
        }
        if ((deviceStatus & DEVICE_STATUS__LOCK_UNLOCK_FAILED) != 0)
        {
            return ResultDeviceStatusLockUnlockFailed();
        }
        if ((deviceStatus & DEVICE_STATUS__COM_CRC_ERROR) != 0)
        {
            return ResultDeviceStatusComCrcError();
        }
        if ((deviceStatus & DEVICE_STATUS__ILLEGAL_COMMAND) != 0)
        {
            return ResultDeviceStatusIllegalCommand();
        }
        if ((deviceStatus & DEVICE_STATUS__DEVICE_ECC_FAILED) != 0)
        {
            return ResultDeviceStatusDeviceEccFailed();
        }
        if ((deviceStatus & DEVICE_STATUS__CC_ERROR) != 0)
        {
            return ResultDeviceStatusCcError();
        }
        if ((deviceStatus & DEVICE_STATUS__ERROR) != 0)
        {
            return ResultDeviceStatusError();
        }
        if ((deviceStatus & DEVICE_STATUS__CID_CSD_OVERWRITE) != 0)
        {
            return ResultDeviceStatusCidCsdOverwrite();
        }
        if ((deviceStatus & DEVICE_STATUS__WP_ERASE_SKIP) != 0)
        {
            return ResultDeviceStatusWpEraseSkip();
        }
        if ((deviceStatus & DEVICE_STATUS__ERASE_RESET) != 0)
        {
            return ResultDeviceStatusEraseReset();
        }
        if (deviceType == IDevice::DeviceType_Mmc)
        {
            if ((deviceStatus & DEVICE_STATUS__SWITCH_ERROR) != 0)
            {
                return ResultDeviceStatusSwitchError();
            }
        }
        return ResultSuccess();
    }

}

void BaseDevice::SetBusIdentificationMode(IHostController::BusPower* pOutBusPower) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    m_pHostController->DisableDeviceClock();
    m_pHostController->SetBusWidth(IHostController::BusWidth_1Bit);
    if ((m_DeiceType == IDevice::DeviceType_Mmc) && m_pHostController->IsSupportBusPower(IHostController::BusPower_1_8V))
    {
        m_pHostController->SetBusPower(IHostController::BusPower_1_8V);
        if (pOutBusPower != nullptr)
        {
            *pOutBusPower = IHostController::BusPower_1_8V;
        }
    }
    else
    {
        m_pHostController->SetBusPower(IHostController::BusPower_3_3V);
        if (pOutBusPower != nullptr)
        {
            *pOutBusPower = IHostController::BusPower_3_3V;
        }
    }
    m_pHostController->SetDeviceClock(&m_DeviceClockFrequencyKHz, IHostController::DeviceClockMode_Identification);
    m_pHostController->EnableDeviceClock();

    // 1ms + 74 clock cycles 待ち
    WaitMicroseconds(1000);
    WaitClocks(m_DeviceClockFrequencyKHz, 74);
}

void BaseDevice::ChangeDeviceClockMode(IHostController::DeviceClockMode deviceClockMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    m_pHostController->DisableDeviceClock();
    m_pHostController->SetDeviceClock(&m_DeviceClockFrequencyKHz, deviceClockMode);
    m_pHostController->EnableDeviceClock();
}

Result BaseDevice::IssueCommandAndCheckR1(uint32_t commandIndex, uint32_t commandArgument, bool isBusy, DeviceState expectedState) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    ResponseType responseType = ResponseType_R1;

    Command command(commandIndex, commandArgument, responseType, isBusy);
    Result result = m_pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    uint32_t response;
    m_pHostController->GetLastResponse(&response, sizeof(response), responseType);
    result = CheckDeviceStatus(response, m_DeiceType);
    if (result.IsFailure())
    {
        return result;
    }

    if (expectedState != DeviceState_Unknown)
    {
        DeviceState currentState = static_cast<DeviceState>((response & DEVICE_STATUS__CURRENT_STATE__MASK) >> DEVICE_STATUS__CURRENT_STATE__SHIFT);
        if (currentState != expectedState)
        {
            return ResultUnexpectedDeviceState();
        }
    }

    return ResultSuccess();
}

Result BaseDevice::IssueCommandGoIdleState() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    // argument[31:0] stuff bit
    Command command(0, 0, ResponseType_R0, false);
    return m_pHostController->IssueCommand(&command);
}

Result BaseDevice::IssueCommandAllSendCid(uint32_t* pOutCid, size_t cidSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    ResponseType responseType = ResponseType_R2;

    // argument[31:0] stuff bit
    Command command(2, 0, responseType, false);
    Result result = m_pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    m_pHostController->GetLastResponse(pOutCid, cidSize, responseType);

    return ResultSuccess();
}

Result BaseDevice::IssueCommandSelectCard(uint16_t rca) const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    // R1b, このコマンドで状態遷移するため、state チェックは除外する
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    return BaseDevice::IssueCommandAndCheckR1(7, argument, true, DeviceState_Unknown);
}

Result BaseDevice::IssueCommandSendCsd(uint32_t* pOutCsd, size_t csdSize, uint16_t rca) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    ResponseType responseType = ResponseType_R2;

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    Command command(9, argument, responseType, false);
    Result result = m_pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    m_pHostController->GetLastResponse(pOutCsd, csdSize, responseType);

    return ResultSuccess();
}

Result BaseDevice::IssueCommandSendStatus(uint16_t rca) const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    return BaseDevice::IssueCommandAndCheckR1(13, argument, false, DeviceState_Tran);
}

Result BaseDevice::IssueCommandMultipleBlock(uint32_t* pOutTransferredNumBlocks, uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, bool isRead, bool isHighCapacity) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pHostController);

    ResponseType responseType = ResponseType_R1;

    // argument[31:0] data address
    uint32_t commandArgument;
    if (isHighCapacity)
    {
        commandArgument = sectorIndex;
    }
    else
    {
        commandArgument = sectorIndex * SectorSize;
    }
    uint32_t commandIndex;
    TransferDirection transferDirection;
    if (isRead)
    {
        commandIndex = 18;
        transferDirection = TransferDirection_ReadFromDevice;
    }
    else
    {
        commandIndex = 25;
        transferDirection = TransferDirection_WriteToDevice;
    }
    Command command(commandIndex, commandArgument, responseType, false);
    TransferData transferData(pBuffer, SectorSize, numSectors, transferDirection, true);
    Result result = m_pHostController->IssueCommand(&command, &transferData, pOutTransferredNumBlocks);
    if (result.IsFailure())
    {
        return result;
    }

    uint32_t response;
    m_pHostController->GetLastResponse(&response, sizeof(response), responseType);
    result = CheckDeviceStatus(response, m_DeiceType);
    if (result.IsFailure())
    {
        return result;
    }

    m_pHostController->GetStopTransmissionResponse(&response, sizeof(response));
    result = CheckDeviceStatus(response, m_DeiceType);
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

Result BaseDevice::ReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) const NN_NOEXCEPT
{
    if (numSectors == 0)
    {
        // 転送セクタ数が 0 ならば、何もしない
        return ResultSuccess();
    }

    // バッファサイズが十分であることを確認
    NN_ABORT_UNLESS((bufferSize / SectorSize) >= numSectors);

    uint32_t currentSectorIndex = sectorIndex;
    uint32_t restNumSectors = numSectors;
    uint8_t* pCurrentBuffer = reinterpret_cast<uint8_t*>(pBuffer);
    while (true)
    {
        uint32_t transferredNumBlocks;
        Result result = IssueCommandMultipleBlock(&transferredNumBlocks, currentSectorIndex, restNumSectors, pCurrentBuffer, isRead, m_IsHighCapacity);
        if (result.IsFailure())
        {
            // TODO: エラーからの復旧
            return result;
        }

        result = IssueCommandSendStatus(m_Rca);
        if (result.IsFailure())
        {
            // TODO: エラーからの復旧
            return result;
        }


        restNumSectors -= transferredNumBlocks;
        if (restNumSectors <= 0)
        {
            break;  // 完了
        }
        currentSectorIndex += transferredNumBlocks;
        pCurrentBuffer += (SectorSize * transferredNumBlocks);
    }

    return ResultSuccess();
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
