﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "util_ProcessImageMemory.h"
#include "util_KipUtil.h"
#include "test_Common.h"
#include <cstring>
#include <nn/nn_SdkAssert.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>

namespace {

// アーキテクチャに依存しないバリアント
const int LZ_BIT_INDEX      = 12;     // 12bit offset
const int  LZ_BIT_LENGTH    =  4;     //  4bit length
const int  LZ_MAX_INDEX     = (1 << LZ_BIT_INDEX);
const int  LZ_MAX_LENGTH    = (1 << LZ_BIT_LENGTH);
const int  LZ_MIN_COPY      =  3;
void UncompressBackward(void* pBottom)
{
    uint8_t* pBufferBottom = static_cast<uint8_t*>(pBottom);
    uint32_t bufferTopOffset =
        (pBufferBottom[-9] << 24) |
        (pBufferBottom[-10] << 16) |
        (pBufferBottom[-11] << 8)  |
        pBufferBottom[-12];
    uint32_t compressBottomOffset =
        (pBufferBottom[-5] << 24) |
        (pBufferBottom[-6] << 16) |
        (pBufferBottom[-7] << 8)  |
        pBufferBottom[-8];
    uint32_t originalBottomOffset =
        (pBufferBottom[-1] << 24) |
        (pBufferBottom[-2] << 16) |
        (pBufferBottom[-3] << 8)  |
        pBufferBottom[-4];
    uint8_t* bufferTop = pBufferBottom - bufferTopOffset;
    uint8_t* compressBottom = pBufferBottom - compressBottomOffset;
    uint8_t* originalBottom = pBufferBottom + originalBottomOffset;
    uint32_t orignalSize = originalBottom - bufferTop;
    uint32_t compressSize = compressBottom - bufferTop;

    while (orignalSize != 0)
    {
        uint32_t flag = bufferTop[--compressSize];
        uint32_t i;

        for (i = 0; i < 8; i++)
        {
            if (flag & 0x80) // 圧縮データか？
            {
                uint16_t half;
                uint32_t len;
                uint32_t index;
                uint32_t i;

                // 展開長を計算
                compressSize -= 2;
                half = bufferTop[compressSize] | (bufferTop[compressSize + 1] << 8);
                len = ((half >> LZ_BIT_INDEX) & (LZ_MAX_LENGTH - 1)) + LZ_MIN_COPY;
                index = (half & (LZ_MAX_INDEX - 1)) + LZ_MIN_COPY;
                if (orignalSize < len)
                {
                    len = orignalSize;
                }
                orignalSize -= len;

                for (i = 0; i < len; i++)
                {
                    bufferTop[orignalSize + i] = bufferTop[orignalSize + index + i];
                }
            }
            else
            {
                bufferTop[--orignalSize] = bufferTop[--compressSize];
            }
            flag <<= 1;
            if (orignalSize == 0)
            {
                return;
            }
        }
    }
}

nn::Result SearchFreeRegion(uintptr_t* pOut, size_t mappingSize) NN_NOEXCEPT
{
    uintptr_t addr = NN_SVC_ADDR_SMALL_MAP_BEGIN;
    for (;;)
    {
        nn::svc::MemoryInfo  memoryInfo;
        nn::svc::PageInfo   pageInfo;
        auto result = nn::svc::QueryMemory(&memoryInfo, &pageInfo, addr);
        if (result.IsFailure())
        {
            return result;
        }
        if (memoryInfo.state == nn::svc::MemoryState_Free && ((memoryInfo.baseAddress + memoryInfo.size) - addr) >= mappingSize)
        {
            *pOut = addr;
            return nn::ResultSuccess();
        }
        if (memoryInfo.state == nn::svc::MemoryState_Inaccessible)
        {
            return nn::svc::ResultOutOfMemory();
        }
        if (memoryInfo.baseAddress + memoryInfo.size < addr)
        {
            return nn::svc::ResultOutOfMemory();
        }
        if (memoryInfo.baseAddress + memoryInfo.size >= NN_SVC_ADDR_SMALL_MAP_END)
        {
            return nn::svc::ResultOutOfMemory();
        }
        addr = memoryInfo.baseAddress + memoryInfo.size;
    }
}
} // namespace


void ProcessImageMemory::Map(
        nn::svc::Handle handle, const KipFormat& kip, uint64_t base)
{
    nn::Result result;

    if (kip.GetExBinarySize())
    {
        size_t size =  PageToSize(SizeToPage(kip.GetExMemorySize()));
        uintptr_t addr = 0;

        result = SearchFreeRegion(&addr, size);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapProcessMemory(
                addr, handle, (base - kip.GetExAddress()) + kip.GetExAddress(), size);
        NN_ASSERT_RESULT_SUCCESS(result);
        m_pEx = reinterpret_cast<void*>(addr);
        m_ExSize = size;
        std::memset(m_pEx, 0, m_ExSize);
        m_MapSize += size;
    }

    if (kip.GetRoBinarySize())
    {
        size_t size =  PageToSize(SizeToPage(kip.GetRoMemorySize()));
        uintptr_t addr = 0;

        result = SearchFreeRegion(&addr, size);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapProcessMemory(
                addr, handle, (base - kip.GetExAddress()) + kip.GetRoAddress(), size);
        NN_ASSERT_RESULT_SUCCESS(result);
        m_pRo = reinterpret_cast<void*>(addr);
        m_RoSize = size;
        std::memset(m_pRo, 0, m_RoSize);
        m_MapSize += size;
    }

    if (kip.GetRwBinarySize())
    {
        size_t size =  PageToSize(SizeToPage(kip.GetRwMemorySize()));
        uintptr_t addr = 0;

        result = SearchFreeRegion(&addr, size);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapProcessMemory(
                addr, handle, (base - kip.GetExAddress()) + kip.GetRwAddress(), size);
        NN_ASSERT_RESULT_SUCCESS(result);
        m_pRw = reinterpret_cast<void*>(addr);
        m_RwSize = size;
        std::memset(m_pRw, 0, m_RwSize);
        m_MapSize += size;
    }
}

void ProcessImageMemory::Unmap(
        nn::svc::Handle handle, const KipFormat& kip, uint64_t base)
{
    if (m_pEx)
    {
        nn::Result result = nn::svc::UnmapProcessMemory(
                reinterpret_cast<uintptr_t>(m_pEx), handle,
                (base - kip.GetExAddress()) + kip.GetExAddress(), m_ExSize);
        NN_ASSERT_RESULT_SUCCESS(result);
        m_pEx = NULL;
    }
    if (m_pRo)
    {
        nn::svc::UnmapProcessMemory(
                reinterpret_cast<uintptr_t>(m_pRo), handle,
                (base - kip.GetExAddress()) + kip.GetRoAddress(), m_RoSize);
        m_pRo = NULL;
    }

    if (m_pRw)
    {
        nn::svc::UnmapProcessMemory(
                reinterpret_cast<uintptr_t>(m_pRw), handle,
                (base - kip.GetExAddress()) + kip.GetRwAddress(), m_RwSize);
        m_pRw = NULL;
    }
}

void ProcessImageMemory::SetPermission(
        nn::svc::Handle handle, const KipFormat& kip, uint64_t base)
{
    nn::Result result;

    uintptr_t argumentAddr = 0;
    if (kip.GetExMemorySize())
    {
        result = nn::svc::SetProcessMemoryPermission(
                handle, (base - kip.GetExAddress()) + kip.GetExAddress(),
                PageToSize(SizeToPage(kip.GetExMemorySize())),
                nn::svc::MemoryPermission_ReadExecute);
        NN_ASSERT_RESULT_SUCCESS(result);
        argumentAddr = (base - kip.GetExAddress()) + kip.GetExAddress() + PageToSize(SizeToPage(kip.GetExMemorySize()));
    }
    if (kip.GetRoMemorySize())
    {
        result = nn::svc::SetProcessMemoryPermission(
                handle, (base - kip.GetExAddress()) + kip.GetRoAddress(),
                PageToSize(SizeToPage(kip.GetRoMemorySize())),
                nn::svc::MemoryPermission_Read);
        NN_ASSERT_RESULT_SUCCESS(result);
        argumentAddr = (base - kip.GetExAddress()) + kip.GetRoAddress() + PageToSize(SizeToPage(kip.GetRoMemorySize()));
    }
    if (kip.GetRwMemorySize() || kip.GetBssMemorySize())
    {
        uint64_t startAddr = kip.GetRwMemorySize()
            ? (base - kip.GetExAddress()) + kip.GetRwAddress()
            : (base - kip.GetExAddress()) + kip.GetBssAddress();
        uint64_t endAddr   = kip.GetBssMemorySize()
            ? (base - kip.GetExAddress()) + kip.GetBssAddress() + kip.GetBssMemorySize()
            : (base - kip.GetExAddress()) + kip.GetRwAddress() + kip.GetRwMemorySize();
        result = nn::svc::SetProcessMemoryPermission(
                handle, startAddr, PageToSize(SizeToPage(endAddr - startAddr)),
                nn::svc::MemoryPermission_ReadWrite);
        NN_ASSERT_RESULT_SUCCESS(result);
        argumentAddr = startAddr + PageToSize(SizeToPage(endAddr - startAddr));
    }

    // Argument の設定
    size_t size = PageToSize(SizeToPage(ArgumentSize));
    result = nn::svc::SetProcessMemoryPermission(
            handle, argumentAddr, size,
            nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr = 0;
    result = SearchFreeRegion(&addr, size);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(
            addr, handle, argumentAddr, size);
    NN_ASSERT_RESULT_SUCCESS(result);

    std::memset(reinterpret_cast<void*>(addr), 0, size);

    result = nn::svc::UnmapProcessMemory(
            addr, handle, argumentAddr, size);
    NN_ASSERT_RESULT_SUCCESS(result);

}

void ProcessImageMemory::LoadImage(
        nn::svc::Handle handle, const char *pFile, size_t fileSize,
        const KipFormat& kip, uint64_t base)
{
    nn::Result result;

    Map(handle, kip, base);

    int64_t offset = sizeof(kip);

    if (kip.GetExBinarySize())
    {
        NN_ASSERT(static_cast<size_t>(offset + kip.GetExBinarySize()) <= fileSize);
        NN_ASSERT(kip.GetExBinarySize() <= m_ExSize);
        ::std::memcpy(m_pEx, pFile + offset, kip.GetExBinarySize());
        if (kip.IsExCompressed())
        {
            UncompressBackward(static_cast<uint8_t*>(m_pEx) + kip.GetExBinarySize());
        }
        offset += kip.GetExBinarySize();
    }

    if (kip.GetRoBinarySize())
    {
        NN_ASSERT(static_cast<size_t>(offset + kip.GetRoBinarySize()) <= fileSize);
        NN_ASSERT(kip.GetRoBinarySize() <= m_RoSize);
        ::std::memcpy(m_pRo, pFile + offset, kip.GetRoBinarySize());
        if (kip.IsRoCompressed())
        {
            UncompressBackward(static_cast<uint8_t*>(m_pRo) + kip.GetRoBinarySize());
        }
        offset += kip.GetRoBinarySize();
    }

    if (kip.GetRwBinarySize())
    {
        NN_ASSERT(static_cast<size_t>(offset + kip.GetRwBinarySize()) <= fileSize);
        NN_ASSERT(kip.GetRwBinarySize() <= m_RwSize);
        ::std::memcpy(m_pRw, pFile + offset, kip.GetRwBinarySize());
        if (kip.IsRwCompressed())
        {
            UncompressBackward(static_cast<uint8_t*>(m_pRw) + kip.GetRwBinarySize());
        }
        offset += kip.GetRwBinarySize();
    }
}

