﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "test_TestTmrDevice_Jetson.h"
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_Tcb.h>

namespace {

const int64_t SleepTime = 100 * 1000 * 1000;
#ifdef INVALID_POINTER_TEST
const int tmpVar = 0;
#endif

nn::Bit32 g_ServerBuffer[0x2000 / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
nn::Bit32 g_ClientBuffer[0x2000 / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));

char g_Buffer[2][0x2000] __attribute__((aligned(0x1000)));
int32_t g_SequenceCount[2];
bool g_Start;

struct TestData
{
    nn::svc::Handle readableEvent;
    nn::svc::Handle writableEvent;
    int index;
};

void TestEventThread(TestData *testData)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t signalIndex;
    nn::svc::Handle handles[1] = { testData->readableEvent };

    g_SequenceCount[testData->index] = 1;

    result = nn::svc::SignalEvent(testData->writableEvent);
    ASSERT_RESULT_SUCCESS(result);

    while(g_SequenceCount[testData->index] != 2)
    {
        nn::svc::SleepThread(SleepTime);
    }

    g_SequenceCount[testData->index] = 3;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
    g_SequenceCount[testData->index] = 4;
}

void SendAsyncIpcThread(nn::svc::Handle* clientSession)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle eventHandle;
    nn::svc::Handle handles[1];
    uintptr_t pClientBuffer = reinterpret_cast<uintptr_t>(g_ClientBuffer);
    size_t clientBufferSize = sizeof(g_ClientBuffer);
    int32_t signalIndex;
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pClientBuffer);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);

    result = nn::svc::SendAsyncRequestWithUserBuffer(
        &eventHandle, pClientBuffer, clientBufferSize, *clientSession);
    ASSERT_RESULT_SUCCESS(result);

    handles[0] = eventHandle;

    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    result = nn::svc::CloseHandle(eventHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void WaitAndAcceptThread(nn::svc::Handle* handle)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle handles[1] = { *handle };
    int32_t signalIndex;

    // TEST 21-9, 21-10, 21-11, 21-12
    // ポートへの接続を待つことが出来る
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(signalIndex == 0);

    result = nn::svc::AcceptSession(&serverSession, *handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    while (!g_Start)
    {
        nn::svc::SleepThread(SleepTime);
    }
}

void WaitToCancelThread(nn::svc::Handle* readableEvent)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle handles[1] = { *readableEvent };
    int32_t signalIndex;

    // TEST 21-27 (同じコア), 21-28 (違うコア)
    // numHandles が 0 の時はシグナルを受け取らない
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 0, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());

    // 二度目numHandles が指定されていれば受け取る
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void WaitToTimeoutThread(nn::svc::Handle* readableEvent)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle handles[1] = { *readableEvent };
    int32_t signalIndex;

    // TEST 21-28 (同じコア), 21-29 (違うコア)
    // numHandles が 0 の時はシグナルを受け取らない
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 0, SleepTime);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // 二度目numHandles が指定されていれば受け取る
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

extern "C" void nnMain();

TEST(WaitSynchronization, WaitMultipleEventsTest)
{
    TestData testData[2];
    nn::Result result;
    nn::svc::Handle eventHandles[2];
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestEventThread);;
    uintptr_t sp[2] = {
        reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]),
        reinterpret_cast<uintptr_t>(&g_Buffer[1]) + sizeof(g_Buffer[1])
    };

    testData[0].index = 0;
    testData[1].index = 1;

    for (int i = 0; i < 2; i++)
    {
        result = nn::svc::CreateEvent(&testData[i].writableEvent, &testData[i].readableEvent);
        ASSERT_RESULT_SUCCESS(result);
        eventHandles[i] = testData[i].readableEvent;
    }


    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority - 1;
                priority++)
        {
            nn::svc::Handle threads[2];
            bool checkFlag[2] = {false, false};

            result = nn::svc::CreateThread(
                    &threads[0], pc, reinterpret_cast<uintptr_t>(&testData[0]),
                    sp[0], priority, 0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CreateThread(
                    &threads[1], pc, reinterpret_cast<uintptr_t>(&testData[1]),
                    sp[1], priority + 1, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            g_SequenceCount[0] = 0;
            g_SequenceCount[1] = 0;

            result = nn::svc::StartThread(threads[0]);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::StartThread(threads[1]);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 21-1 (同じコア), TEST 21-2 (違うコア)
            // 複数のイベントを待つことが出来る
            int32_t signalIndex;
            for(int32_t i = 0; i < 2; i++)
            {
                result = nn::svc::WaitSynchronization(&signalIndex, eventHandles, 2, -1);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(0 <= signalIndex && signalIndex < 2);
                ASSERT_TRUE(!checkFlag[signalIndex]);
                checkFlag[signalIndex] = true;
                ASSERT_TRUE(g_SequenceCount[signalIndex] == 1);
                result = nn::svc::ClearEvent(eventHandles[signalIndex]);
                ASSERT_RESULT_SUCCESS(result);
                g_SequenceCount[signalIndex] = 2;
            }

            // TEST 21-3 (同じコア), TEST 21-4 (違うコア)
            // イベント待ちをキャンセルさせることが出来る
            result = nn::svc::CancelSynchronization(threads[0]);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CancelSynchronization(threads[1]);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 21-5 (同じコア), TEST 21-6 (違うコア)
            // スレッドの終了を待つことが出来る
            result = nn::svc::WaitSynchronization(&signalIndex, threads, 2, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(0 <= signalIndex && signalIndex < 2);
            ASSERT_TRUE(checkFlag[signalIndex]);
            checkFlag[signalIndex] = false;
            result = nn::svc::CloseHandle(threads[signalIndex]);
            ASSERT_RESULT_SUCCESS(result);
            threads[0] = signalIndex == 0 ? threads[1] : threads[0];
            threads[1] = nn::svc::INVALID_HANDLE_VALUE;

            result = nn::svc::WaitSynchronization(&signalIndex, threads, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(0 == signalIndex);
            result = nn::svc::CloseHandle(threads[signalIndex]);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    for (int i = 0; i < 2; i++)
    {
        result = nn::svc::CloseHandle(testData[i].writableEvent);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(testData[i].readableEvent);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(WaitSynchronization, WaitReadableEventForIpcTest)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle eventHandle;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pServerBuffer = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    size_t serverBufferSize = sizeof(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pServerBuffer);
    int32_t signalIndex;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendAsyncIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]);

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    handles[0] = serverSession;

    // TEST 21-7 (同じコア), 21-8 (違うコア)
    // IPC の読み込みイベントを待つことが出来る
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            result = nn::svc::CreateThread(
                    &handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
                    sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &signalIndex, pServerBuffer, serverBufferSize,
                    handles, sizeof(handles) / sizeof(*handles),
                    nn::svc::INVALID_HANDLE_VALUE, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

            SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &signalIndex, pServerBuffer, serverBufferSize, handles, 0,
                    serverSession, 0);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

            int32_t waitIndex;
            result = nn::svc::WaitSynchronization(&waitIndex, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, WaitNamedPortTest)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientSession;
    const char* PortName = "Test2109";
    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitAndAcceptThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer[0]) + sizeof(g_Buffer[0]);
    int32_t waitIndex;

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-9 (同じコア), 21-10 (違うコア)
    // 名前付きポートへの接続を待つことが出来る
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {

            result = nn::svc::CreateThread(&handle, pc,
                    reinterpret_cast<uintptr_t>(&serverPort),
                    sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&waitIndex, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(clientSession);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, WaitPortTest)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitAndAcceptThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]);
    int32_t waitIndex;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-11 (同じコア), 21-12 (違うコア)
    // ポートへの接続を待つことが出来る
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {

            result = nn::svc::CreateThread(&handle, pc,
                    reinterpret_cast<uintptr_t>(&serverPort),
                    sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::ConnectToPort(&clientSession, clientPort);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&waitIndex, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(clientSession);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, InvalidPseudoHandleTest)
{
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t signalIndex;

    // TEST 21-14
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 21-15
    handles[0] = nn::svc::PSEUDO_HANDLE_CURRENT_THREAD;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 21-16
    handles[0] = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(WaitSynchronization, CloseThreadHandleTest)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle handles[1];
    int32_t signalIndex;
    uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]);

    result = nn::svc::CreateThread(&handle, pc, 0, sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-17
    // クローズしたスレッドのハンドルは受け付けない
    handles[0] = handle;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(WaitSynchronization, CloseCreateEventHandleTest)
{
    nn::Result result;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    nn::svc::Handle handles[1];
    int32_t signalIndex;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 20-18
    // クローズした書き込みイベントは受け付けない
    handles[0] = writableEvent;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 20-19
    // クローズした読み込みイベントは受け付けない
    handles[0] = readableEvent;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(WaitSynchronization, CloseReadEventForIpcTest)
{
    nn::Result result;
    nn::svc::Handle eventHandle;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    size_t size = sizeof(g_ServerBuffer);
    nn::svc::Handle handles[1];
    int32_t signalIndex;

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(eventHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-20
    // クローズした読み込みイベントは受け付けない
    handles[0] = eventHandle;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, CloseNamedPortTest)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    const char* PortName = "Test2121";
    nn::svc::Handle handles[1];
    int32_t signalIndex;

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-21
    // クローズしたサーバーポートは受け付けない
    handles[0] = serverPort;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, ClosePortTest)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle handles[1];
    int32_t signalIndex;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-22
    // クローズしたサーバーポートは受け付けない
    handles[0] = serverPort;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 21-23
    // クローズしたクライアントポートは受け付けない
    handles[0] = clientPort;
    result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(WaitSynchronization, TimeoutTest)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle handles[1];
    uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]);
    int32_t signalIndex;

    // TEST 21-25 (同じコア), 21-26 (違うコア)
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            g_Start = false;
            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            handles[0] = handle;
            volatile int64_t before = nn::svc::GetSystemTick();
            result = nn::svc::WaitSynchronization(&signalIndex, handles, 1, SleepTime);
            volatile int64_t after = nn::svc::GetSystemTick();
            ASSERT_TRUE(
                    ((after - before) / (NN_HW_TICKS_PER_SECOND / (1000 * 1000)) * 1000)
                    >= SleepTime);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
            g_Start = true;

            result = nn::svc::WaitSynchronization(&signalIndex, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(WaitSynchronization, NumHandlesTestWithCancel)
{
    nn::Result result;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitToCancelThread);;
    uintptr_t sp = reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]);
    int32_t waitIndex;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            nn::svc::Handle thread;

            result = nn::svc::CreateThread(
                    &thread, pc, reinterpret_cast<uintptr_t>(&readableEvent),
                    sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 21-27 (同じコア), TEST 21-28 (違うコア)
            // numHandles が 0 の時はシグナルを受信せず、キャンセルで起きる
            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(thread);
            ASSERT_RESULT_SUCCESS(result);

            // スレッドが寝るのを待つ
            nn::svc::SleepThread(SleepTime);

            result = nn::svc::CancelSynchronization(thread);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&waitIndex, &thread, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(thread);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, NumHandlesTestWithTimeout)
{
    nn::Result result;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitToTimeoutThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(&g_Buffer[0]) + sizeof(g_Buffer[0]);
    int32_t waitIndex;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            nn::svc::Handle thread;

            result = nn::svc::CreateThread(
                    &thread, pc, reinterpret_cast<uintptr_t>(&readableEvent), sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 21-29 (同じコア), TEST 21-30 (違うコア)
            // numHandles が 0 の時はシグナルを受信せず、タイムアウトで起きる
            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(thread);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&waitIndex, &thread, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(thread);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, NumHandlesTest)
{
    nn::Result result;
    int32_t waitIndex;

    TestHeap heap(sizeof(nn::svc::Handle) * (nn::svc::ArgumentHandleCountMax + 1) * 2);
    nn::svc::Handle* writeHandles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
    nn::svc::Handle* readHandles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress() + sizeof(nn::svc::Handle) * (nn::svc::ArgumentHandleCountMax + 1));

    for (int32_t i = 0; i <= nn::svc::ArgumentHandleCountMax; i++)
    {
        result = nn::svc::CreateEvent(&writeHandles[i], &readHandles[i]);
        ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::svc::WaitSynchronization(
            &waitIndex, readHandles, nn::svc::ArgumentHandleCountMax, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::WaitSynchronization(
            &waitIndex, readHandles, nn::svc::ArgumentHandleCountMax + 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    for (int32_t i = 0; i <= nn::svc::ArgumentHandleCountMax; i++)
    {
        result = nn::svc::CloseHandle(writeHandles[i]);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readHandles[i]);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(WaitSynchronization, LightSessionTest)
{
    nn::Result result;

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t waitIndex;
    // TEST 21-44
    // Light セッションを待つことが出来ない
    result = nn::svc::WaitSynchronization(&waitIndex, &serverSession, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::WaitSynchronization(&waitIndex, &clientSession, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

// クライアントセッションは受け付けない
TEST(WaitSynchronization, ClientSessionTest)
{
    nn::Result result;

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    int32_t waitIndex;

    // TEST 21-47
    // NamedPort
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;
        const char* PortName = "TestWait";

        result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);

        result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::WaitSynchronization(&waitIndex, &clientSession, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::WaitSynchronization(&waitIndex, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle serverSession;
        result = nn::svc::AcceptSession(&serverSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);

        result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 21-48
    // Port
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::WaitSynchronization(&waitIndex, &clientSession, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::WaitSynchronization(&waitIndex, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle serverSession;
        result = nn::svc::AcceptSession(&serverSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
    }

    // TEST 21-49
    // Session
    {
        result = nn::svc::CreateSession(&serverSession, &clientSession, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::WaitSynchronization(&waitIndex, &clientSession, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }
}

TEST(WaitSynchronization, WritableEvent)
{
    nn::Result result;
    int32_t waitIndex;

    nn::svc::Handle writableHandle;
    nn::svc::Handle readableHandle;
    result = nn::svc::CreateEvent(&writableHandle, &readableHandle);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableHandle);
    AutoHandleClose rEventCloser(readableHandle);

    // TEST 21-50
    // 書き込みイベントは受け付けない
    result = nn::svc::WaitSynchronization(&waitIndex, &writableHandle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
TEST(WaitSynchronization, InterruptEvent)
{
    nn::Result result;
    nn::svc::InterruptType type = nn::svc::InterruptType_Level;
    int32_t name = TmrDefaultInterruptNumber;
    nn::svc::Handle handle;

    result = nn::svc::CreateInterruptEvent(&handle, name, type);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(handle);

    TestTmrDriver tmr;
    tmr.SetTimer(100);
    tmr.StartTimer();

    // TEST 21-13
    // CreateInterruptEvent の読み込みイベントを待つことが出来る
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, 100 * 1000 * 1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-24
    // Close した CreateInterruptEvent の読み込みイベントを渡すと失敗する
    eventCloser.Close();
    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}
#endif

TEST(WaitSynchronization, InvalidHandleTest)
{
    nn::Result result;
    nn::svc::Handle handles[3];
    int32_t waitIndex;

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    nn::svc::Handle writableHandle;
    nn::svc::Handle readableHandle;
    result = nn::svc::CreateEvent(&writableHandle, &readableHandle);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableHandle);
    AutoHandleClose rEventCloser(readableHandle);


    handles[0] = serverSession;
    handles[1] = readableHandle;
    result = nn::svc::WaitSynchronization(&waitIndex, handles, 2, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 21-45
    // 不正なハンドルが紛れていると失敗する
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    handles[1] = serverSession;
    handles[2] = readableHandle;
    result = nn::svc::WaitSynchronization(&waitIndex, handles, 3, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    handles[0] = serverSession;
    handles[1] = nn::svc::INVALID_HANDLE_VALUE;
    handles[2] = readableHandle;
    result = nn::svc::WaitSynchronization(&waitIndex, handles, 3, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    handles[0] = serverSession;
    handles[1] = readableHandle;
    handles[2] = nn::svc::INVALID_HANDLE_VALUE;
    result = nn::svc::WaitSynchronization(&waitIndex, handles, 3, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

}

#ifdef INVALID_POINTER_TEST
TEST(WaitSynchronization, InvalidPointerTest)
{
    nn::Result result;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    nn::svc::Handle handles[1];

    // 前準備
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // シグナルを受けてからメモリのチェックをするのであればいる
    nn::svc::SignalEvent(writableEvent);
    handles[0] = readableEvent;

#ifdef INVALID_POINTER_TEST
    // TEST 21-31
    // NULL を与えると失敗する
    result = nn::svc::WaitSynchronization(NULL, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 21-32
    // 自然数に整列されていないアドレスを与えると失敗する
    // 未実装

#ifdef INVALID_POINTER_TEST
    // TEST 21-33
    // MemoryPermission_None を与えると失敗する
    result = nn::svc::WaitSynchronization(
            reinterpret_cast<int32_t*>(g_FreeAreaBegin), handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 21-34
    // MemoryPermission_Read を与えると失敗する
    uintptr_t addr = reinterpret_cast<uintptr_t>(&tmpVar);
    result = nn::svc::WaitSynchronization(reinterpret_cast<int32_t*>(addr), handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 21-35
    // MemoryPermission_ReadExecute を与えると失敗する
    result = nn::svc::WaitSynchronization(reinterpret_cast<int32_t*>(nnMain), handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 21-36
    // MemoryPermission_None を与えると失敗する
    int32_t signalIndex;
    result = nn::svc::WaitSynchronization(
            &signalIndex, reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin), 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // TEST 21-37
    // handles のアドレスと numHandles の値を組み合わせると、
    // オーバーフローが起きるアドレス場合は、失敗する
    uintptr_t tmpAddr = static_cast<uintptr_t>(-1);
    result = nn::svc::WaitSynchronization(
            &signalIndex, reinterpret_cast<nn::svc::Handle*>(tmpAddr), 2, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // 後処理
    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);
}
#endif // INVALID_POINTER_TEST

