﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>

namespace {
const nn::svc::MemoryPermission ValidPerm[] = {
    nn::svc::MemoryPermission_ReadWrite,
    nn::svc::MemoryPermission_Read,
    nn::svc::MemoryPermission_None,
};
} // namespace

// TEST 128-31
// マージされたブロックに対しても、解除はマップしたときの単位でしか出来ない
TEST(UnmapTransferMemory, MergeBlockTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[2];
    nn::svc::MemoryInfo blockInfo;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];

        TestHeap heap(HeapAlign);
        size_t size = 0x1000;
        uintptr_t fromAddr = heap.GetAddress();

        GetMemoryInfo(&blockInfo, fromAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= fromAddr);
        ASSERT_TRUE(blockInfo.size >= HeapAlign);

        result = nn::svc::CreateTransferMemory(&handles[0], fromAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser1(handles[0]);

        GetMemoryInfo(&blockInfo, fromAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == perm);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        ASSERT_TRUE(blockInfo.baseAddress == fromAddr);
        ASSERT_TRUE(blockInfo.size == size);

        result = nn::svc::CreateTransferMemory(&handles[1], fromAddr + size, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser2(handles[1]);

        GetMemoryInfo(&blockInfo, fromAddr + size);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == perm);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        ASSERT_TRUE(blockInfo.baseAddress == fromAddr);
        ASSERT_TRUE(blockInfo.size == size * 2);

        uintptr_t toAddr = g_FreeAreaBegin;

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= toAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= toAddr + 2 * size - 1);

        result = nn::svc::MapTransferMemory(handles[0], toAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, toAddr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress == toAddr);
        ASSERT_TRUE(blockInfo.size == size);

        result = nn::svc::MapTransferMemory(handles[1], toAddr + size, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, toAddr + size);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress == toAddr);
        ASSERT_TRUE(blockInfo.size == size * 2);

        // マージされたブロックを解除できない
        result = nn::svc::UnmapTransferMemory(handles[0], toAddr, size * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        result = nn::svc::UnmapTransferMemory(handles[1], toAddr, size * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        GetMemoryInfo(&blockInfo, toAddr + size);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress == toAddr);
        ASSERT_TRUE(blockInfo.size == size * 2);

        result = nn::svc::UnmapTransferMemory(handles[0], toAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= toAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size == toAddr + size);

        GetMemoryInfo(&blockInfo, toAddr + size);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress == toAddr + size);
        ASSERT_TRUE(blockInfo.size == size);

        result = nn::svc::UnmapTransferMemory(handles[1], toAddr + size, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, toAddr + size);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= toAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= toAddr + 2 * size - 1);
    }
} // NOLINT(readability/fn_size)

// TEST 128-32
// handle を使ってマップしたTransferMemory の領域とは違う領域を指定することは出来ない
TEST(UnmapTransferMemory, InvalidCombinationTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[2];

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];

        TestHeap heap(HeapAlign);
        size_t size = 0x1000;
        uintptr_t fromAddr = heap.GetAddress();

        result = nn::svc::CreateTransferMemory(&handles[0], fromAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser1(handles[0]);

        result = nn::svc::CreateTransferMemory(&handles[1], fromAddr + size, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser2(handles[1]);

        uintptr_t toAddr = g_FreeAreaBegin;

        result = nn::svc::MapTransferMemory(handles[0], toAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::MapTransferMemory(handles[1], toAddr + size, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        // handle とマップの関係が一致していないと失敗する
        result = nn::svc::UnmapTransferMemory(handles[0], toAddr + size, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

        result = nn::svc::UnmapTransferMemory(handles[1], toAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

        result = nn::svc::UnmapTransferMemory(handles[0], toAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::UnmapTransferMemory(handles[1], toAddr + size, size);
        ASSERT_RESULT_SUCCESS(result);
    }
}

