﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

namespace {
} // namespace


TEST(UnmapProcessCodeMemory, NormalCase)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_Free, nn::svc::MemoryPermission_None, 0);
    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_ReadWrite, 0);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_None,
            nn::svc::MemoryAttribute_Locked);

    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_AliasCode, nn::svc::MemoryPermission_None, 0);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-51
    // マップを解放した fromAddr の領域が MemoryPermission_ReadWrite である
    // TEST 120-52
    // マップを解放した fromAddr の領域が MemoryState_Normal である
    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_ReadWrite, 0);

    // TEST 120-53
    // マップを解放した toAddr の領域が MemoryPermission_None である
    // TEST 120-54
    // マップを解放した toAddr の領域が MemoryState_Free である
    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_Free, nn::svc::MemoryPermission_None, 0);
}

TEST(UnmapProcessCodeMemory, MergeBlockTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size * 2);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr + size, fromAddr + size, size);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, fromAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == fromAddr);
    ASSERT_TRUE(blockInfo.size == size * 2);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, toAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == toAddr);
    ASSERT_TRUE(blockInfo.size == size * 2);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_AliasCode);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 120-55
    // 連続した領域をマップされ、マージされたブロックを一気に解放出来る
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size * 2);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessCodeMemory, InvalidAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    // TEST 120-56
    // マップされていない領域を指定すると失敗する
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
}

TEST(UnmapProcessCodeMemory, InvalidCombinationTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size * 2);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // 1つ分ページを飛ばしてマップ
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr + size * 2, fromAddr + size * 2, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-57
    // fromAddr と toAddr が違う組み合わせだと失敗する
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr + size * 2, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr + size * 2, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    // 終了処理
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr + size * 2, fromAddr + size * 2, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessCodeMemory, ReleasePartOfMappedAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size * 2);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size * 2);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-58
    // マップされている領域の一部分を解放することが出来る
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr + size, fromAddr + size, size);
    ASSERT_RESULT_SUCCESS(result);
}

