﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_TestSvc.h"
#include "test_TestCondition.h"
#include "util_TestResource.h"

namespace {
const uintptr_t TmrPhysicalBaseAddr = 0x60005000;
const int32_t DefaultTmrNo = 1;
const int32_t TmrOffset[] =
{
    0x88, // TMR0
    0x00, // TMR1
    0x08, // TMR2
    0x50, // TMR3
    0x58, // TMR4
    0x60, // TMR5
    0x68, // TMR6
    0x70, // TMR7
    0x78, // TMR8
    0x80, // TMR9
    0x90, // TMR10
    0x98, // TMR11
    0xa0, // TMR12
    0xa8, // TMR13
};

const int32_t InterruptOffset = 32;
const int32_t TmrInterruptNumber[] =
{
    156 + InterruptOffset, // TMR0
    0   + InterruptOffset, // TMR1
    1   + InterruptOffset, // TMR2
    41  + InterruptOffset, // TMR3
    42  + InterruptOffset, // TMR4
    121 + InterruptOffset, // TMR5
    152 + InterruptOffset, // TMR6
    153 + InterruptOffset, // TMR7
    154 + InterruptOffset, // TMR8
    155 + InterruptOffset, // TMR9
    176 + InterruptOffset, // TMR10
    177 + InterruptOffset, // TMR11
    178 + InterruptOffset, // TMR12
    179 + InterruptOffset, // TMR13
};
const int32_t TmrDefaultInterruptNumber = TmrInterruptNumber[DefaultTmrNo];

class TestTmrDriver
{
    public:
        explicit TestTmrDriver(int32_t tmrNo = DefaultTmrNo)
            : m_Msec(100), m_InterruptNumber(tmrNo)
        {
            NN_ASSERT(tmrNo >= 0 && tmrNo <= 13);
            m_InterruptNumber = TmrInterruptNumber[tmrNo];
            nn::Result result = nn::svc::QueryIoMapping(&m_BaseAddr, TmrPhysicalBaseAddr, 0x1000);
            NN_ASSERT_RESULT_SUCCESS(result);
            m_pPtv = reinterpret_cast<int32_t*>(m_BaseAddr + TmrOffset[tmrNo]);
            m_pPcr = reinterpret_cast<int32_t*>(m_BaseAddr + 0x4 + TmrOffset[tmrNo]);
            m_pUsecCfg = reinterpret_cast<int32_t*>(m_BaseAddr + 0x10 + 0x4 + TmrOffset[tmrNo]);
            *m_pPtv = 0;
            ClearInterrupt();
        }

        ~TestTmrDriver()
        {
            *m_pPtv = 0;
        }

        void SetTimer(uint32_t msec)
        {
            NN_ASSERT(msec < (1 << 28));
            m_Msec = msec;
        }

        void StartTimer()
        {
            *m_pUsecCfg = 0xb;
            *m_pPtv = m_Msec;
            *m_pPtv |= 1 << 31; // Enable Timer
        }

        void ClearInterrupt()
        {
            *m_pPcr |= 1 << 30; // clear interrupt
        }

        bool IsUsedTimer()
        {
            return *m_pPtv != 0;
        }

        int32_t GetInterruptNumber()
        {
            return m_InterruptNumber;
        }

    private:
        uintptr_t m_BaseAddr;
        volatile int32_t* m_pPtv;
        volatile int32_t* m_pPcr;
        volatile int32_t* m_pUsecCfg;
        uint32_t m_Msec;
        int32_t m_InterruptNumber;

};
}

