﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <cstring>

#if defined NN_BUILD_CONFIG_COMPILER_GCC
#pragma GCC optimize("O0")
#endif
extern "C" int TestSetJmp(uint64_t*);
extern "C" void TestLongJmp(uint64_t*);
#if defined NN_BUILD_CONFIG_CPU_ARM64
const size_t JmpBufSize = 30;
extern "C" void SetFpuContext(uint64_t*, uint32_t*, uint32_t*);
extern "C" void GetFpuContext(uint64_t*, uint32_t*, uint32_t*);
#elif defined NN_BUILD_CONFIG_CPU_ARM
const size_t JmpBufSize = 14;
extern "C" void SetFpuContext(uint64_t*, uint32_t*);
extern "C" void GetFpuContext(uint64_t*, uint32_t*);
#else
#error NN_BUILD_CONFIG_CPU_
#endif
extern int32_t g_ProcessIdealCore;

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

const int SleepTime = 1000000;
const int NumTest = 100;
bool g_LoopFlag;

#if defined NN_BUILD_CONFIG_CPU_ARM64
void GetSetFpu()
{
    uint64_t fpr0[64];
    uint64_t fpr1[64];
    uint32_t fpcr0;
    uint32_t fpcr1;
    uint32_t fpsr0;
    uint32_t fpsr1;
    static uint64_t s_Value = 0;
    uint64_t jmpbuf[JmpBufSize];

    ::std::memset(fpr1, 0, sizeof(fpr1));
    fpcr1 = 0;
    fpsr1 = 0;
    for (int j = 0; j < static_cast<int>(sizeof(fpr0) / sizeof(*fpr0)); j++)
    {
        fpr0[j] = (s_Value << 56) | (s_Value << 48) | (s_Value << 40) | (s_Value << 32) | (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
        s_Value++;
        s_Value &= 0xFF;
    }
    fpcr0 = (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
    s_Value++;
    s_Value &= 0xFF;
    fpsr0 = (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
    s_Value++;
    s_Value &= 0xFF;
    if (TestSetJmp(jmpbuf) == 0)
    {
        SetFpuContext(fpr0, &fpcr0, &fpsr0);
        GetFpuContext(fpr0, &fpcr0, &fpsr0);
        GetFpuContext(fpr1, &fpcr1, &fpsr1);
        TestLongJmp(jmpbuf);
    }
    ASSERT_TRUE(::std::memcmp(fpr0, fpr1, sizeof(fpr0)) == 0);
    ASSERT_TRUE(fpcr0 == fpcr1);
    ASSERT_TRUE(fpsr0 == fpsr1);
}
#elif defined NN_BUILD_CONFIG_CPU_ARM
void GetSetFpu()
{
    uint64_t fpr0[NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS];
    uint64_t fpr1[NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS];
    uint32_t fpscr0;
    uint32_t fpscr1;
    static uint64_t s_Value = 0;
    uint64_t jmpbuf[JmpBufSize];

    ::std::memset(fpr1, 0, sizeof(fpr1));
    fpscr1 = 0;
    for (int j = 0; j < static_cast<int>(sizeof(fpr0) / sizeof(*fpr0)); j++)
    {
        fpr0[j] = (s_Value << 56) | (s_Value << 48) | (s_Value << 40) | (s_Value << 32) | (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
        s_Value++;
        s_Value &= 0xFF;
    }
    fpscr0 = (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
    s_Value++;
    s_Value &= 0xFF;
    if (TestSetJmp(jmpbuf) == 0)
    {
        SetFpuContext(fpr0, &fpscr0);
        GetFpuContext(fpr0, &fpscr0);
        GetFpuContext(fpr1, &fpscr1);
        TestLongJmp(jmpbuf);
    }
    ASSERT_TRUE(::std::memcmp(fpr0, fpr1, sizeof(fpr0)) == 0);
    ASSERT_TRUE(fpscr0 == fpscr1);
}
#else
#error NN_BUILD_CONFIG_CPU_
#endif

void TestFunction()
{
    AutoThreadExit autoExit;

    while (g_LoopFlag)
    {
        GetSetFpu();
    }
}
}


TEST(SetThreadCoreMask, SetNumCoreTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    g_LoopFlag = true;
    result = nn::svc::CreateThread(
            &handle,
            reinterpret_cast<uintptr_t>(TestFunction),
            0,
            reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer)),
            TestLowestThreadPriority,
            0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-3
    // 実行前のスレッドに対して、idealCore とaffinityMask を変更することが出来る
    for (int n = 0; n < NumTest; n++)
    {
        for (int i = 0; i < NumCore; i++)
        {
            {
                nn::Bit64 mask = (1 << i);
                nn::Result result = nn::svc::SetThreadCoreMask(handle, i, mask);
                ASSERT_RESULT_SUCCESS(result);
            }
            {
                nn::Bit64 mask = 0;
                int32_t tmp;
                nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &mask, handle);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(i == tmp);
            }
        }
    }

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    g_LoopFlag = false;

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-4
    // 実行後のスレッドに対して、idealCore とaffinityMask を変更することが出来る
    for (int n = 0; n < NumTest; n++)
    {
        for (int i = 0; i < NumCore; i++)
        {
            {
                nn::Bit64 mask = (1 << i);
                nn::Result result = nn::svc::SetThreadCoreMask(handle, i, mask);
                ASSERT_RESULT_SUCCESS(result);
            }
            {
                nn::Bit64 mask = 0;
                int32_t tmp;
                nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &mask, handle);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(i == tmp);
            }
        }
    }

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    {
        nn::Bit64 mask = 1 << g_ProcessIdealCore;
        nn::Result result = nn::svc::SetThreadCoreMask(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
                g_ProcessIdealCore, mask);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(SetThreadCoreMask, SetIdealCoreUseProcessValueTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::Bit64 mask = (1 << g_ProcessIdealCore);

    g_LoopFlag = true;
    result = nn::svc::CreateThread(
            &handle,
            reinterpret_cast<uintptr_t>(TestFunction),
            0,
            reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer)),
            TestLowestThreadPriority,
            (g_ProcessIdealCore + 1) % NumCore);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-5
    // 実行前のスレッドに対して、IdealCoreUseProcessValue を指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreUseProcessValue, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            nn::Bit64 tmpMask = 0;
            int32_t tmp;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_ProcessIdealCore == tmp);
            ASSERT_TRUE(tmpMask == mask);
        }
    }

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-6
    // 実行中のスレッドに対して、IdealCoreUseProcessValue を指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreUseProcessValue, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        nn::svc::SleepThread(SleepTime);
        {
            nn::Bit64 tmpMask = 0;
            int32_t tmp;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_ProcessIdealCore == tmp);
            ASSERT_TRUE(tmpMask == mask);
        }
    }


    g_LoopFlag = false;

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-7
    // 実行後のスレッドに対して、IdealCoreUseProcessValue を指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreUseProcessValue, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            nn::Bit64 tmpMask = 0;
            int32_t tmp;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_ProcessIdealCore == tmp);
            ASSERT_TRUE(tmpMask == mask);
        }
    }

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    {
        nn::Bit64 mask = 1 << g_ProcessIdealCore;
        nn::Result result = nn::svc::SetThreadCoreMask(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
                g_ProcessIdealCore, mask);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(SetThreadCoreMask, SetIdealCoreNotUpdateTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int coreNo = 1;
    nn::Bit64 mask = 1 << coreNo;

    g_LoopFlag = true;
    result = nn::svc::CreateThread(
            &handle,
            reinterpret_cast<uintptr_t>(TestFunction),
            0,
            reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer)),
            TestLowestThreadPriority,
            coreNo);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-8
    // 実行前のスレッドに対して、IdealCoreNoUpdate が指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreNoUpdate, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            int32_t tmp;
            nn::Bit64 tmpMask = 0;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(coreNo == tmp);
        }
    }

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-9
    // 実行中のスレッドに対して、IdealCoreNoUpdate が指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreNoUpdate, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        nn::svc::SleepThread(SleepTime);
        {
            int32_t tmp;
            nn::Bit64 tmpMask = 0;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(coreNo == tmp);
        }
    }


    g_LoopFlag = false;

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-10
    // 実行後のスレッドに対して、IdealCoreNoUpdate が指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreNoUpdate, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            int32_t tmp;
            nn::Bit64 tmpMask = 0;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(coreNo == tmp);
        }
    }

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    {
        nn::Bit64 mask = 1 << g_ProcessIdealCore;
        nn::Result result = nn::svc::SetThreadCoreMask(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
                g_ProcessIdealCore, mask);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(SetThreadCoreMask, SetIdealCoreDontCareTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int coreNo = 1;
    nn::Bit64 mask = 1 << coreNo;

    g_LoopFlag = true;
    result = nn::svc::CreateThread(
            &handle,
            reinterpret_cast<uintptr_t>(TestFunction),
            0,
            reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer)),
            TestLowestThreadPriority,
            coreNo);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-11
    // 実行前のスレッドに対して、IdealCoreDontCare が指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreDontCare, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            int32_t tmp;
            nn::Bit64 tmpMask = 0;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(nn::svc::IdealCoreDontCare == tmp);
        }
    }

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-6
    // 実行中のスレッドに対して、IdealCoreDontCare が指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreDontCare, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        nn::svc::SleepThread(SleepTime);
        {
            int32_t tmp;
            nn::Bit64 tmpMask = 0;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(nn::svc::IdealCoreDontCare == tmp);
        }
    }


    g_LoopFlag = false;

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-7
    // 実行後のスレッドに対して、IdealCoreDontCare が指定できる
    for (int n = 0; n < NumTest; n++)
    {
        {
            nn::Result result = nn::svc::SetThreadCoreMask(
                    handle, nn::svc::IdealCoreDontCare, mask);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            int32_t tmp;
            nn::Bit64 tmpMask = 0;
            nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(nn::svc::IdealCoreDontCare == tmp);
        }
    }

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    {
        nn::Bit64 mask = 1 << g_ProcessIdealCore;
        nn::Result result = nn::svc::SetThreadCoreMask(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
                g_ProcessIdealCore, mask);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(SetThreadCoreMask, InvalidCombinationTest)
{
    TestThreadLeak leakTest;
    // TEST 13-14
    // IdealCore 番目のビットがaffinityMask に立っていないと失敗する
    for (int idealCore = 0; idealCore < NumCore; idealCore++)
    {
        nn::Result result = nn::svc::SetThreadCoreMask(
                nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
                idealCore,
                0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
}

TEST(SetThreadCoreMask, InvalidHandleTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int coreNo = 0;
    nn::Bit64 mask = 1 << coreNo;

    // TEST 13-15
    result = nn::svc::SetThreadCoreMask(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            coreNo,
            mask);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 13-16
    result = nn::svc::SetThreadCoreMask(
            nn::svc::INVALID_HANDLE_VALUE,
            coreNo,
            mask);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 13-17
    // Close したスレッドハンドルを渡すと失敗する
    result = nn::svc::CreateThread(
            &handle,
            reinterpret_cast<uintptr_t>(TestFunction),
            0,
            reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer)),
            TestLowestThreadPriority,
            coreNo);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetThreadCoreMask(
            handle,
            coreNo,
            mask);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SetThreadCoreMask, InvalidCoreNumberTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::Bit64 affinityMask = 0;

    for (int i = NumCore; i < 64; i++)
    {
        affinityMask = 1ull << i;
        result = nn::svc::SetThreadCoreMask(
                nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, i, affinityMask);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());
    }
}

