﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

namespace {

const int StackSize = 0x2000;
char g_Buffer[StackSize] __attribute__((aligned(0x1000)));
nn::Bit32 g_UserBuffer0[0x2000 / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
nn::Bit32 g_UserBuffer1[0x2000 / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));

void ReceiveAndCloseSession(nn::svc::Handle session)
{
    nn::Result result;
    int32_t index;
    nn::svc::Handle handles[1] = { session };
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer1);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer1);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
        &index, pMessage, size,
        handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    result = nn::svc::CloseHandle(session);
    ASSERT_RESULT_SUCCESS(result);
}

void ReceiveAndCloseSessionThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::svc::Handle* session = reinterpret_cast<nn::svc::Handle*>(arg);
    ReceiveAndCloseSession(*session);
}

void CheckMemoryStateThread(nn::svc::Handle* session)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    int32_t index;
    nn::svc::Handle handles[1] = { *session };
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer1);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer1);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
        &index, pMessage, size,
        handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    // クライアントのユーザーバッファはアクセスできないようになっている
    result = nn::svc::QueryMemory
        (&blockInfo, &pageInfo, reinterpret_cast<uintptr_t>(g_UserBuffer0));
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
        &index, pMessage, size,
        handles, 0, *session, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::CloseHandle(*session);
    ASSERT_RESULT_SUCCESS(result);
}


void SendIpcThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);
    ASSERT_TRUE(clientSession != NULL);

    // リプライが送信される前にセッションが閉じられると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, *clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

} //namespace

TEST(SendSyncRequestWithUserBuffer, UserBufferStateTestWithNamedPort)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle threadHandle;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    const char* PortName = "Test29149";
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(CheckMemoryStateThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + sizeof(g_Buffer);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t index;

    // 前準備
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(&threadHandle, pc, param, sp,
            TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-149
    // 結果が返ってくるまでユーザーバッファにアクセスできない
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    // TEST 29-150
    // 結果が返ってきたら、ユーザーバッファにアクセスできる
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // 終了処理
    result = nn::svc::WaitSynchronization(&index, &threadHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(threadHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, UserBufferStateTestWithPort)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle threadHandle;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(CheckMemoryStateThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + sizeof(g_Buffer);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t index;

    // 前準備
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(&threadHandle, pc, param, sp,
        TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-151
    // 結果が返ってくるまでユーザーバッファにアクセスできない
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    // TEST 29-152
    // 結果が返ってきたら、ユーザーバッファにアクセスできる
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // 終了処理
    result = nn::svc::WaitSynchronization(&index, &threadHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(threadHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, UserBufferStateTestWithSession)
{
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle threadHandle;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(CheckMemoryStateThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + sizeof(g_Buffer);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);

    // 前準備
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(&threadHandle, pc, param, sp,
        TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-153
    // 結果が返ってくるまでユーザーバッファにアクセスできない
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    // TEST 29-154
    // 結果が返ってきたら、ユーザーバッファにアクセスできる
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // 終了処理
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &threadHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(threadHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, NamedPortSessionCloseTest)
{
    TestSessionLeak leakTest;
    const char* PortName = "Test2927";
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    int32_t index;

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-27
    // リプライが送信される前にセッションが閉じられると失敗する
    result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
            sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    ReceiveAndCloseSession(serverSession);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, PortSessionCloseTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    int32_t index;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-28
    // リプライが送信される前にセッションが閉じられると失敗する
    result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
            sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    ReceiveAndCloseSession(serverSession);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, SessionCloseTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-29
    // リプライが送信される前にセッションが閉じられると失敗する
    result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
            sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    ReceiveAndCloseSession(serverSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, UserBufferStateTestWithNamedPortOnError)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle threadHandle;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    const char* PortName = "Test29155";
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReceiveAndCloseSessionThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + sizeof(g_Buffer);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t index;

    // 前準備
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(
            &threadHandle, pc, param, sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-155
    // 呼び出しに失敗したときに、ユーザーバッファの状態が元に戻っている
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // 終了処理
    result = nn::svc::CloseHandle(threadHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, UserBufferStateTestWithPortOnError)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle threadHandle;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReceiveAndCloseSessionThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + sizeof(g_Buffer);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t index;

    // 前準備
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(
            &threadHandle, pc, param, sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-156
    // 呼び出しに失敗したときに、ユーザーバッファの状態が元に戻っている
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // 終了処理
    result = nn::svc::CloseHandle(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
}


TEST(SendSyncRequestWithUserBuffer, UserBufferStateTestWithSessionOnError)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle threadHandle;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReceiveAndCloseSessionThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + sizeof(g_Buffer);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);

    // 前準備
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(
            &threadHandle, pc, param, sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-157
    // 呼び出しに失敗したときに、ユーザーバッファの状態が元に戻っている
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, pMessage);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // 終了処理
    result = nn::svc::CloseHandle(threadHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, SessionCloseTestOnNamedPort)
{
    TestSessionLeak leakTest;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    nn::Result result;
    TestHeap heap(HeapAlign);
    const char* name = "AsyncReq";

    nn::svc::Handle namedPort;
    result = nn::svc::ManageNamedPort(&namedPort, name, 1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToNamedPort(&clientSession, name);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, namedPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-156
    // サーバーセッションが先に閉じられると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    result = nn::svc::CloseHandle(namedPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&namedPort, name, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(namedPort == nn::svc::INVALID_HANDLE_VALUE);
}

TEST(SendSyncRequestWithUserBuffer, SessionCloseTestOnPort)
{
    TestSessionLeak leakTest;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    nn::Result result;
    TestHeap heap(HeapAlign);

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-157
    // サーバーセッションが先に閉じられると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

TEST(SendSyncRequestWithUserBuffer, SessionCloseTestOnSession)
{
    TestSessionLeak leakTest;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    nn::Result result;
    TestHeap heap(HeapAlign);

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-158
    // サーバーセッションが先に閉じられると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

