﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReceiveLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReplyLightRequest(nn::svc::Handle handle);

namespace {

#ifdef INVALID_POINTER_TEST
const int tmpVar = 0;
#endif
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
int g_Sequence;
const int64_t SleepTime = 100 * 1000 * 1000;
const int BeginThread = 0;
const int RunningThread = 1;
const int EndThread = 2;

void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

void SendThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(arg);

    while(g_Sequence != EndThread)
    {
        if (g_Sequence == BeginThread)
        {
            g_Sequence = RunningThread;
        }

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
        // IsFailure になることもあるので、result のチェックはしない
        result = nn::svc::SendSyncRequest(*handle);
    }
}

void SendLightRequestThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    result = ClientSendLightRequest(*handle);
}

void CallReplyAndReceiveThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t index;


    // セッションの設定
    nn::svc::Handle* session = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::svc::Handle handles[1] = { *session };

    ASSERT_TRUE(g_Sequence == BeginThread);

    // TEST 67-139
    // 指定した時間以上待った上で、タイムアウトする
    {
        for (int64_t timeoutValue = 0; timeoutValue <= 0x100000000;)
        {
            SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
            volatile int64_t beginTick = nn::svc::GetSystemTick();
            result = nn::svc::ReplyAndReceive(&index, handles, 1,
                                              nn::svc::INVALID_HANDLE_VALUE, timeoutValue);
            volatile int64_t endTick = nn::svc::GetSystemTick();
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
            ASSERT_TRUE(((endTick - beginTick) / (NN_HW_TICKS_PER_SECOND / (1000 * 1000)) * 1000)
                        >= timeoutValue);
            timeoutValue = (timeoutValue == 0) ? 1 : timeoutValue << 1;
        }
    }

    g_Sequence = RunningThread;

    // TEST 67-140
    // 負の値を指定した場合、タイムアウトしない
    {
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(&index, handles, 1,
                                          nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(&index, handles, 1,
                                          *session, 0x8000000040000000);
        ASSERT_RESULT_SUCCESS(result);
    }

    // 終了処理
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(&index, handles, 0, *session, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

} // namespace

extern "C" void nnMain();

#ifdef INVALID_POINTER_TEST
TEST(ReplyAndReceive, InvalidPointerForpIndex)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 67-103
    // NULL ポインタは受け付けない
#ifdef INVALID_POINTER_TEST
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            NULL, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 67-104
    // MemoryPermission_None は受け付けない
#ifdef INVALID_POINTER_TEST
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            reinterpret_cast<int32_t*>(g_FreeAreaBegin), handles, 1,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 67-105
    // MemoryPermission_Read は受け付けない
#ifdef INVALID_POINTER_TEST
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    uintptr_t addr = reinterpret_cast<uintptr_t>(&tmpVar);
    result = nn::svc::ReplyAndReceive(
            reinterpret_cast<int32_t*>(addr), handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 67-106
    // MemoryPermission_ReadExecute は受け付けない
#ifdef INVALID_POINTER_TEST
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            reinterpret_cast<int32_t*>(nnMain), handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}
#endif // INVALID_POINTER_TEST

TEST(ReplyAndReceive, AcceptHandleForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t index;

    // TEST 67-131
    // Event を受け付ける
    {
        nn::svc::Handle writableEvent;
        nn::svc::Handle readableEvent;

        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser(writableEvent);
        AutoHandleClose rEventCloser(readableEvent);

        result = nn::svc::SignalEvent(writableEvent);

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        handles[0] = readableEvent;
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 67-132
    // Thread を受け付ける
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        handles[0] = thread.GetHandle();
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(ReplyAndReceive, InvalidPointerForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 67-107
    // NULL ポインタは受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, NULL, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // TEST 67-108
    // MemoryPermission_None は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin),
            1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceive, InvalidHandleForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[3];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    // TEST 67-109
    // INVALID_HANDLE_VALUE は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 67-110
    // T_CurrentThread は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    handles[0] = nn::svc::PSEUDO_HANDLE_CURRENT_THREAD;
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 67-111
    // T_CurrentProcess は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    handles[0] = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 67-130
    // Light Session を受け付けない
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        nn::svc::Handle lightClientSession;
        nn::svc::Handle lightServerSession;

        result = nn::svc::ConnectToPort(&lightClientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(lightClientSession);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&lightServerSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(lightServerSession);

        handles[0] = lightServerSession;
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        handles[0] = lightClientSession;
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // 一応確認
    handles[0] = serverSession;
    handles[1] = thread.GetHandle();
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 2, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, handles[0], 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 67-145
    // 不正なハンドルが紛れていると失敗する
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    handles[1] = serverSession;
    handles[2] = thread.GetHandle();
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    handles[0] = serverSession;
    handles[1] = nn::svc::INVALID_HANDLE_VALUE;
    handles[2] = thread.GetHandle();
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    handles[0] = serverSession;
    handles[1] = thread.GetHandle();
    handles[2] = nn::svc::INVALID_HANDLE_VALUE;
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 67-112
    // Close したセッションは受け付けない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    handles[0] = serverSession;
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // 終了処理
    g_Sequence = EndThread;

    thread.Wait();
} // NOLINT(impl/function_size)

TEST(ReplyAndReceive, NumHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 67-113
    // 負の値は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, -1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    result = nn::svc::ReplyAndReceive(&index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    g_Sequence = EndThread;

    // TEST 67-138
    // 0 を受け付ける
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(&index, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 67-144
    // ArgumentHandleCountMax 以上の値を指定することが出来ない
    TestHeap heap(sizeof(nn::svc::Handle) * nn::svc::ArgumentHandleCountMax);
    nn::svc::Handle* pHandles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
    for (int32_t i = 0; i < nn::svc::ArgumentHandleCountMax + 1; i++)
    {
        pHandles[i] = serverSession;
    }

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(&index, pHandles, nn::svc::ArgumentHandleCountMax,
            nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(&index, pHandles, nn::svc::ArgumentHandleCountMax + 1,
            nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    // 念のためint32_t の最大値も調べる
    // ただし、ハンドルをそんなに用意出来ないので、pHandles はそのままにする
    // そのため、ResultInvalidHandle になる可能性がある
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(&index, pHandles, 0x7FFFFFFF,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    thread.Wait();
}

TEST(ReplyAndReceive, OverFlowTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    // TEST 67-114
    // handles と numHandle の組み合わせでオーバーフローが起きる
    uintptr_t addr = static_cast<uintptr_t>(-1) & ~(0xFFF);
    result = nn::svc::ReplyAndReceive(
            &index, reinterpret_cast<nn::svc::Handle*>(addr), 0x1000,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE(result);

    // 終了処理
    handles[0] = serverSession;
    if (g_Sequence == RunningThread)
    {
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        g_Sequence = EndThread;
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceive, InvalidHandleForReplyTarget)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 67-115
    // INVALID_HANDLE_VALUE は受け付ける
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

    // TEST 67-116
    // T_CurrentThread は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 67-117
    // T_CurrentProcess は受け付けない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // 一度結果を返しておく
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 67-118
    // 要求を受け付けていないセッションに対して返信できない
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE(result);

    // 再び要求を受信しておく
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

    // TEST 67-119
    // Closeしたセッションは受け付けない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // 一旦終了処理
    g_Sequence = EndThread;
    result = nn::svc::CancelSynchronization(thread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);

    thread.Wait();
    thread.Close(); // sp を解放するため

    // Event
    {
        nn::svc::Handle writableEvent;
        nn::svc::Handle readableEvent;

        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser(writableEvent);
        AutoHandleClose rEventCloser(readableEvent);

        result = nn::svc::SignalEvent(writableEvent);

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        handles[0] = readableEvent;
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        // TEST 67-133
        // 読み込みイベントを受け付けない
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, readableEvent, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        // TEST 67-134
        // 書き込みイベントを受け付けない
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, writableEvent, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Thread
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        handles[0] = thread.GetHandle();
        result = nn::svc::ReplyAndReceive(
                &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        // TEST 67-135
        // Thread を受け付けない
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, handles[0], 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Light Session
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        nn::svc::Handle lightClientSession;
        nn::svc::Handle lightServerSession;

        result = nn::svc::ConnectToPort(&lightClientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(lightClientSession);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&lightServerSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(lightServerSession);

        uintptr_t pc = reinterpret_cast<uintptr_t>(SendLightRequestThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = reinterpret_cast<uintptr_t>(&lightClientSession);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        result = ServerReceiveLightRequest(lightServerSession);
        ASSERT_RESULT_SUCCESS(result);

        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        // TEST 67-136
        // Light Session を受け付けない
        handles[0] = thread.GetHandle();
        result = nn::svc::ReplyAndReceive(
                &index, handles, 0, lightServerSession, 0);

        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::CloseHandle(lightServerSession);
        ASSERT_RESULT_SUCCESS(result);

        thread.Wait();
    }
} // NOLINT(readability/fn_size)

TEST(ReplyAndReceive, SuccessToReply)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
            &index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    // TEST 67-120
    // handles に登録されていなくても、replyTarget に要求を受信したセッションを指定できる
    g_Sequence = EndThread;
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // 終了処理
    thread.Wait();
}

TEST(ReplyAndReceive, TimeoutTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(CallReplyAndReceiveThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, param, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    // TEST 67-139 を実施している間待つ
    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    // TEST 67-140
    // 負の値を指定した場合、タイムアウトしない
    nn::svc::SleepThread(SleepTime);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Reply);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    // 0x40000000 が約1 秒なので、2秒待ってみる
    nn::svc::SleepThread(2 * 1000 * 1000 * 1000);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Reply);

    // 終了処理
    thread.Wait();
}

