﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReceiveLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReplyLightRequest(nn::svc::Handle handle);

extern "C" void IpcHeaderAsm();

namespace {

#ifdef INVALID_POINTER_TEST
const int tmpVar = 0;
#endif
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
char g_UserBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
int g_Sequence;
const int64_t SleepTime = 100 * 1000 * 1000;
const int BeginThread = 0;
const int RunningThread = 1;
const int EndThread = 2;

void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

void SendThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    while(g_Sequence != EndThread)
    {
        if (g_Sequence == BeginThread)
        {
            g_Sequence = RunningThread;
        }
        // IsFailure になることもあるので、result のチェックはしない
        ipcMsg.SetNull();
        result = nn::svc::SendSyncRequest(*handle);
    }
}

void SendLightRequestThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    result = ClientSendLightRequest(*handle);
}

void CallReplyAndReceiveWithUserBufferThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t index;


    // セッションの設定
    nn::svc::Handle* session = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::svc::Handle handles[1] = { *session };

    ASSERT_TRUE(g_Sequence == BeginThread);

    // バッファの設定
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // TEST 67-139
    // 指定した時間以上待った上で、タイムアウトする
    {
        for (int64_t timeoutValue = 0; timeoutValue <= 0x100000000;)
        {
            ipcMsg.SetNull();
            volatile int64_t beginTick = nn::svc::GetSystemTick();
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, pMsgBuffer, msgSize,
                        handles, 1, nn::svc::INVALID_HANDLE_VALUE, timeoutValue);
            volatile int64_t endTick = nn::svc::GetSystemTick();
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
            ASSERT_TRUE(((endTick - beginTick) / (NN_HW_TICKS_PER_SECOND / (1000 * 1000)) * 1000)
                        >= timeoutValue);
            timeoutValue = (timeoutValue == 0) ? 1 : timeoutValue << 1;
        }
    }

    g_Sequence = RunningThread;

    // TEST 67-140
    // 負の値を指定した場合、タイムアウトしない
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, pMsgBuffer, msgSize,
                    handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, pMsgBuffer, msgSize,
                    handles, 1, *session, 0x8000000040000000);
        ASSERT_RESULT_SUCCESS(result);
    }

    // 終了処理
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 0, *session, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        size_t msgSize = 0x1000;

        ASSERT_TRUE(array[i]->GetSize() >= msgSize && ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize();

        int32_t index;
        uintptr_t pMsgBuffer = array[i]->GetAddress();
        nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, &handle, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, &handle, 0, handle, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;
    nn::Bit32 msgBuffer[2];
    SetOnlyIpcTag(msgBuffer, DefaultIpcTag_Send);

    for (int i = 0; i < numState; i++)
    {
        size_t msgSize = 0x1000;
        if (!(array[i]->GetSize() >= msgSize && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize(&msgBuffer, sizeof(msgBuffer));

        int32_t index;
        uintptr_t pMsgBuffer = array[i]->GetAddress();
        if (array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            pMsgBuffer = reinterpret_cast<uintptr_t>(IpcHeaderAsm);
            nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(pMsgBuffer);
            ASSERT_EQ(ptr[0], 0x00000001);
            ASSERT_EQ(ptr[1], 0x00000000);

            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, pMsgBuffer);
            ASSERT_EQ(info.state, nn::svc::MemoryState_Code);
            ASSERT_EQ(info.permission, nn::svc::MemoryPermission_ReadExecute);
            ASSERT_LE(info.baseAddress, pMsgBuffer);
            ASSERT_GE(info.baseAddress + info.size - 1, pMsgBuffer + msgSize - 1);
        }
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, &handle, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        if (array[i]->CanWriteData() || array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
        else
        {
            ASSERT_RESULT_FAILURE(result);
        }

        array[i]->Close();
    }
}


} // namespace

extern "C" void nnMain();

#ifdef INVALID_POINTER_TEST
TEST(ReplyAndReceiveWithUserBuffer, InvalidPointerForpIndex)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-217
    // NULL ポインタは受け付けない
#ifdef INVALID_POINTER_TEST
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            0, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 68-218
    // MemoryPermission_None は受け付けない
#ifdef INVALID_POINTER_TEST
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            reinterpret_cast<int32_t*>(g_FreeAreaBegin), pMsgBuffer, msgSize, handles, 1,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 68-219
    // MemoryPermission_Read は受け付けない
#ifdef INVALID_POINTER_TEST
    ipcMsg.SetNull();
    uintptr_t addr = reinterpret_cast<uintptr_t>(&tmpVar);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            reinterpret_cast<int32_t*>(addr), pMsgBuffer, msgSize, handles, 1,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // TEST 68-220
    // MemoryPermission_ReadExecute は受け付けない
#ifdef INVALID_POINTER_TEST
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            reinterpret_cast<int32_t*>(nnMain), pMsgBuffer, msgSize, handles, 1,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        int32_t index;
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}
#endif // INVALID_POINTER_TEST

TEST(ReplyAndReceiveWithUserBuffer, InvalidAlignment)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-221
    // pMessage が 4KB にアライメントされていないと失敗する
    for (int i = 1; i < 0x1000; i++)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer + i, msgSize, handles, 1,
                nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // TEST 68-222
    // bufferSize が 4KB にアライメントされていないと失敗する
    for (int i = 1; i < 0x1000; i++)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, i, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, InvalidMemoryAccessForMessage)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-223
    // NULL ポインタは受け付けない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, 0, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 68-224
    // MemoryPermission_None は受け付けない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, g_FreeAreaBegin, msgSize, handles,
            1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 68-225
    // MemoryPermission_Read は受け付けない
    {
        TestHeap heap(HeapAlign);
        uintptr_t heapPtr = heap.GetAddress();
        {
            TestMemoryPermission perm(heapPtr, msgSize, nn::svc::MemoryPermission_Read);

            ipcMsg.SetNull();
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, heapPtr, msgSize, handles,
                    1, nn::svc::INVALID_HANDLE_VALUE, -1);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }

    // TEST 68-226
    // MemoryPermission_ReadExecute は受け付けない
    uintptr_t addr = reinterpret_cast<uintptr_t>(nnMain);
    addr = (addr + 0xFFF) & ~0xFFF;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, addr, msgSize, handles,
            1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, MemoryStateTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif // SUPPORT_ALIAS_CODE_DATA
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);


    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, serverSession);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, serverSession);

    uint32_t successAttribute[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr, serverSession);

    const int NumInvalidAttrs = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrs[NumInvalidAttrs];
    header->GetTestListExceptAttributes(invalidAttrs, NumInvalidAttrs, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttrs, NumInvalidAttrs, serverSession);

    {
        TestCodeDataMemoryState codeData;
        nn::svc::MemoryInfo info;
        GetMemoryInfo(&info, codeData.GetAddress());
        uintptr_t addr = info.baseAddress;

        // メモリ状態が一様でないと失敗する
        // 前方
        ipcMsg.SetNull();
        pMsgBuffer = addr - msgSize;
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize * 2, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        // 後方
        ipcMsg.SetNull();
        pMsgBuffer = addr + info.size - msgSize;
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize * 2, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        // オーバーフローする組み合わせを指定できない
        ipcMsg.SetNull();
        pMsgBuffer = addr + msgSize;
        size_t overSize = static_cast<size_t>(-msgSize);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, overSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        codeData.Close();
    }

    // 終了処理
    pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, OverFlowTestForMessage)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-228
    // pMessage と bufferSize の組み合わせでオーバーフローが起こるものは指定できない
    ipcMsg.SetNull();
    uintptr_t addr = static_cast<uintptr_t>(-1) & ~(0xFFF);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, addr, 0x2000, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // 終了処理
    handles[0] = serverSession;
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, InvalidPointerForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-229
    // NULL ポインタは受け付けない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, NULL, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // TEST 68-230
    // MemoryPermission_None は受け付けない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin),
            1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, AcceptHandleForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t index;
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // TEST 68-286
    // Event を受け付ける
    {
        nn::svc::Handle writableEvent;
        nn::svc::Handle readableEvent;

        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser(writableEvent);
        AutoHandleClose rEventCloser(readableEvent);

        result = nn::svc::SignalEvent(writableEvent);

        handles[0] = readableEvent;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 68-287
    // Thread を受け付ける
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        handles[0] = thread.GetHandle();
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(ReplyAndReceiveWithUserBuffer, InvalidHandleForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[3];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-231
    // INVALID_HANDLE_VALUE は受け付けない
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 68-232
    // T_CurrentThread は受け付けない
    handles[0] = nn::svc::PSEUDO_HANDLE_CURRENT_THREAD;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 68-233
    // T_CurrentProcess は受け付けない
    handles[0] = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // 一応確認
    handles[0] = serverSession;
    handles[1] = thread.GetHandle();
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, handles, 2, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, handles, 0, handles[0], 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 68-300
    // 不正なハンドルが紛れていると失敗する
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    handles[1] = serverSession;
    handles[2] = thread.GetHandle();
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    handles[0] = serverSession;
    handles[1] = nn::svc::INVALID_HANDLE_VALUE;
    handles[2] = thread.GetHandle();
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    handles[0] = serverSession;
    handles[1] = thread.GetHandle();
    handles[2] = nn::svc::INVALID_HANDLE_VALUE;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, handles, 3, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());


    // TEST 68-234
    // Close したセッションは受け付けない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    handles[0] = serverSession;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 67-130
    // Light Session を受け付けない
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        nn::svc::Handle lightClientSession;
        nn::svc::Handle lightServerSession;

        result = nn::svc::ConnectToPort(&lightClientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(lightClientSession);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&lightServerSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(lightServerSession);

        handles[0] = lightServerSession;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        handles[0] = lightClientSession;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::CloseHandle(lightServerSession);
        ASSERT_RESULT_SUCCESS(result);
    }


    // 終了処理
    g_Sequence = EndThread;

    thread.Wait();
} // NOLINT(readability/fn_size)

TEST(ReplyAndReceiveWithUserBuffer, InvalidNumHandle)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-235
    // 負の値は受け付けない
    handles[0] = serverSession;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, -1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    g_Sequence = EndThread;

    // TEST 68-293
    // 0 を受け付ける
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 68-299
    // ArgumentHandleCountMax より大きな値を指定することが出来ない
    TestHeap heap(sizeof(nn::svc::Handle) * nn::svc::ArgumentHandleCountMax + 1);
    nn::svc::Handle* pHandles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
    for (int32_t i = 0; i < nn::svc::ArgumentHandleCountMax + 1; i++)
    {
        pHandles[i] = serverSession;
    }
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize,
            pHandles, nn::svc::ArgumentHandleCountMax, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize,
            pHandles, nn::svc::ArgumentHandleCountMax + 1, nn::svc::INVALID_HANDLE_VALUE, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    // 念のためint32_t の最大値も調べる
    // ただし、ハンドルをそんなに用意出来ないので、pHandles はそのままにする
    // そのため、ResultInvalidHandle になる可能性がある
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize,
            pHandles, 0x7FFFFFFF, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, OverFlowTestForHandles)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-236
    // handles と numHandle の組み合わせでオーバーフローが起きる
    uintptr_t addr = static_cast<uintptr_t>(-1) & ~(0xFFF);
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize,
            reinterpret_cast<nn::svc::Handle*>(addr), 0x1000,
            nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE(result);

    // 終了処理
    handles[0] = serverSession;
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, InvalidHandleForReplyTarget)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();
    ASSERT_RESULT_SUCCESS(result);

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-237
    // INVALID_HANDLE_VALUE は受け付ける
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);

    // TEST 68-238
    // T_CurrentThread は受け付けない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 68-239
    // T_CurrentProcess は受け付けない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // 一度結果を返しておく
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 68-240
    // 要求を受け付けていないセッションに対して返信できない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE(result);

    // 再び要求を受信しておく
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 68-241
    // Closeしたセッションは受け付けない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // 一旦終了処理
    g_Sequence = EndThread;

    thread.Wait();
    thread.Close(); // sp を解放する必要があるため

    // Event
    {
        nn::svc::Handle writableEvent;
        nn::svc::Handle readableEvent;

        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser(writableEvent);
        AutoHandleClose rEventCloser(readableEvent);

        result = nn::svc::SignalEvent(writableEvent);

        handles[0] = readableEvent;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 68-288
        // 読み込みイベントを受け付けない
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 0, readableEvent, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        // TEST 68-289
        // 書き込みイベントを受け付けない
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 0, writableEvent, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // TEST 67-290
    // Thread を受け付けない
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        handles[0] = thread.GetHandle();
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                handles, 0, thread.GetHandle(), 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    }

    // Light Session
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        nn::svc::Handle lightClientSession;
        nn::svc::Handle lightServerSession;

        result = nn::svc::ConnectToPort(&lightClientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(lightClientSession);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&lightServerSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(lightServerSession);

        uintptr_t pc = reinterpret_cast<uintptr_t>(SendLightRequestThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = reinterpret_cast<uintptr_t>(&lightClientSession);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        result = ServerReceiveLightRequest(lightServerSession);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 67-136
        // Light Session を受け付けない
        handles[0] = thread.GetHandle();
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, lightServerSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }
} // NOLINT(readability/fn_size)

TEST(ReplyAndReceiveWithUserBuffer, SuccessToReply)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 68-242
    // handles に登録されていなくても、replyTarget に要求を受信したセッションを指定できる
    g_Sequence = EndThread;
    handles[0] = nn::svc::INVALID_HANDLE_VALUE;
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, InvalidSizeTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_UserBuffer);
    size_t msgSize = sizeof(g_UserBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));

    // セッションの作成
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t paramAddr = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, paramAddr, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // TEST 68-252
    // size に 0 を指定することが出来ない
    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, 0,
            handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // 終了処理
    if (g_Sequence == RunningThread)
    {
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence = EndThread;
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    g_Sequence = EndThread;

    thread.Wait();
}

TEST(ReplyAndReceiveWithUserBuffer, TimeoutTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle handles[1];
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);
    handles[0] = serverSession;

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(CallReplyAndReceiveWithUserBufferThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestThread thread(pc, param, sp, priority, idealCore);

    g_Sequence = BeginThread;
    thread.Start();

    // TEST 68-294 を実施している間待つ
    while(g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // TEST 67-140
    // 負の値を指定した場合、タイムアウトしない
    nn::svc::SleepThread(SleepTime);
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // 0x40000000 が約1 秒なので、2秒待ってみる
    nn::svc::SleepThread(2 * 1000 * 1000 * 1000);
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    thread.Wait();
}

