﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_Server.h>

extern "C" void nnMain();

namespace {

const int ConstVar = 0;
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    nn::svc::PhysicalMemoryInfo info;

    for (int i = 0; i < numState; i++)
    {
        array[i]->Initialize();
#ifndef ENABLE_CREATE_PROCESS_STATIC_MAP
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }
#endif

        uintptr_t addr = array[i]->GetAddress();

        // array[i] に入っている Code 領域は実行可能
        if (array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            addr = reinterpret_cast<uintptr_t>(&ConstVar);
        }

        result = nn::svc::QueryPhysicalAddress(&info, addr);
        if(result.IsFailure())
        {
            NN_LOG("state=%d, perm=%d, attr=%d\n", array[i]->GetState(), array[i]->GetPermission(), array[i]->GetAttribute());
        }
        ASSERT_RESULT_SUCCESS(result);

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    nn::svc::PhysicalMemoryInfo info;

    for (int i = 0; i < numState; i++)
    {
        array[i]->Initialize();
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        uintptr_t addr = array[i]->GetAddress();
        result = nn::svc::QueryPhysicalAddress(&info, addr);
        if(result.IsSuccess())
        {
            NN_LOG("state=%d, perm=%d, attr=%d\n", array[i]->GetState(), array[i]->GetPermission(), array[i]->GetAttribute());
        }
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->Close();
    }
}

void CheckMemoryState(uintptr_t addr, nn::svc::MemoryState state)
{
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    nn::Result result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(blockInfo.state == state);
}

void CheckMemoryPermission(uintptr_t addr, nn::svc::MemoryPermission permission)
{
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    nn::Result result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(blockInfo.permission == permission);
}
} // namespace

#ifdef INVALID_POINTER_TEST
TEST(QueryPhysicalAddress, InvalidPointerTest)
{
    nn::Result result;
    nn::svc::PhysicalMemoryInfo* blockInfo;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&ConstVar);

#ifdef INVALID_POINTER_TEST
    // TEST 82-1
    // NULL アドレスは受け付けない
    result = nn::svc::QueryPhysicalAddress(NULL, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif

#ifdef INVALID_POINTER_TEST
    // TEST 82-2
    // MemoryPermission_None の領域を渡すと失敗する
    blockInfo = reinterpret_cast<nn::svc::PhysicalMemoryInfo*>(g_FreeAreaBegin);
    result = nn::svc::QueryPhysicalAddress(blockInfo, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif

#ifdef INVALID_POINTER_TEST
    // TEST 82-3
    // MemoryPermission_Read の領域を渡すと失敗する
    {
        TestHeap heap(HeapAlign);
        {
            TestMemoryPermission perm(heap.GetAddress(), 0x1000, nn::svc::MemoryPermission_Read);
            blockInfo = reinterpret_cast<nn::svc::PhysicalMemoryInfo*>(heap.GetAddress());
            result = nn::svc::QueryPhysicalAddress(blockInfo, addr);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
        }
    }
#endif

#ifdef INVALID_POINTER_TEST
    // TEST 82-4
    // MemoryPermission_ReadExecute の領域を渡すと失敗する
    blockInfo = reinterpret_cast<nn::svc::PhysicalMemoryInfo*>(nnMain);
    result = nn::svc::QueryPhysicalAddress(blockInfo, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif
}
#endif // INVALID_POINTER_TEST

TEST(QueryPhysicalAddress, MemoryStateTest)
{
    nn::Result result;
    nn::svc::PhysicalMemoryInfo blockInfo;
    uintptr_t addr;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Static,
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif // SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_Ipc,
        nn::svc::MemoryState_NonSecureIpc,
        nn::svc::MemoryState_NonDeviceIpc,
        nn::svc::MemoryState_Stack,
        nn::svc::MemoryState_Transfered,
        nn::svc::MemoryState_SharedTransfered,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);
    CheckInvalidMemoryArea(invalidArea, NumInvalid);

    // 属性を持っていても成功する
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    TestMemoryInfo* successAttrs[NumTestMemoryAttribute];
    header->GetTestListWithAttributes(
            successAttrs, AllMemoryAttribute, NumTestMemoryAttribute);
    CheckSuccessMemoryArea(successAttrs, NumTestMemoryAttribute);

    TestHeap heap(HeapAlign);

    // TEST 82-22
    // 実行可能なパーミッションを持っていると失敗する
    addr = reinterpret_cast<uintptr_t>(nnMain);
    CheckMemoryState(addr, nn::svc::MemoryState_Code);
    CheckMemoryPermission(addr, nn::svc::MemoryPermission_ReadExecute);
    result = nn::svc::QueryPhysicalAddress(&blockInfo, addr);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 82-23
    // 読み込みまたは書き込みのパーミッションを持っていないと失敗する
    {
        addr = heap.GetAddress();
        size_t size = 0x1000;

        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);
        CheckMemoryState(addr, nn::svc::MemoryState_Normal);
        CheckMemoryPermission(addr, nn::svc::MemoryPermission_None);

        result = nn::svc::QueryPhysicalAddress(&blockInfo, addr);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
} // NOLINT (readability/fn_size)

// TEST 82-24
// 同じブロック内の仮想アドレスを渡すと、同じブロックの情報が得られる
TEST(QueryPhysicalAddress, BlockTest)
{
    nn::Result result;
    nn::svc::PhysicalMemoryInfo blockInfo;

    TestHeap heap(HeapAlign);
    uintptr_t addr = heap.GetAddress();
    CheckMemoryState(addr, nn::svc::MemoryState_Normal);
    CheckMemoryPermission(addr, nn::svc::MemoryPermission_ReadWrite);

    // 始端
    result = nn::svc::QueryPhysicalAddress(&blockInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.virtualAddress == addr);
    TEST_RW(addr);
    size_t size = blockInfo.size;

    nn::svc::PhysicalMemoryInfo tmpBlockInfo;

    // 中間
    addr = heap.GetAddress() + size / 2;
    result = nn::svc::QueryPhysicalAddress(&tmpBlockInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.physicalAddress == tmpBlockInfo.physicalAddress);
    ASSERT_TRUE(blockInfo.virtualAddress == tmpBlockInfo.virtualAddress);
    ASSERT_TRUE(blockInfo.size == tmpBlockInfo.size);
    TEST_RW(addr);

    // 終端
    addr = heap.GetAddress() + size - 1;
    result = nn::svc::QueryPhysicalAddress(&tmpBlockInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.physicalAddress == tmpBlockInfo.physicalAddress);
    ASSERT_TRUE(blockInfo.virtualAddress == tmpBlockInfo.virtualAddress);
    ASSERT_TRUE(blockInfo.size == tmpBlockInfo.size);
    TEST_RW(addr);
}

TEST(QueryPhysicalAddress, MemoryAreaTest)
{
    nn::Result result;
    nn::svc::PhysicalMemoryInfo blockInfo;

    TestHeap heap(HeapAlign);
    uintptr_t addr = heap.GetAddress();
    size_t size = 0x1000;

    // TEST 82-25
    // addr で指し示す領域の一様なアクセス権を持つ範囲のブロックの情報を取得する
    CheckMemoryState(addr, nn::svc::MemoryState_Normal);
    CheckMemoryPermission(addr, nn::svc::MemoryPermission_ReadWrite);
    {
        TestMemoryPermission permission(addr, size, nn::svc::MemoryPermission_Read);

        result = nn::svc::QueryPhysicalAddress(&blockInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.virtualAddress == addr);
    }
}

