﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {
const nn::svc::MemoryPermission ValidPerm[] = {
    nn::svc::MemoryPermission_ReadWrite,
    nn::svc::MemoryPermission_Read,
    nn::svc::MemoryPermission_None,
};

char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
char g_Transfer[0x1000] __attribute__((aligned(0x1000)));
char g_SharedTransfer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    nn::svc::Handle transHandle;
    nn::svc::Handle sharedHandle;
    nn::svc::MemoryPermission transPerm = nn::svc::MemoryPermission_None;
    nn::svc::MemoryPermission sharedPerm = nn::svc::MemoryPermission_ReadWrite;

    uintptr_t transAddr = reinterpret_cast<uintptr_t>(g_Transfer);
    uintptr_t sharedAddr = reinterpret_cast<uintptr_t>(g_SharedTransfer);
    size_t size = 0x1000;

    CheckMemory(
            transAddr,
            nn::svc::MemoryState_CodeData,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    CheckMemory(
            sharedAddr,
            nn::svc::MemoryState_CodeData,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    result = nn::svc::CreateTransferMemory(&transHandle, transAddr, size, transPerm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose transCloser(transHandle);

    result = nn::svc::CreateTransferMemory(&sharedHandle, sharedAddr, size, sharedPerm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(sharedHandle);

    CheckMemory(
            transAddr,
            nn::svc::MemoryState_CodeData,
            transPerm,
            nn::svc::MemoryAttribute_Locked);

    CheckMemory(
            sharedAddr,
            nn::svc::MemoryState_CodeData,
            sharedPerm,
            nn::svc::MemoryAttribute_Locked);

    for (int i = 0; i < numState; i++)
    {
        SCOPED_TRACE("Success Memory Area");
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::MapTransferMemory(transHandle, addr, size, transPerm);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                nn::svc::MemoryState_Transfered,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        result = nn::svc::UnmapTransferMemory(transHandle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        result = nn::svc::MapTransferMemory(sharedHandle, addr, size, sharedPerm);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                nn::svc::MemoryState_SharedTransfered,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        result = nn::svc::UnmapTransferMemory(sharedHandle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    nn::svc::Handle transHandle;
    nn::svc::Handle sharedHandle;
    nn::svc::MemoryPermission transPerm = nn::svc::MemoryPermission_None;
    nn::svc::MemoryPermission sharedPerm = nn::svc::MemoryPermission_ReadWrite;

    uintptr_t transAddr = reinterpret_cast<uintptr_t>(g_Transfer);
    uintptr_t sharedAddr = reinterpret_cast<uintptr_t>(g_SharedTransfer);
    size_t size = 0x1000;

    CheckMemory(
            transAddr,
            nn::svc::MemoryState_CodeData,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    CheckMemory(
            sharedAddr,
            nn::svc::MemoryState_CodeData,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    result = nn::svc::CreateTransferMemory(&transHandle, transAddr, size, transPerm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose transCloser(transHandle);

    result = nn::svc::CreateTransferMemory(&sharedHandle, sharedAddr, size, sharedPerm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(sharedHandle);

    CheckMemory(
            transAddr,
            nn::svc::MemoryState_CodeData,
            transPerm,
            nn::svc::MemoryAttribute_Locked);

    CheckMemory(
            sharedAddr,
            nn::svc::MemoryState_CodeData,
            sharedPerm,
            nn::svc::MemoryAttribute_Locked);

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        SCOPED_TRACE("Invalid Memory Area");

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::MapTransferMemory(transHandle, addr, size, transPerm);
        if (IsInAslrRegion(addr))
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
        }

        array[i]->CheckDefaultState();

        result = nn::svc::MapTransferMemory(sharedHandle, addr, size, sharedPerm);
        if (IsInAslrRegion(addr))
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
        }

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

} // namespace

TEST(MapTransferMemory, HandleTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    TestHeap heap(HeapAlign);
    size_t size = heap.GetSize();
    uintptr_t fromAddr = heap.GetAddress();
    uintptr_t toAddr = g_FreeAreaBegin;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];

        GetMemoryInfo(&blockInfo, fromAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::CreateTransferMemory(&handle, fromAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        // CreateTransferMemory で作成したハンドルを受け付ける
        {
            GetMemoryInfo(&blockInfo, fromAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
            ASSERT_TRUE(blockInfo.permission == perm);
            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

            GetMemoryInfo(&blockInfo, toAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);

            // TEST 127-1
            result = nn::svc::MapTransferMemory(handle, toAddr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&blockInfo, toAddr);
            if (perm == nn::svc::MemoryPermission_None)
            {
                ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
            }
            else
            {
                ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
            }
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.attribute == 0);

            // TEST 128-1
            result = nn::svc::UnmapTransferMemory(handle, toAddr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&blockInfo, toAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);
        }

        // TEST 127-2
        // INVALID_HANDLE_VALUE を受け付けない
        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapTransferMemory(nn::svc::INVALID_HANDLE_VALUE, toAddr, size, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 127-3
        // スレッドの擬似ハンドルを受け付けない
        result = nn::svc::MapTransferMemory(
                nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, toAddr, size, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 127-4
        // プロセスの擬似ハンドルを受け付けない
        result = nn::svc::MapTransferMemory(
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, toAddr, size, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 127-5
        // Close した CreateTransferMemory のハンドルを受け付けない
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, toAddr, size, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, toAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 127-34
        // 共有メモリのハンドルは受け付けない
        {
            nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadWrite;
            result = nn::svc::CreateSharedMemory(&handle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(handle);

            result = nn::svc::MapTransferMemory(handle, toAddr, size, perm);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

            GetMemoryInfo(&blockInfo, toAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);
        }

    }
}

TEST(MapTransferMemory, AddrTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];

        uintptr_t heapAddr;
        size_t heapSize = g_FreeAreaEnd - g_FreeAreaBegin;
        size_t size = 0x1000;

        TestHeap heap(heapSize);
        heapAddr = heap.GetAddress();
        ASSERT_TRUE(heapSize == heap.GetSize());
        ASSERT_TRUE((heapAddr & (0xfff)) == 0);

        result = nn::svc::CreateTransferMemory(&handle, heapAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        // 4KB にアライメントされている値を受け付ける
        for (uintptr_t addr = g_FreeAreaBegin; addr < g_FreeAreaEnd; addr += 0x1000)
        {
            // TEST 127-6
            result = nn::svc::MapTransferMemory(handle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&blockInfo, addr);
            if (perm == nn::svc::MemoryPermission_None)
            {
                ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
            }
            else
            {
                ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
            }
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.attribute == 0);

            // TEST 128-6
            result = nn::svc::UnmapTransferMemory(handle, addr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&blockInfo, addr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);
        }

        // TEST 127-7
        // 4KB にアライメントされていないと失敗する
        for (uintptr_t delta = 1; delta < 0x1000; delta++)
        {
            result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin + delta, size, perm);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

            GetMemoryInfo(&blockInfo, g_FreeAreaBegin + delta);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);
        }
    }
}

TEST(MapTransferMemory, SizeTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    uintptr_t heapAddr;
    size_t heapSize = g_FreeAreaEnd - g_FreeAreaBegin; // 16MB
    TestHeap heap(heapSize);
    heapAddr = heap.GetAddress();
    ASSERT_TRUE(heapSize == heap.GetSize());
    ASSERT_TRUE((heapAddr & (0xfff)) == 0);

    uintptr_t addr = g_FreeAreaBegin;
    size_t size = 0x1000;
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadWrite;

    // CreateTransferMemory と一致していると成功する
    for (size = 0x1000; size < heapSize; size += 0x1000)
    {
        result = nn::svc::CreateTransferMemory(
                &handle, heapAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        // TEST 127-8
        result = nn::svc::MapTransferMemory(handle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 128-9
        result = nn::svc::UnmapTransferMemory(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    size = 0x1000;
    result = nn::svc::CreateTransferMemory(&handle, heapAddr, size, perm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose transferCloser(handle);

    // TEST 127-9
    // CreateTransferMemory と一致していないと失敗する
    for (size_t delta = 1; delta < 0x1000; delta++)
    {
        result = nn::svc::MapTransferMemory(handle, addr, delta, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

TEST(MapTransferMemory, MemoryAreaTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Free,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid);

    uint32_t successAttribute[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    TestNoneAttribute* pNone = static_cast<TestNoneAttribute*>(successAttrArea[0]);
    pNone->SetNoneArea(g_FreeAreaBegin, TestMemorySize);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr);

    const int NumInvalidAttrs = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrs[NumInvalidAttrs];
    header->GetTestListExceptAttributes(invalidAttrs, NumInvalidAttrs, successAttribute, NumSuccessAttr);

    CheckInvalidMemoryArea(invalidAttrs, NumInvalidAttrs);


    // ヒープを使った異常系のテスト
    {
        TestHeap heap(HeapAlign);
        uintptr_t heapAddr = heap.GetAddress();
        size_t size = 0x1000;
        nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_None;
        uintptr_t addr;

        nn::svc::Handle tmpHandle;
        result = nn::svc::CreateTransferMemory(&tmpHandle, heapAddr, size * 2, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(tmpHandle);

        // TEST 127-25
        // メモリ状態が一様でないと失敗する
        {
            uintptr_t fromAddr = reinterpret_cast<uintptr_t>(g_Buffer);
            uintptr_t toAddr = g_FreeAreaBegin + size;
            TestMapMemory mapMem(toAddr, fromAddr, size);

            CheckMemory(
                    toAddr,
                    nn::svc::MemoryState_Stack,
                    nn::svc::MemoryPermission_ReadWrite,
                    0);

            addr = g_FreeAreaBegin;
            CheckMemory(
                    addr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);

            result = nn::svc::MapTransferMemory(tmpHandle, addr, size * 2, perm);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            CheckMemory(
                    toAddr,
                    nn::svc::MemoryState_Stack,
                    nn::svc::MemoryPermission_ReadWrite,
                    0);

            CheckMemory(
                    addr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);
        }

        // TEST 127-26
        // オーバーフローする組み合わせを指定できない
        {
            addr = g_FreeAreaBegin;
            CheckMemory(
                    addr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);

            size = static_cast<size_t>(-0x1000);

            result = nn::svc::MapTransferMemory(tmpHandle, addr, size, perm);
            ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                    result <= nn::svc::ResultInvalidRegion() ||
                    result <= nn::svc::ResultInvalidCurrentMemory());

            CheckMemory(
                    addr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);
        }
    }

    {
        TestHeap heap(HeapAlign);
        size_t size = 0x1000;
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_ReadWrite,
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_None
        };
        for (int i = 0; i < sizeof(perms) / sizeof(perms[0]); i++)
        {
            nn::svc::MemoryPermission perm = perms[0];

            result = nn::svc::CreateTransferMemory(
                    &handle, heap.GetAddress(), size, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose transferCloser(handle);

            // Reserved 領域
            result = nn::svc::MapTransferMemory(handle, g_ReservedAreaBegin, size, perm);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

            // Large 領域
            uint64_t largeAddr = 0;
            bool check = GetLargeFreeArea(&largeAddr, size);
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
            ASSERT_TRUE(!check);
#else
            ASSERT_TRUE(check);
#endif
            if(check)
            {
                uintptr_t addr = static_cast<uintptr_t>(largeAddr);
                result = nn::svc::MapTransferMemory(handle, addr, size, perm);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::UnmapTransferMemory(handle, addr, size);
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
} // NOLINT(readability/fn_size)

