﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestSdmmc.h"
#include <nn/svc/svc_Dd.h>
#include <cstring>

#ifdef SUPPORT_SDCARD_TEST

namespace {
    const uint64_t SpaceAddr = 0;
    // const nn::sdmmc1::Port TestPort = nn::sdmmc1::Port_Mmc0;    // sdmmc4a
    const nn::sdmmc1::Port TestPort = nn::sdmmc1::Port_SdCard0; // sdmmc3a

    const uint32_t TestSectorIndex = 0;
    const uint32_t TestSize = 32 * 1024;
    const uint32_t TestNumSectors = TestSize / nn::sdmmc1::SectorSize;
    NN_ALIGNAS(4096) uint8_t g_BackupBuffer[TestSize];

extern "C" void nnMain();

class RevertDetach
{
public:
    explicit RevertDetach(nn::svc::Handle handle)
        : m_Handle(handle)
    {
        nn::Result result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, m_Handle);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
    ~RevertDetach()
    {
        if (m_Handle.IsValid())
        {
            nn::Result result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, m_Handle);
            NN_ASSERT_RESULT_SUCCESS(result);
        }
    }
    void Revert()
    {
        nn::Result result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, m_Handle);
        NN_ASSERT_RESULT_SUCCESS(result);
        m_Handle = nn::svc::INVALID_HANDLE_VALUE;
    }
private:
    nn::svc::Handle m_Handle;
};

void AccessCheckTest(
        nn::svc::Handle handle, uintptr_t addr, uint64_t deviceAddress, uintptr_t backupBuffer, size_t size)
{
    nn::Result result;
    char* buffer = reinterpret_cast<char*>(addr);

    Sdmmc sdmmc(buffer, size, deviceAddress, TestPort);
    ASSERT_TRUE(sdmmc.Activate());

    uint32_t numSectors = size / nn::sdmmc1::SectorSize;

    // 読み込みチェック
    {
        std::memset(buffer, 0xa, size);

        nn::sdmmc1::Read(buffer, size, TestPort, TestSectorIndex, numSectors);

        bool check = false;
        for (int i = 0; i < size; i++)
        {
            if(buffer[i] != 0xa)
            {
                check = true;
                break;
            }
        }

        // 念のため
        if (!check)
        {
            std::memset(buffer, 0x0, size);
            nn::sdmmc1::Read(buffer, size, TestPort, TestSectorIndex, numSectors);
            for (int i = 0; i < size; i++)
            {
                if(buffer[i] != 0x0)
                {
                    check = true;
                    break;
                }
            }
        }

        ASSERT_TRUE(check);
    }

    // 戻すためにデータをバックアップ
    char* p_Backup = reinterpret_cast<char*>(backupBuffer);
    std::memcpy(p_Backup, buffer, size);

    // 書き込みチェック
    {
        ::std::memset(buffer, 0xa, size);
        nn::sdmmc1::Write(TestPort, TestSectorIndex, numSectors, buffer, size);

        // 正しく書き込めているかを確認
        std::memset(buffer, 0, size);
        nn::sdmmc1::Read(buffer, size, TestPort, TestSectorIndex, numSectors);

        for (size_t i = 0; i < size; i++)
        {
            ASSERT_TRUE(buffer[i] == 0xa);
        }
    }

    // デタッチするとアクセスできないことを確認
    RevertDetach detach(handle);

    // 読み込み出来ない
    {
        Sdmmc sdmmc(buffer, size, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());
        uint32_t numSectors = size / nn::sdmmc1::SectorSize;

        std::memset(buffer, 0x0, size);
        nn::sdmmc1::Read(buffer, size, TestPort, TestSectorIndex, numSectors);

        for (size_t i = 0; i < size; i++)
        {
            ASSERT_TRUE(buffer[i] == 0x0);
        }
    }

    // 書き込み出来ない
    {
        ::std::memset(buffer, 0xb, size);
        nn::sdmmc1::Write(TestPort, TestSectorIndex, numSectors, buffer, size);
    }

    detach.Revert();

    // 確認
    {
        Sdmmc sdmmc(buffer, size, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());

        std::memset(buffer, 0x0, size);
        nn::sdmmc1::Read(buffer, size, TestPort, TestSectorIndex, numSectors);

        bool check = false;
        for (size_t i = 0; i < size; i++)
        {
            if (buffer[i] != 0xb)
            {
                check = true;
                break;
            }
        }
        ASSERT_TRUE(check);
    }

    // 元に戻す
    {
        Sdmmc sdmmc(buffer, size, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());
        uint32_t numSectors = size / nn::sdmmc1::SectorSize;

        std::memcpy(buffer, p_Backup, size);
        nn::sdmmc1::Write(TestPort, TestSectorIndex, numSectors, buffer, size);
    }
}

} // namespace

TEST(MapDeviceAddressSpace, AlignTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0x400000;
    size_t outSize;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    uintptr_t heapAddr;
    size_t heapSize;

    // TEST 132-57
    // 4MB 未満
    {
        heapSize = align * 4;
        TestHeap heap(heapSize);
        heapAddr = ((heap.GetAddress() + (align - 1)) & ~(align - 1)) + 0x1000;
        deviceAddress = 0x1000;

        ASSERT_TRUE((TestSize & 0xFFF) == 0);
        ASSERT_TRUE((heapAddr & (align - 1)) == (deviceAddress & (align - 1)));

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress <= heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= heapAddr + TestSize - 1);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute
                == nn::svc::MemoryAttribute_DeviceShared);
        ASSERT_TRUE(blockInfo.deviceCount == 1);

        AccessCheckTest(handle, heapAddr, deviceAddress, heap.GetAddress() + align * 3, outSize);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-58
    // 4MB 境界 (アライン)
    {
        heapSize = align * 4;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, align, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        AccessCheckTest(handle, heapAddr, deviceAddress, heap.GetAddress() + align * 3, outSize);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-59
    // 4MB 境界 (非アラインアドレス)
    {
        heapSize = align * 4;
        TestHeap heap(heapSize);
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align + 0x1000;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + 0x1000, align, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr + 0x1000);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        AccessCheckTest(handle, heapAddr + 0x1000, deviceAddress, heap.GetAddress() + align * 3, outSize);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + 0x1000, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-60
    // 4MB 境界 (非アラインサイズ)
    {
        heapSize = align * 5;
        TestHeap heap(heapSize);
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        heapSize -= heapAddr - heap.GetAddress();
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, align + 0x1000, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        AccessCheckTest(handle, heapAddr, deviceAddress, heap.GetAddress() + align * 3, outSize);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
} // NOLINT(readability/fn_size)

TEST(MapDeviceAddressSpace, PermissionAccessTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    uintptr_t heapAddr;
    size_t heapSize;
    char* buffer;
    size_t outSize;

    // 元に戻すためのデータを取得しておく
    {
        heapSize = align * 3;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        buffer = reinterpret_cast<char*>(heapAddr);
        Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(outSize == TestSize);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        std::memset(buffer, 0, TestSize);
        nn::sdmmc1::Read(buffer, TestSize, TestPort, TestSectorIndex, TestNumSectors);

        std::memcpy(g_BackupBuffer, buffer, TestSize);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // Read
    {
        heapSize = align * 3;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        buffer = reinterpret_cast<char*>(heapAddr);
        Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        // デバイスがメモリを読むことが出来る
        ::std::memcpy(buffer, reinterpret_cast<void*>(nnMain), TestSize);
        nn::sdmmc1::Write(TestPort, TestSectorIndex, TestNumSectors, buffer, TestSize);

        // デバイスがメモリに書き出すことはできない
        std::memset(buffer, 0, TestSize);
        nn::sdmmc1::Read(buffer, TestSize, TestPort, TestSectorIndex, TestNumSectors);
        for (int i = 0; i < TestSize; i++)
        {
            ASSERT_TRUE(buffer[i] == 0);
        }

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        // 書き込みが出来ていることをチェック
        std::memset(buffer, 0, TestSize);
        nn::sdmmc1::Read(buffer, TestSize, TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_TRUE(::std::memcmp(buffer, reinterpret_cast<void*>(nnMain), TestSize) == 0);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // Write
    {
        heapSize = align * 3;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        buffer = reinterpret_cast<char*>(heapAddr);
        Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_Write);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        // デバイスがメモリに書き出すことは出来る
        ::std::memset(buffer, 0, TestSize);
        nn::sdmmc1::Read(buffer, TestSize, TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_TRUE(::std::memcmp(buffer, reinterpret_cast<char*>(nnMain), TestSize) == 0);

        // デバイスがメモリを読むことができない
        ::std::memcpy(buffer, reinterpret_cast<char*>(nnMain) + 0x100, TestSize);
        nn::sdmmc1::Write(TestPort, TestSectorIndex, TestNumSectors, buffer, TestSize);

        // デバイスがメモリに書き出すことは出来る
        ::std::memset(buffer, 0, TestSize);
        nn::sdmmc1::Read(buffer, TestSize, TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_TRUE(::std::memcmp(buffer, reinterpret_cast<char*>(nnMain) + 0x100, TestSize) != 0);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // 元に戻す
    {
        heapSize = align * 3;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        buffer = reinterpret_cast<char*>(heapAddr);
        Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort);
        ASSERT_TRUE(sdmmc.Activate());

        result = nn::svc::MapDeviceAddressSpace(&outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(outSize == TestSize);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        std::memcpy(buffer, g_BackupBuffer, TestSize);
        nn::sdmmc1::Write(TestPort, TestSectorIndex, TestNumSectors, buffer, TestSize);

        std::memset(buffer, 0, TestSize);
        nn::sdmmc1::Read(buffer, TestSize, TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_TRUE(std::memcmp(buffer, g_BackupBuffer, TestSize) == 0);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
} // NOLINT(readability/fn_size)

// TEST 132-62
// 同じパーミッションを隣り合うブロックに指定してマップをすると、ブロックがマージされる
TEST(MapDeviceAddressSpace, MergeBlockTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0x400000;
    nn::svc::MemoryInfo blockInfo;
    uint64_t align = 0x400000;
    size_t outSize;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    TestHeap heap(HeapAlign * 3);

    uint64_t addr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
    uint64_t size = 0x1000;
    result = nn::svc::MapDeviceAddressSpace(&outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    GetMemoryInfo(&blockInfo, addr + size);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::MapDeviceAddressSpace(&outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + size, size, deviceAddress + size,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == size * 2);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    GetMemoryInfo(&blockInfo, addr + size);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + size, size, deviceAddress + size);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
    ASSERT_TRUE(blockInfo.baseAddress <= addr);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= addr + size * 2);
}

#endif // SUPPORT_SDCARD_TEST

