﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

namespace {

const size_t StackSize = 0x1000;
char g_Stack[2][StackSize] __attribute__((aligned(0x1000)));

#ifdef ENABLE_MAX_HANDLE_TEST
void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}
#endif // ENABLE_MAX_HANDLE_TEST

void NormalClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    char* name = reinterpret_cast<char*>(arg);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToNamedPort(&clientSession, name);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(clientSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void NormalServerThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    nn::svc::Handle* serverPort = reinterpret_cast<nn::svc::Handle*>(arg);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, *serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(serverSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void CheckIpc(nn::svc::Handle serverPort, const char* name)
{
    nn::Result result;

    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    pc[0] = reinterpret_cast<uintptr_t>(NormalServerThread);
    pc[1] = reinterpret_cast<uintptr_t>(NormalClientThread);

    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + StackSize;
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + StackSize;

    param[0] = reinterpret_cast<uintptr_t>(&serverPort);
    param[1] = reinterpret_cast<uintptr_t>(name);

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    serverThread.Wait();
    clientThread.Wait();
}

} // namespace

TEST(ManageNamedPort, NotFoundTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle namedPort;

    // TEST 113-20
    // maxSessions = 0 の時に、対象となる名前を持ったポートが存在しないと失敗する
    result = nn::svc::ManageNamedPort(&namedPort, "NotFound", 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
}

TEST(ManageNamedPort, UnregisterPortTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle namedPort;
    const char* name = "DeletePort";

    result = nn::svc::ManageNamedPort(&namedPort, name, 2);
    ASSERT_RESULT_SUCCESS(result);

    CheckIpc(namedPort, name);

    nn::svc::Handle tmpHandle;
    result = nn::svc::ManageNamedPort(&tmpHandle, name, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);

    // TESE 113-21
    // 登録を削除された名前付きポートに接続は出来ない
    result = nn::svc::ConnectToNamedPort(&tmpHandle, name);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());

    result = nn::svc::CloseHandle(namedPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 113-22
    //一度登録を消した名前を再度登録することが出来る
    result = nn::svc::ManageNamedPort(&namedPort, name, 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(namedPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&tmpHandle, name, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);
}

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(ManageNamedPort, HandleResourceTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;

    nn::svc::Handle thread;
    uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
    uintptr_t sp = 0; // 起動させないので、問題ない
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    {
        ConsumeHandle maxHandle;

        // TEST 113-23
        // ハンドル数が上限に達している状態で呼び出すと失敗する
        nn::svc::Handle namedPort;
        const char* name = "MaxHandle";

        nn::svc::Handle* handles = maxHandle.GetHandles();

        result = nn::svc::ManageNamedPort(&namedPort, name, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());

        result = nn::svc::CloseHandle(handles[0]);
        ASSERT_RESULT_SUCCESS(result);
        handles[0] = nn::svc::INVALID_HANDLE_VALUE;

        // TEST 113-24
        // ハンドルを解放すれば、名前付きポートを作成できるようになる
        result = nn::svc::ManageNamedPort(&namedPort, name, 1);
        ASSERT_RESULT_SUCCESS(result);

        // 念のため、スレッドが作成できないことを確認
        result = nn::svc::CreateThread(&thread, pc, param, sp, priority, idealCore);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());

        // TEST 113-25
        // maxSessions = 0 の時に、ハンドルをClose しないとポートが破棄されない
        nn::svc::Handle tmpHandle;
        result = nn::svc::ManageNamedPort(&tmpHandle, name, 0);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);

        result = nn::svc::CreateThread(&thread, pc, param, sp, priority, idealCore);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());

        result = nn::svc::CloseHandle(namedPort);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CreateThread(&thread, pc, param, sp, priority, idealCore);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(thread);
        ASSERT_RESULT_SUCCESS(result);
    }

}
#endif // ENABLE_MAX_HANDLE_TEST

TEST(ManageNamedPort, ReAssignTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle namedPort;
    const char* name = "Test11332";

    result = nn::svc::ManageNamedPort(&namedPort, name, 1);
    ASSERT_RESULT_SUCCESS(result);

    CheckIpc(namedPort, name);

    result = nn::svc::CloseHandle(namedPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 113-32
    // 作成した名前付きポートのハンドルを閉じた後に、もう一度登録しようとしても失敗する
    result = nn::svc::ManageNamedPort(&namedPort, name, 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

    nn::svc::Handle tmpHandle;
    result = nn::svc::ManageNamedPort(&tmpHandle, name, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);
}

