﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>
#include <cstring>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/util/util_BitPack.h>
#include <nn/nn_BitTypes.h>
#include <nn/os.h>
#include <nn/svc/svc_Tcb.h>

// TEST 27-5, 66-5
// 任意データを送ることが出来る
extern nn::Result ClientLightSession(nn::svc::Handle handle);
extern nn::Result ServerLightSession(nn::svc::Handle handle);

namespace {
nn::svc::Handle g_Thread0;
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

void LightSessionThread0(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    nn::Result result;

    result = ClientLightSession(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

void LightSessionThread1(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::SleepThread(1000 * 1000 * 1000);

    result = ClientLightSession(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

void PortLightSessionThread0(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle session;

    result = nn::svc::ConnectToPort(&session, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = ClientLightSession(session);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(session);
    ASSERT_RESULT_SUCCESS(result);
}

void PortLightSessionThread1(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle session;

    result = nn::svc::ConnectToPort(&session, handle);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::SleepThread(1000 * 1000 * 1000);

    result = ClientLightSession(session);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(session);
    ASSERT_RESULT_SUCCESS(result);
}

}



TEST(CreateSession, LightTest0)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(LightSessionThread0);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateSession(&serverSessionHandle, &clientSessionHandle, true, 1);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 27-1
        // 相手が受信待ち状態になる前に、Light Session に対して、リクエストを送信することが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        nn::svc::SleepThread(1000 * 1000 * 1000);

        // TEST 66-1
        // ReplyAndReceiveLight を呼ぶ前に送られてきた要求を LightSession を通じて受信することが出来る
        // TEST 66-3
        // LightSession を通じて、結果を送信することができる
        result = ServerLightSession(serverSessionHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreateSession, LightTest1)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(LightSessionThread1);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateSession(&serverSessionHandle, &clientSessionHandle, true, 1);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 27-2
        // 相手が受信待ち状態になった後に、Light Session に対して、リクエストを送信することが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 66-2
        // ReplyAndReceiveLight を呼ぶ前に送られてきた要求を LightSession を通じて受信することが出来る
        // TEST 66-3
        // LightSession を通じて、結果を送信することができる
        result = ServerLightSession(serverSessionHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreatePort, LightTest0)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(PortLightSessionThread0);
        nn::svc::Handle clientPortHandle;
        nn::svc::Handle serverPortHandle;
        nn::svc::Handle serverSessionHandle;
        result = nn::svc::CreatePort(&serverPortHandle, &clientPortHandle, 2, true, 0);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 27-3
        // 相手が受信待ち状態になる前に、Light Session に対して、リクエストを送信することが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientPortHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &serverPortHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&serverSessionHandle, serverPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::SleepThread(1000 * 1000 * 1000);

        // TEST 66-4
        // ReplyAndReceiveLight を呼ぶ前に送られてきた要求を LightSession を通じて受信することが出来る
        // TEST 66-3
        // LightSession を通じて、結果を送信することができる
        result = ServerLightSession(serverSessionHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(clientPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreatePort, LightTest1)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(PortLightSessionThread1);
        nn::svc::Handle clientPortHandle;
        nn::svc::Handle serverPortHandle;
        nn::svc::Handle serverSessionHandle;
        result = nn::svc::CreatePort(&serverPortHandle, &clientPortHandle, 2, true, 0);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 27-4
        // 相手が受信待ち状態になった後に、Light Session に対して、リクエストを送信することが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientPortHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &serverPortHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&serverSessionHandle, serverPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 66-5
        // ReplyAndReceiveLight を呼び出した後に送られてきた要求を LightSession を通じて受信することが出来る
        // TEST 66-3
        // LightSession を通じて、結果を送信することができる
        result = ServerLightSession(serverSessionHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(clientPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
    }
}


